import graphlab.connect as _mt
import graphlab as _graphlab
from random import randint as _randint
import math
import itertools as _itertools
import random


def tf_idf(dataset, term_frequency_weight='linear'):
    """
    Compute the TF-IDF scores for each word in each document. The collection
    of documents must be in bag-of-words format.

    .. math::
        \mbox{TF-IDF}(w, d) = tf(w, d) * log(N / f(w))

    where :math:`tf(w, d)` is the number of times word :math:`w` appeared in
    document :math:`d`, :math:`f(w)` is the number of documents word :math:`w`
    appeared in, and :math:`N` is the number of documents.

    Parameters
    ----------
    dataset : SArray of type dict
        A bag of words representation of a document corpus.
        Each element is a dictionary representing a single document, where
        the keys are words and the values are the number of times that word
        occurs in that document.

    term_frequency_weight : str, optional
        Possible values: 'linear' and 'log'. If 'log', then a log
        transformation is applied to each term frequency prior to
        computing TF-IDF.

    Returns
    -------
    out : SArray of type dict
        The same document corpus where each score has been replaced by the
        TF-IDF transformed

    Notes
    -----
    For more information, see:
    https://en.wikipedia.org/wiki/TFIDF
    """
    if type(dataset) != _graphlab.SArray:
        raise TypeError, 'tf_idf requires an SArray of dict type, where each dictionary whose keys are words and whose values are word frequency.'

    if len(dataset) == 0:
        return _graphlab.SArray()

    # Convert into 'stacked' format
    d = _graphlab.SFrame({'x':dataset,
                          'doc_id':range(len(dataset))}).stack('x', ['word', 'tf'], [str, int])

    # Compute number of docs each word appears in
    doc_freq = d.groupby('word', {'doc_freq':_graphlab.aggregate.COUNT})

    # For each word, get document frequency
    d = d.join(doc_freq, on='word')

    # Compute tf-idf
    N = len(dataset)

    if term_frequency_weight == 'log':
        d['tf'] = d['tf'].apply(lambda x: math.log(x+1))

    d['tfidf'] = d['tf'] * d['doc_freq'].apply(lambda x: math.log(N/x))

    # Return SArray of documents
    return d[['doc_id', 'word', 'tfidf']].unstack(['word', 'tfidf'], 'scores').sort('doc_id')['scores']

def parse_sparse(filename, vocab_filename):
    """
    Parse a file that's in sparse format. This has a close relationship to
    the format needed for libSVM and Vowpal Wabbit.

    Params
    ------
    filename : str
        The name of the file to parse. Each line should represent a document,
        and be in the following format.

        num_unique_words_in_doc word_id:count another_id:count

        The word_ids have 0-based indexing, i.e. 0 corresponds to the first
        word in the vocab filename.

    vocab_filename : str
        A list of words that are used for this data set.

    Returns
    -------
    out : SArray
        Each element represents a document in bag-of-words format.

    """
    _mt._get_metric_tracker().track('toolkit.text.util.parse_sparse')

    vocab = _graphlab.SFrame.read_csv(vocab_filename, header=None)['X1']
    vocab = list(vocab)

    docs = _graphlab.SFrame.read_csv(filename, header=None)

    # Remove first word
    docs = docs['X1'].apply(lambda x: x.split(' ')[1:])

    # Helper function that checks whether we get too large a word id
    def get_word(word_id):
        assert int(word_id) < len(vocab), \
                "Text data contains integers that are larger than the \
                 size of the provided vocabulary."
        return vocab[word_id]

    def make_dict(pairs):
        pairs = [z.split(':') for z in pairs]
        ret = {}
        for k, v in pairs:
            ret[get_word(int(k))] = int(v)
        return ret

    # Split word_id and count and make into a dictionary
    docs = docs.apply(lambda x: make_dict(x))

    return docs

def parse_docword(filename, vocab_filename):
    """
    Parse a file that's in "docword" format. This consists of a 3-line header
    comprised of the document count, the vocabulary count, and the number of
    tokens, i.e. unique (doc_id, word_id) pairs. After the header, each line
    contains a space-separated triple of (doc_id, word_id, frequency), where
    frequency is the number of times word_id occurred in document doc_id.

    This format assumes that documents and words are identified by a positive
    integer (whose lowest value is 1).
    Thus, the first word in the vocabulary file has word_id=1.

    1 5 1
    1 105 3
    1 272 5
    2 1 3
    ...

    Params
    ------
    filename : str
        The name of the file to parse.

    vocab_filename : str
        A list of words that are used for this data set.

    Returns
    -------
    out : SArray
        Each element represents a document in bag-of-words format.
    """
    _mt._get_metric_tracker().track('toolkit.text.util.parse_docword')


    vocab = _graphlab.SFrame.read_csv(vocab_filename, header=None)['X1']
    vocab = list(vocab)

    indexed_docs = _graphlab.SFrame.read_csv(filename, header=None,
                                             delimiter=' ',
                                             column_type_hints=int)

    docs = indexed_docs.unstack(['X2', 'X3'], 'bow').sort('X1')['bow']
    docs = docs.apply(lambda x: {vocab[k-1]:v for (k, v) in x.iteritems()})

    return docs


def random_split(dataset, prob=.5):
    """
    Utility for performing a random split for text data that is already in
    bag-of-words format. For each (word, count) pair in a particular element,
    the counts are uniformly partitioned in either a training set or a test
    set.

    Params
    ------
    dataset : SArray
        A data set in bag-of-words format.

    prob : float
        Probability for sampling a word to be placed in the test set.

    Returns
    -------
    train, test : pair of SArrays
        Two data sets in bag-of-words format, where the combined counts are
        equal to the counts in the original data set.

    Example
    -------

    >>> train, test = topic_model.random_split(docs)

    If the word "hello" was said 10 times in the first document, then the
    counts for "hello" in the first document of train and test respectively
    will add to 10.
    """
    _mt._get_metric_tracker().track('toolkit.text.util.random_split')


    def grab_values(x, train=True):
        if train:
            ix = 0
        else:
            ix = 1
        return dict([(key, value[ix]) for key, value in x.iteritems() \
                    if value[ix] != 0])

    def word_count_split(n, p):
        num_in_test = 0
        for i in range(n):
            if random.random() < p:
                num_in_test += 1
        return [n - num_in_test, num_in_test]

    # Get an SArray where each word has a 2 element list containing
    # the count that will be for the training set and the count that will
    # be assigned to the test set.
    data = dataset.apply(lambda x: dict([(key, word_count_split(value, prob)) \
                                for key, value in x.iteritems()]))

    # Materialize the data set
    data.__materialize__()

    # Grab respective counts for each data set
    train = data.apply(lambda x: grab_values(x, train=True))
    test  = data.apply(lambda x: grab_values(x, train=False))

    return train, test


def generate_bar_example(num_topics=10,
                         num_documents=500,
                         num_words_per_doc=100,
                         alpha=1, beta=1):
    """
    Generate the classic "bars" example, a synthetic data set of small
    black 5x5 pixel images with a single white bar that is either horizontal
    or vertical.

    See Steyvers' MATLAB Topic Modeling Toolbox,
    http://psiexp.ss.uci.edu/research/programs_data/exampleimages1.html,

    and the original paper:
    Griffiths, T., & Steyvers, M. (2004). Finding Scientific Topics.
    Proceedings of the National Academy of Sciences, 101 (suppl. 1), 5228-5235.


    Returns
    -------
    out : SArray
        Each element represents a 'document' where the words are strings that
        represent a single pixel in the image in a colon-separated format.
        For example, 'horizontal_location:vertical_location'. Each word is
        associated with a count of the number of generated occurrences.
    """
    _mt._get_metric_tracker().track('toolkit.text.util.generate_bar_example')


    width = 5

    vocab_size = width * width

    zeros = [[0 for i in range(width)] for j in range(width)]
    topic_squares = [zeros for i in range(num_topics)]
    for i in range(width):
        for j in range(width):
            topic_squares[i][i][j] = 1./width
    for i in range(width):
        for j in range(width):
            topic_squares[width+i][j][i] = 1./width
    topics = []
    for k in range(num_topics):
        topics.append(list(_itertools.chain(*topic_squares[k])))

    def weighted_choice(probs):
        total = sum(probs)
        r = random.uniform(0, total)
        upto = 0
        for i, w in enumerate(probs):
            if upto + w > r:
                return i
            upto += w
        assert False, "Shouldn't get here"

    documents = []
    thetas = []
    for d in range(num_documents):
        doc = [0 for i in range(width * width)]
        topic_dist = [random.gammavariate(1, 1) for k in range(num_topics)]
        topic_dist = [z / sum(topic_dist) for z in topic_dist]
        for i in range(num_words_per_doc):
            k = weighted_choice(topic_dist)
            w = weighted_choice(topics[k])
            doc[w] += 1
        thetas.append(topic_dist)
        documents.append(doc)

    sparse_documents = []
    for d in documents:
        sd = {}
        for i in range(width):
            for j in range(width):
                k = str(i) + ',' + str(j)
                sd[k] = d[i * width + j]
        sparse_documents.append(sd)
    bow_documents = _graphlab.SArray(sparse_documents)
    return bow_documents

def stopwords(lang='en'):
    """
    Get common words that are often removed during preprocessing of text data,
    i.e. "stopwords".

    Params
    ------
    lang : str
        The desired language. Default: en.

    Returns
    -------
    out : set
        A set of str.
    """
    if lang=='en' or lang=='english':
        return set(['a', 'able', 'about', 'above', 'according', 'accordingly', 'across', 'actually', 'after', 'afterwards', 'again', 'against', 'all', 'allow', 'allows', 'almost', 'alone', 'along', 'already', 'also', 'although', 'always', 'am', 'among', 'amongst', 'an', 'and', 'another', 'any', 'anybody', 'anyhow', 'anyone', 'anything', 'anyway', 'anyways', 'anywhere', 'apart', 'appear', 'appreciate', 'appropriate', 'are', 'around', 'as', 'aside', 'ask', 'asking', 'associated', 'at', 'available', 'away', 'awfully', 'b', 'be', 'became', 'because', 'become', 'becomes', 'becoming', 'been', 'before', 'beforehand', 'behind', 'being', 'believe', 'below', 'beside', 'besides', 'best', 'better', 'between', 'beyond', 'both', 'brief', 'but', 'by', 'c', 'came', 'can', 'cannot', 'cant', 'cause', 'causes', 'certain', 'certainly', 'changes', 'clearly', 'co', 'com', 'come', 'comes', 'concerning', 'consequently', 'consider', 'considering', 'contain', 'containing', 'contains', 'corresponding', 'could', 'course', 'currently', 'd', 'definitely', 'described', 'despite', 'did', 'different', 'do', 'does', 'doing', 'done', 'down', 'downwards', 'during', 'e', 'each', 'edu', 'eg', 'eight', 'either', 'else', 'elsewhere', 'enough', 'entirely', 'especially', 'et', 'etc', 'even', 'ever', 'every', 'everybody', 'everyone', 'everything', 'everywhere', 'ex', 'exactly', 'example', 'except', 'f', 'far', 'few', 'fifth', 'first', 'five', 'followed', 'following', 'follows', 'for', 'former', 'formerly', 'forth', 'four', 'from', 'further', 'furthermore', 'g', 'get', 'gets', 'getting', 'given', 'gives', 'go', 'goes', 'going', 'gone', 'got', 'gotten', 'greetings', 'h', 'had', 'happens', 'hardly', 'has', 'have', 'having', 'he', 'hello', 'help', 'hence', 'her', 'here', 'hereafter', 'hereby', 'herein', 'hereupon', 'hers', 'herself', 'hi', 'him', 'himself', 'his', 'hither', 'hopefully', 'how', 'howbeit', 'however', 'i', 'ie', 'if', 'ignored', 'immediate', 'in', 'inasmuch', 'inc', 'indeed', 'indicate', 'indicated', 'indicates', 'inner', 'insofar', 'instead', 'into', 'inward', 'is', 'it', 'its', 'itself', 'j', 'just', 'k', 'keep', 'keeps', 'kept', 'know', 'knows', 'known', 'l', 'last', 'lately', 'later', 'latter', 'latterly', 'least', 'less', 'lest', 'let', 'like', 'liked', 'likely', 'little', 'look', 'looking', 'looks', 'ltd', 'm', 'mainly', 'many', 'may', 'maybe', 'me', 'mean', 'meanwhile', 'merely', 'might', 'more', 'moreover', 'most', 'mostly', 'much', 'must', 'my', 'myself', 'n', 'name', 'namely', 'nd', 'near', 'nearly', 'necessary', 'need', 'needs', 'neither', 'never', 'nevertheless', 'new', 'next', 'nine', 'no', 'nobody', 'non', 'none', 'noone', 'nor', 'normally', 'not', 'nothing', 'novel', 'now', 'nowhere', 'o', 'obviously', 'of', 'off', 'often', 'oh', 'ok', 'okay', 'old', 'on', 'once', 'one', 'ones', 'only', 'onto', 'or', 'other', 'others', 'otherwise', 'ought', 'our', 'ours', 'ourselves', 'out', 'outside', 'over', 'overall', 'own', 'p', 'particular', 'particularly', 'per', 'perhaps', 'placed', 'please', 'plus', 'possible', 'presumably', 'probably', 'provides', 'q', 'que', 'quite', 'qv', 'r', 'rather', 'rd', 're', 'really', 'reasonably', 'regarding', 'regardless', 'regards', 'relatively', 'respectively', 'right', 's', 'said', 'same', 'saw', 'say', 'saying', 'says', 'second', 'secondly', 'see', 'seeing', 'seem', 'seemed', 'seeming', 'seems', 'seen', 'self', 'selves', 'sensible', 'sent', 'serious', 'seriously', 'seven', 'several', 'shall', 'she', 'should', 'since', 'six', 'so', 'some', 'somebody', 'somehow', 'someone', 'something', 'sometime', 'sometimes', 'somewhat', 'somewhere', 'soon', 'sorry', 'specified', 'specify', 'specifying', 'still', 'sub', 'such', 'sup', 'sure', 't', 'take', 'taken', 'tell', 'tends', 'th', 'than', 'thank', 'thanks', 'thanx', 'that', 'thats', 'the', 'their', 'theirs', 'them', 'themselves', 'then', 'thence', 'there', 'thereafter', 'thereby', 'therefore', 'therein', 'theres', 'thereupon', 'these', 'they', 'think', 'third', 'this', 'thorough', 'thoroughly', 'those', 'though', 'three', 'through', 'throughout', 'thru', 'thus', 'to', 'together', 'too', 'took', 'toward', 'towards', 'tried', 'tries', 'truly', 'try', 'trying', 'twice', 'two', 'u', 'un', 'under', 'unfortunately', 'unless', 'unlikely', 'until', 'unto', 'up', 'upon', 'us', 'use', 'used', 'useful', 'uses', 'using', 'usually', 'uucp', 'v', 'value', 'various', 'very', 'via', 'viz', 'vs', 'w', 'want', 'wants', 'was', 'way', 'we', 'welcome', 'well', 'went', 'were', 'what', 'whatever', 'when', 'whence', 'whenever', 'where', 'whereafter', 'whereas', 'whereby', 'wherein', 'whereupon', 'wherever', 'whether', 'which', 'while', 'whither', 'who', 'whoever', 'whole', 'whom', 'whose', 'why', 'will', 'willing', 'wish', 'with', 'within', 'without', 'wonder', 'would', 'would', 'x', 'y', 'yes', 'yet', 'you', 'your', 'yours', 'yourself', 'yourselves', 'z', 'zero'])
    else:
        raise NotImplementedError, 'Only English stopwords are currently available.'
