from graphlab.data_structures.sgraph import SGraph as _SGraph
import graphlab.toolkits.main as _main
from graphlab.toolkits.graph_analytics.model_base import GraphAnalyticsModel as _ModelBase


def get_default_options():
    """
    Get the default options for :func:`graphlab.graph_coloring.create`.

    Returns
    -------
    out : dict

    Examples
    --------
    >>> graphlab.graph_coloring.get_default_options()
    """
    return _main.run('graph_coloring_default_options', {})


class GraphColoringModel(_ModelBase):
    """
    Model object containing the color ID for each vertex and the total number
    of colors used in coloring the entire graph.

    The coloring is the result of a greedy algorithm and therefore is not
    optimal.  Finding optimal coloring is in fact NP-complete.

    Below is a list of queryable fields for this model:

    +------------+-----------------------------------------------------+
    | Field      | Description                                         |
    +============+=====================================================+
    | graph      | A new SGraph with the color id as a vertex property |
    +------------+-----------------------------------------------------+
    | runtime    | Total runtime of the toolkit                        |
    +------------+-----------------------------------------------------+
    | num_colors | Number of colors in the graph                       |
    +------------+-----------------------------------------------------+

    An instance of this model can be created using
    :func:`graphlab.graph_coloring.create`.  Do NOT construct the model
    directly.

    See Also
    --------
    create
    """
    def __init__(self, model):
        '''__init__(self)'''
        self.__proxy__ = model

    def _result_fields(self):
        ret = super(GraphColoringModel, self)._result_fields()
        ret['num_colors'] = self['num_colors']
        ret['colorid'] = "SFrame with each vertex's color id. See m['colorid']"
        return ret


def create(graph, verbose=True, plot=False):
    """
    Compute the graph coloring. Assign a color to each vertex such that no
    adjacent vertices have the same color. Return a model object with total
    number of colors used as well as the color ID for each vertex in the graph.
    This algorithm is greedy and is not guaranteed to find the **minimum** graph
    coloring. It is also not deterministic, so successive runs may return
    different answers.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the coloring.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : GraphColoringModel

    References
    ----------
    - `Wikipedia - graph coloring <http://en.wikipedia.org/wiki/Graph_coloring>`_

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.graph_coloring.GraphColoringModel` as follows:

    >>> g = graphlab.load_graph('http://snap.stanford.edu/data/email-Enron.txt.gz', format='snap')
    >>> gc = graphlab.graph_coloring.create(g)

    We can obtain the ``color id`` corresponding to each vertex in the graph ``g``
    as follows:

    >>> color_id = gc['colorid']  # SFrame

    We can obtain the total number of colors required to color the graph ``g``
    as follows:

    >>> num_colors = gc['num_colors']

    See Also
    --------
    GraphColoringModel
    """
    if not isinstance(graph, _SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for graph coloring is not yet implemented."

    params = _main.run('graph_coloring', {'graph': graph.__proxy__}, verbose, plot)

    return GraphColoringModel(params['model'])
