"""
The GraphLab Create recommender toolkit provides a unified interface to train a
variety of recommender models and use them to make recommendations. 

Recommender models can be created using :meth:`graphlab.recommender.create` or
loaded from a previously saved model using :meth:`graphlab.load_model`. 
When calling `create`, the default recommender is a 
:class:`~graphlab.recommender.MatrixFactorizationModel` if a `target` column is 
provided in the input dataset; otherwise the default recommender is a
:class:`~graphlab.recommender.ItemSimilarityModel`. For more details, please 
see the documentation for :meth:`graphlab.recommender.create`.

A recommender model object can perform key tasks including
`predict`, `recommend`, `evaluate`, and `save`. Model attributes and
statistics may be obtained via `m.get()`, where `m` is a model object. In
particular, trained model parameters may be accessed using
`m.get('coefficients')` or equivalently `m['coefficients']`. For more details, 
please see individual model API documentation below.

In addition to the API documentation, please see the data science `Gallery
<http://www.graphlab.com/learn/gallery>`_, `How-tos
<http://www.graphlab.com/learn/how-to>`_, and the `recommender systems chapter of the
User Guide 
<http://www.graphlab.com/learn/userguide.html#Modeling_data_Recommender_systems>`_
for more details and extended examples.

Examples
--------
.. sourcecode:: python

  >>> sf = graphlab.SFrame({'user_id': ["0", "0", "0", "1", "1", "2", "2", "2"], 
  ...                       'item_id': ["a", "b", "c", "a", "b", "b", "c", "d"],
  ...                       'rating': [1, 3, 2, 5, 4, 1, 4, 3]})
  >>> m = graphlab.recommender.create(sf, target='rating')
  >>> recs = m.recommend()
  >>> print recs
  +---------+---------+---------------+------+
  | user_id | item_id |     score     | rank |
  +---------+---------+---------------+------+
  |    0    |    d    | 2.42301885789 |  1   |
  |    1    |    c    | 5.52301720893 |  1   |
  |    1    |    d    | 5.20882169849 |  2   |
  |    2    |    a    |  2.149379798  |  1   |
  +---------+---------+---------------+------+
  [4 rows x 4 columns]

  >>> m['coefficients']
  {'intercept': 3.1321961361684068, 'item_id': Columns:
    item_id str
    linear_terms  float
    factors array
   
   Rows: 4
   
   Data:
   +---------+-----------------+--------------------------------+
   | item_id |   linear_terms  |            factors             |
   +---------+-----------------+--------------------------------+
   |    a    | -0.381912890376 | array('d', [0.006779233276 ... |
   |    b    | -0.482275197699 | array('d', [-3.57188659440 ... |
   |    c    |  0.664901063905 | array('d', [-0.00025265078 ... |
   |    d    |  0.352987048665 | array('d', [-0.00197509767 ... |
   +---------+-----------------+--------------------------------+
   [4 rows x 3 columns], 'user_id': Columns:
    user_id str
    linear_terms  float
    factors array
   
   Rows: 3
   
   Data:
   +---------+-----------------+--------------------------------+
   | user_id |   linear_terms  |            factors             |
   +---------+-----------------+--------------------------------+
   |    0    |  -1.06188402031 | array('d', [-0.00321943390 ... |
   |    1    |  1.72356956865  | array('d', [0.005337682218 ... |
   |    2    | -0.604970370745 | array('d', [-0.00274082382 ... |
   +---------+-----------------+--------------------------------+
   [3 rows x 3 columns]}
"""

import popularity
import matrix_factorization
import linear_regression
import item_similarity
import item_means

from item_similarity import ItemSimilarityModel
from matrix_factorization import MatrixFactorizationModel
from popularity import PopularityModel
from item_means import ItemMeansModel
from linear_regression import LinearRegressionModel
from factorization_model import FactorizationModel

from graphlab.toolkits.recommender.util import create
from graphlab.toolkits.recommender.util import compare_models
from graphlab.toolkits.recommender.util import precision_recall_by_user
from graphlab.toolkits.recommender.util import random_split_by_user



