"""
Models that rank items based on their popularity.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.recommender.recommender import RecommenderModel
from graphlab.data_structures.sframe import SFrame as _SFrame
import logging as _logging


class FactorizationModel(RecommenderModel):
    r"""
    The Factorization Machine recommender model approximates target rating
    values as a weighted combination of user and item latent factors, biases,
    side features, and their pairwise combinations.

    The Factorization Machine [Rendle]_ is a generalization of Matrix Factorization.
    In particular, while Matrix Factorization learns latent factors for only the
    user and item interactions, the Factorization Machine learns latent factors
    for all variables, including side features, and also allows for interactions
    between all pairs of variables. Thus the Factorization Machine is capable of
    modeling complex relationships in the data. Typically,
    :class:`~graphlab.recommender.FactorizationModel` outperforms
    :class:`~graphlab.recommender.MatrixFactorization`, but may require a longer
    training time.

    **Creating a FactorizationModel**

    This model can be created using
    :func:`graphlab.recommender.create(..., method='factorization_model') <graphlab.recommender.create>`.
    Do NOT instantiate this model class directly.

    For further documentation, please refer to 
    :class:`~graphlab.recommender.MatrixFactorizationModel`.

    See Also
    --------
    MatrixFactorizationModel
    LinearRegressionModel
    
    References
    ----------
    .. [Rendle] Steffen Rendle, `"Factorization Machines,"
        <http://www.csie.ntu.edu.tw/~b97053/paper/Rendle2010FM.pdf>`_ in
        Proceedings of the 10th IEEE International Conference on Data Mining
        (ICDM), 2010.
    """

    def __init__(self, model_proxy):
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return FactorizationModel(model_proxy)
        return model_wrapper
