"""

Supervised learning is the problem of learning a functional relationship
between input features and an output target using a set of training data.  The
functional form learned depends on the choice of model. GraphLab Create's
supervised learning toolkit implements several models for both both regression,
which is the prediction of continuous responses, and classification, which
predicts discrete targets. The models are built to work with millions of
features and billions of examples. These algorithms differ in how they make
predictions, but conform to the same API.  With all models, call ``create()``
to create a model, ``predict()`` to make predictions on the returned model, and
``evaluate()`` to measure performance of the predictions. 

Each of these models can incorporate categorical variables, sparse features,
and text data. 

Concise descriptions of the supervised learning models are contained in this API
documentation, along with a small number of simple examples. For more
information and examples, please see the `How-Tos
<http://www.graphlab.com/learn/how-to>`_, data science `Gallery
<http://www.graphlab.com/learn/gallery>`_, and `supervised learning chapter of
the User Guide
<http://www.graphlab.com/learn/userguide.html#Modeling_data_Supervised_learning>`_.

.. sourcecode:: python

    # Load the data 
    >>> import graphlab as gl
    >>> data =  gl.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')
    
    # Make a linear regression model
    >>> model = gl.linear_regression.create(data, target='price', features=['bath', 'bedroom', 'size'])

    # Extract the coefficients as an SFrame
    >>> coefficients = model['coefficients']
    
    # Make predictions as an SArray
    >>> predictions = model.predict(data)   
    
    # Evaluate the model 
    >>> results = model.evaluate(data)          
    

"""
import graphlab as _graphlab
from graphlab.toolkits.model import Model
from graphlab.data_structures.sframe import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.deps import pandas as _pandas, HAS_PANDAS as _HAS_PANDAS


def _vw_solver_warning():
    import logging
    logging.warning("\nsolver='vw' will not be supported in future releases. Use graphlab.vowpal_wabbit.create() for Vowpal Wabbit.\n")


class SupervisedLearningModel(Model):
    """
    Supervised learning module to predict a target variable as a function of
    several feature variables.
    """

    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__class__.__name__

    def __repr__(self):
        """
        Returns a string description of the model, including (where relevant)
        the schema of the training data, description of the training data,
        training statistics, and model hyperparameters.

        Returns
        -------
        out : string
            A description of the model.
        """
        raise NotImplementedError


    def summary(self):
        """
        Display a summary of the model, training options, and training
        statistics.
        """
        raise NotImplementedError

    def get_default_options(self):
        """
        Return a dictionary with the model's default options.

        Returns
        -------
        out : dict
            Dictionary with model default options.
        """

        opts = {'model': self.__proxy__,
                'model_name': self.__name__}
        return _graphlab.toolkits.main.run('supervised_learning_get_default_options',
            opts)

    def get_current_options(self):
        """
        Return a dictionary with the options used to define and train the model.

        Returns
        -------
        out : dict
            Dictionary with options used to define and train the model.
        """

        opts = {'model': self.__proxy__,
                'model_name': self.__name__}
        return _graphlab.toolkits.main.run('supervised_learning_get_current_options',
            opts)

    def predict(self, dataset):
        """
        Return predictions for ``dataset``, using the trained supervised_learning
        classification model. Predictions are generated as class labels (0 or
        1).

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the features used for model training, but does not require
            a target column. Additional columns are ignored.

        Returns
        -------
        out : SArray
            An SArray with model predictions.
        """

        if _HAS_PANDAS and type(dataset) is _pandas.DataFrame:
            dataset = _SFrame(dataset)
        opts = {'model': self.__proxy__,
                'model_name': self.__name__,
                'dataset': dataset}

        init_opts = _graphlab.toolkits.main.run('supervised_learning_predict_init', opts)
        opts.update(init_opts)
        target = _graphlab.toolkits.main.run('supervised_learning_predict', opts)
        return _SArray(None, _proxy=target['predicted'])

    def evaluate(self, dataset):
        """
        Evaluate the model by making predictions of target values and comparing
        these to actual values.
        """
        raise NotImplementedError


    def training_stats(self):
        """
        Return a dictionary containing statistics collected during model
        training. These statistics are also available with the ``get`` method,
        and are described in more detail in the documentation for that method.

        Notes
        -----
        """
        opts = {'model': self.__proxy__, 'model_name': self.__name__}
        return _graphlab.toolkits.main.run("supervised_learning_get_train_stats", opts)

    def list_fields(self):
        """
        List the fields stored in the model, including data, model, and
        training options. Each field can be queried with the ``get`` method.

        Returns
        -------
        out : list
            List of fields queryable with the ``get`` method.
        """
        opts = {'model': self.__proxy__,
                'model_name': self.__name__}
        response = _graphlab.toolkits.main.run('supervised_learning_list_keys', opts)
        return sorted(response.keys())
