"""
Methods to be used internally by graphlab.canvas.server to handle HTTP requests.
"""

import graphlab
import graphlab.connect as _mt
from graphlab.canvas import statichandler as static

import json
import os
import sys
import tornado.web

def __error_handler(handler):
    """
    A Tornado handler that does nothing and returns {},
    for use as a fallback when errors occur during request handling.
    """
    handler.set_status(500)
    handler.write({})

def __make_handler(fn, state, *args, **kwargs):
    """
    Allows partial function application in a way that is compatible with Tornado handlers.
    Using functools.partial does not work with instance methods for some reason, but we can
    pass functions wrapped in __make_handler as instance methods and it works.
    Also does exception handling through state.add_exception to get exception messages to the UI.
    """
    def applied(*b, **c):
        d = {}
        d.update(kwargs)
        d.update(c)
        try:
          return fn(state, *(args + b), **d)
        except:
          e = sys.exc_info()
          state.add_exception(e)
          return __error_handler(b[0]) # assumes the first argument is 'handler'
    return applied

def objectSpecificRoute(state, method, handler, path):
    view = state.get_selected_variable()[1]
    view._handle_request(method, path, handler)

def expand_selected_variable(state):
    # get descriptives for variable
    selected_variable = state.get_selected_variable()
    if selected_variable is None:
        return None
    name, var = selected_variable
    if var is None:
        return None
    data = {
        'name': name,
        'type': var.objectType,
        'view_file': var.get_js_file(),
        'view_component': var.get_js_component()
    };
    data.update(var.get_metadata())
    return data

def __ping(state, server, handler):
    """
    Used when the window is not focused to just send an alive ping
    """
    server.ping()

def __get_var(state, server, handler):
    """
    Gives the browser data about Python variables, columns, selected state, etc.
    """
    server.ping()
    data = {
        'variables': state.get_variables(),
        'selected_variable': expand_selected_variable(state),
        'exceptions': state.get_exceptions(),
        'gl_version': graphlab.version,
        'gl_product_key': graphlab.product_key.get_product_key()
    }
    handler.write(data)

def __post_var(state, server, handler):
    """
    Sets the selected variable
    """
    selected_var = handler.get_argument('selected')
    selected_var = json.loads(selected_var) # json decode->list
    if type(selected_var) == list:
        selected_var = map(lambda x: x.encode('utf-8'), selected_var) # convert unicode
        selected_var = tuple(selected_var) # list->tuple
    else:
        selected_var = selected_var.encode('utf-8')
    state.set_selected_variable(selected_var)
    # send the global state (selection info) back to the browser
    __get_var(state, server, handler)

def __post_metrics(state, handler):
    """
    Allows the posting of metrics from JavaScript through our usual Python code.
    """
    kwargs = {
        'event_name': 'canvas.js.%s' % handler.get_argument('metric'),
        'value': int(handler.get_argument('value', default='1')),
        'properties': json.loads(handler.get_argument('properties', default="{}"))
    }
    _mt._get_metric_tracker().track(**kwargs)

class IndexHandler(static.Handler):
    def set_extra_headers(self, path):
        """
        Makes sure index.html gets sent with an xsrf token
        """
        # reading this property is sufficient to set the xsrf_token as a cookie
        # see http://tornado.readthedocs.org/en/latest/overview.html#cross-site-request-forgery-protection
        self.xsrf_token
        super(static.Handler, self).set_extra_headers(path)

class JSONResponseHandler(tornado.web.RequestHandler):
    def write(self, chunk):
        """
        Make sure we write chunks as valid JSON (by default Tornado does not -- it includes Infinity and NaN)
        """
        if isinstance(chunk, dict):
            chunk = json.dumps(chunk, allow_nan=False)
            self.set_header("Content-Type", "application/json; charset=UTF-8")
        super(JSONResponseHandler, self).write(chunk)

__handler_class_id = 0
def __make_handler_class(url, get=None, post=None):
    """
    Creates a tornado.web.RequestHandler class from the given get and post handlers.
    """
    global __handler_class_id
    handlerFunctions = {}
    if get is not None:
        handlerFunctions['get'] = get
    if post is not None:
        handlerFunctions['post'] = post
    __handler_class_id += 1
    return (url, type('CanvasHandler%d' % __handler_class_id, (JSONResponseHandler,), handlerFunctions))

def __get_static_handlers():
    """
    Get a list of handlers for static files as tornado.web.RequestHandler instances.
    """
    return [("/(.*\\.%s)" % ext, static.Handler, {'path': os.path.dirname(__file__)}) for ext in \
      ['css',
       'html',
       'ico',
       'jpg',
       'js',
       'png',
       'woff',
       'ttf',
       'eot',
       'otf',
       'svg']]

def get_handlers(server, state):
    """
    Get a list of handlers as tornado.web.RequestHandler instances.
    """
    ret = [
        __make_handler_class(
            url=r"/ping",
            get=__make_handler(__ping, state, server)),
        __make_handler_class(
            url=r"/vars",
            get=__make_handler(__get_var, state, server),
            post=__make_handler(__post_var, state, server)),
        __make_handler_class(
            url=r"/metrics",
            post=__make_handler(__post_metrics, state)),
        (r"/(index\.html)", IndexHandler, {'path': os.path.dirname(__file__)}),
        (r"/(js\/.*)", static.Handler, {'path': os.path.dirname(__file__)}),
    ]
    ret.extend(__get_static_handlers())
    ret.extend([
        __make_handler_class(
            url=r"/(.*)",
            get=__make_handler(objectSpecificRoute, state, 'get'),
            post=__make_handler(objectSpecificRoute, state, 'post'))
    ])
    return ret
