from graphlab.data_structures.sgraph import SGraph as _SGraph
import graphlab.toolkits.main as _main
from graphlab.toolkits.graph_analytics.model_base import GraphAnalyticsModel as _ModelBase


def get_default_options():
    """
    Get the default options for :func:`graphlab.connected_components.create`.

    Returns
    -------
    out : dict

    Examples
    --------
    >>> graphlab.connected_components.get_default_options()
    """
    return _main.run('connected_components_default_options', {})


class ConnectedComponentsModel(_ModelBase):
    """
    Model object containing the component ID for each vertex and the total
    number of weakly connected components in the graph.

    A weakly connected component is a maximal set of vertices such that there
    exists an undirected path between any two vertices in the set.

    Below is a list of queryable fields for this model:

    +----------------+-----------------------------------------------------+
    | Field          | Description                                         |
    +================+=====================================================+
    | graph          | A new SGraph with the color id as a vertex property |
    +----------------+-----------------------------------------------------+
    | runtime        | Total runtime of the toolkit                        |
    +----------------+-----------------------------------------------------+
    | component_size | An SFrame with the size of each component           |
    +----------------+-----------------------------------------------------+
    | componentid    | An SFrame with each vertex's component id           |
    +----------------+-----------------------------------------------------+

    An instance of this model can be created using
    :func:`graphlab.connected_components.create`.  Do NOT construct the model
    directly.

    See Also
    --------
    create
    """
    def __init__(self, model):
        '''__init__(self)'''
        self.__proxy__ = model

    def _result_fields(self):
        ret = super(ConnectedComponentsModel, self)._result_fields()
        ret['component_size'] = "SFrame with each component's size. See m['component_size']"
        ret['componentid'] = "SFrame with each vertex's component id. See m['componentid']"
        return ret


def create(graph, verbose=True, plot=False):
    """
    Compute the number of weakly connected components in the graph. Return a
    model object with total number of weakly connected components as well as the
    component ID for each vertex in the graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the triangle counts.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : ConnectedComponentsModel

    References
    ----------
    - `Mathworld Wolfram - Weakly Connected Component
      <http://mathworld.wolfram.com/WeaklyConnectedComponent.html>`_

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.connected_components.ConnectedComponentsModel` as
    follows:

    >>> g = graphlab.load_graph('http://snap.stanford.edu/data/email-Enron.txt.gz', format='snap')
    >>> cc = graphlab.connected_components.create(g)
    >>> cc.summary()

    We can obtain the ``component id`` corresponding to each vertex in the
    graph ``g`` as follows:

    >>> cc_ids = cc['componentid']  # SFrame

    We can obtain a graph with additional information about the ``component
    id`` corresponding to each vertex as follows:

    >>> cc_graph = cc['graph']      # SGraph

    See Also
    --------
    ConnectedComponentsModel
    """
    if not isinstance(graph, _SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for connected components is not yet " + \
            "implemented."

    params = _main.run('connected_components', {'graph': graph.__proxy__},
                       verbose, plot)
    return ConnectedComponentsModel(params['model'])
