from graphlab.data_structures.sgraph import SGraph as _SGraph
import graphlab.toolkits.main as _main
from graphlab.toolkits.graph_analytics.model_base import GraphAnalyticsModel as _ModelBase


def get_default_options():
    """
    Get the default options for :func:`graphlab.kcore.create`.

    Returns
    -------
    out : dict

    Examples
    --------
    >>> graphlab.kcore.get_default_options()
    """
    return _main.run('kcore_default_options', {})


class KcoreModel(_ModelBase):
    """
    Model object containing a core ID for each vertex, and the total number of
    cores in the graph.

    The core ID of a vertex is a measure of its global centrality.

    The algorithms iteratively remove vertices that has less than :math:`k`
    neighbors **recursively**. The algorithm guarantees that at iteration
    :math:`k+1`, all vertices left in the graph will have at least :math:`k+1`
    neighbors.  The vertices removed at iteration :math:`k` is assigned with a
    core ID equal to :math:`k`.

    Below is a list of queryable fields for this model:

    +---------+----------------------------------------------------+
    | Field   | Description                                        |
    +=========+====================================================+
    | coreid  | An SFrame with each vertex's core id               |
    +---------+----------------------------------------------------+
    | graph   | A new SGraph with the core id as a vertex property |
    +---------+----------------------------------------------------+
    | kmax    | The maximun core id assigned to any vertex         |
    +---------+----------------------------------------------------+
    | kmin    | The minimun core id assigned to any vertex         |
    +---------+----------------------------------------------------+
    | runtime | Total runtime of the toolkit                       |
    +---------+----------------------------------------------------+

    An instance of this model can be created using :func:`graphlab.kcore.create`.
    Do NOT construct the model directly.

    See Also
    --------
    create
    """
    def __init__(self, model):
        '''__init__(self)'''
        self.__proxy__ = model

    def _result_fields(self):
        ret = super(KcoreModel, self)._result_fields()
        ret['coreid'] = "SFrame with each vertex's coreid. See m['coreid']"
        return ret

    def _setting_fields(self):
        ret = super(KcoreModel, self)._setting_fields()
        for k in ['kmin', 'kmax']:
            ret[k] = self[k]
        return ret


def create(graph, kmin=0, kmax=10, verbose=True, plot=False):
    """
    Compute the K-core decomposition of the graph. Return a model object with
    total number of cores as well as the core id for each vertex in the graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the k-core decomposition.

    kmin : int, optional
        Minimun core id. Vertices having smaller core id than `kmin` will be
        assigned with coreid = `kmin`.

    kmax : int, optional
        Maximun core id. Vertices having larger core id than `kmax` will be
        assigned with coreid=`kmax`.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : KcoreModel

    References
    ----------
    - Alvarez-Hamelin, J.I., et al. (2005) `K-Core Decomposition: A Tool for the
      Visualization of Large Networks <http://arxiv.org/abs/cs/0504107>`_.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.kcore.KcoreModel` as follows:

    >>> g = graphlab.load_graph('http://snap.stanford.edu/data/email-Enron.txt.gz', format='snap')
    >>> kc = graphlab.kcore.create(g)

    We can obtain the ``core id`` corresponding to each vertex in the graph
    ``g`` using:

    >>> kcore_id = kc['coreid']     # SFrame

    See Also
    --------
    KcoreModel
    """
    if not isinstance(graph, _SGraph):
        raise TypeError('graph input must be a SGraph object.')

    opts = {'graph': graph.__proxy__, 'kmin': kmin, 'kmax': kmax}
    params = _main.run('kcore', opts, verbose, plot)

    return KcoreModel(params['model'])
