"""
Model base for graph analytics models
"""
from graphlab.toolkits.model import Model
from prettytable import PrettyTable as _PrettyTable
from graphlab.cython.cy_graph import UnityGraphProxy
from graphlab.cython.cy_sframe import UnitySFrameProxy
import graphlab.toolkits.main as _main
from graphlab.data_structures.sframe import SFrame
from graphlab.data_structures.sgraph import SGraph


class GraphAnalyticsModel(Model):

    def get(self, field):
        """
        Return the value for the queried field.

        Get the value of a given field. The list of all queryable fields is
        documented in the beginning of the model class.

        Each of these fields can be queried in one of two ways:

        >>> out = m['graph']
        >>> out = m.get('graph')  # equivalent to previous line

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : value
            The current value of the requested field.

        Examples
        --------
        >>> g = m.get('graph')
        """
        if field in self.list_fields():
            obj = self.__proxy__.get(field)
            if type(obj) == UnityGraphProxy:
                return SGraph(_proxy=obj)
            elif type(obj) == UnitySFrameProxy:
                return SFrame(_proxy=obj)
            else:
                return obj
        else:
            raise KeyError('Key \"%s\" not in model. Available fields are %s.' % (field, ', '.join(self.list_fields())))

    def _get_wrapper(self):
        """
        Returns the constructor for the model.
        """
        return lambda m: self.__class__(m)

    @classmethod
    def _describe_fields(cls):
        """
        Return a pretty table for the class fields description.
        """
        dispatch_table = {
            'ShortestPathModel': 'sssp_model_fields',
            'GraphColoringModel': 'graph_coloring_model_fields',
            'PagerankModel': 'pagerank_model_fields',
            'ConnectedComponentsModel': 'connected_components_model_fields',
            'TriangleCountingModel': 'triangle_counting_model_fields',
            'KcoreModel': 'kcore_model_fields'
        }
        try:
            fields_description = _main.run(dispatch_table[cls.__name__], {})
            tbl = _PrettyTable(['Field', 'Description'])
            for k, v in fields_description.iteritems():
                tbl.add_row([k, v])
            tbl.align['Field'] = 'l'
            tbl.align['Description'] = 'l'
            return tbl
        except:
            raise RuntimeError('Model %s does not have fields description' % cls.__name__)

    def _format(self, title, key_values):
        if len(key_values) == 0:
            return ""
        tbl = _PrettyTable(header=False)
        for k, v in key_values.iteritems():
                tbl.add_row([k, v])
        tbl.align['Field 1'] = 'l'
        tbl.align['Field 2'] = 'l'
        s = title + ":\n"
        s += tbl.__str__() + '\n'
        return s

    def __repr__(self):
        g = self['graph']
        s = self.name() + '\n'
        s += self._format('Graph', g.summary())
        s += self._format('Result', self._result_fields())
        s += self._format('Method', self._method_fields())
        return s

    def __str__(self):
        return self.__repr__()

    def summary(self):
        """
        Print a detailed summary of the model.

        Examples
        --------
        >>> m.summary()
        """
        s = self.__repr__()
        s += self._format('Setting', self._setting_fields())
        s += self._format('Metric', self._metric_fields())
        s += 'Queriable Fields\n'
        s += self._describe_fields().__str__()
        print s

    def _setting_fields(self):
        "Return model fields related to input setting"
        return {}

    def _method_fields(self):
        "Return model fields related to model methods"
        return {}

    def _result_fields(self):
        return {'graph': "SGraph. See m['graph']"}

    def _metric_fields(self):
        "Return model fields related to training metric"
        return {'runtime': self['runtime']}
