from graphlab.data_structures.sgraph import SGraph as _SGraph
import graphlab.toolkits.main as _main
from graphlab.toolkits.graph_analytics.model_base import GraphAnalyticsModel as _ModelBase


def get_default_options():
    """
    Get the default options for :func:`graphlab.pagerank.create`.

    Returns
    -------
    out : dict

    Examples
    --------
    >>> graphlab.pagerank.get_default_options()
    """
    return _main.run('pagerank_default_options', {})


class PagerankModel(_ModelBase):
    """
    Model object containing the pagerank value for each vertex.
    The pagerank value characterize the importance of a vertex
    in the graph using the following recursive definition:

        .. math::
          pr(i) =  RESET_PROB + (1-RESET_PROB) \sum_{j\in N(i)} pr(j) / out_degree(j)

    where :math:`N(i)` is the set containing all vertices :math:`j` such that
    there is an edge going from :math:`j` to :math:`i`.

    Below is a list of queryable fields for this model:

    +----------------+-----------------------------------------------------------+
    | Field          | Description                                               |
    +================+===========================================================+
    | reset_prob     | The probablity of randomly jumps to any node in the graph |
    +----------------+-----------------------------------------------------------+
    | graph          | A new SGraph with the pagerank as a vertex property       |
    +----------------+-----------------------------------------------------------+
    | delta          | Change in pagerank for the last iteration in L1 norm      |
    +----------------+-----------------------------------------------------------+
    | pagerank       | An SFrame with each vertex's pagerank                     |
    +----------------+-----------------------------------------------------------+
    | num_iterations | Number of iterations                                      |
    +----------------+-----------------------------------------------------------+
    | threshold      | The convergence threshold in L1 norm                      |
    +----------------+-----------------------------------------------------------+
    | runtime        | Total runtime of the toolkit                              |
    +----------------+-----------------------------------------------------------+
    | max_iterations | The maximun number of iterations to run                   |
    +----------------+-----------------------------------------------------------+

    An instance of this model can be created using
    :func:`graphlab.pagerank.create`.  Do NOT construct the model directly.

    See Also
    --------
    create
    """
    def __init__(self, model):
        '''__init__(self)'''
        self.__proxy__ = model

    def _result_fields(self):
        ret = super(PagerankModel, self)._result_fields()
        ret['pagerank'] = "SFrame with each vertex's pagerank. See m['pagerank']"
        ret['delta'] = self['delta']
        return ret

    def _metric_fields(self):
        ret = super(PagerankModel, self)._metric_fields()
        ret['num_iterations'] = self['num_iterations']
        return ret

    def _setting_fields(self):
        ret = super(PagerankModel, self)._setting_fields()
        for k in ['reset_prob', 'threshold', 'max_iterations']:
            ret[k] = self[k]
        return ret


def create(graph, reset_prob=0.15,
           threshold=1e-2,
           max_iterations=20,
           verbose=True, plot=False):
    """
    Compute the PageRank for each vertex in the graph. Return a model object
    with total PageRank as well as the PageRank value for each vertex in the
    graph.

    Parameters
    ----------
    graph : SGraph
        The graph on which to compute the pagerank value.

    reset_prob : float, optional
        Probability that a random surfer jumps to an arbitrary page.

    threshold : float, optional
        Threshold for convergence, measured in the L1 norm
        (the sum of absolute value) of the delta of each vertex's
        pagerank value.

    max_iterations : int, optional
        The maximun number of iterations to run.

    verbose : bool, optional
        If True, print progress updates.

    plot : bool, optional
        If True, display the progress plot.

    Returns
    -------
    out : PagerankModel

    References
    ----------
    - `Wikipedia - PageRank <http://en.wikipedia.org/wiki/PageRank>`_
    - Page, L., et al. (1998) `The PageRank Citation Ranking: Bringing Order to
      the Web <http://ilpubs.stanford.edu:8090/422/1/1999-66.pdf>`_.

    Examples
    --------
    If given an :class:`~graphlab.SGraph` ``g``, we can create
    a :class:`~graphlab.pagerank.PageRankModel` as follows:

    >>> g = graphlab.load_graph('http://snap.stanford.edu/data/email-Enron.txt.gz', format='snap')
    >>> pr = graphlab.pagerank.create(g)

    We can obtain the page rank corresponding to each vertex in the graph ``g``
    using:

    >>> pr_out = pr['pagerank']     # SFrame

    See Also
    --------
    PagerankModel
    """
    if not isinstance(graph, _SGraph):
        raise TypeError('graph input must be a SGraph object.')

    if plot is True:
        print "The plot functionality for pagerank is not yet implemented."

    opts = {'threshold': threshold, 'reset_prob': reset_prob,
            'max_iter': max_iterations, 'graph': graph.__proxy__}
    params = _main.run('pagerank', opts, verbose, plot)
    return PagerankModel(params['model'])
