"""@package graphlab.toolkits

Defines a basic interface for a model object.
"""
import graphlab.connect as _mt
from graphlab.util import make_internal_url
import graphlab.connect.main as glconnect


def load_model(location):
    """
    Load a model. Assumes the model was previously saved in GraphLab format,
    with Model.save().

    Parameters
    ----------
    location : string
        Location of the model to load. Can be a local path or a remote URL.
        Because models are saved as directories, there is no file extension.

    Examples
    ----------
    >>> model.save('my_model_file')
    >>> loaded_model = gl.load_model('my_model_file')
    """
    _mt._get_metric_tracker().track('toolkit.model.load_model')

    return glconnect.get_unity().load_model(make_internal_url(location))


class Model(object):
    """
    This class defines the minimal interface of a model object
    storing the results from a toolkit function.

    User can query the model object via the `get` function, and
    the list of the queriable fields can be obtained via `list_fields`.

    Model object also supports dictionary type access, e.g. m['pagerank']
    is equivalent to m.get('pagerank').
    """

    """The UnityModel Proxy Object"""
    __proxy__ = None

    def name(self):
        """
        Returns the name of the model.

        Returns
        -------
        out : str
            The name of the model object.

        Examples
        --------
        >>> model_name = m.name()
        """
        return self.__class__.__name__

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get(field)`` function or ``m[field]``.

        Returns
        -------
        out : list[str]
            A list of fields that can be queried using the ``get`` method.

        Examples
        --------
        >>> fields = m.list_fields()
        """
        _mt._get_metric_tracker().track(self.__class__.__module__ + '.list_fields')
        return self.__proxy__.list_fields()

    def get(self, field):
        """Return the value for the queried field.

        Each of these fields can be queried in one of two ways:

        >>> out = m['field']
        >>> out = m.get('field')  # equivalent to previous line

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : value
            The current value of the requested field.

        """
        if field in self.list_fields():
            return self.__proxy__.get(field)
        else:
            raise KeyError('Field \"%s\" not in model. Available fields are %s.' % (field, ', '.join(self.list_fields())))

    def summary(self):
        """
        Print a summary of the model.
        The summary includes a description of training
        data, options, hyper-parameters, and statistics measured during model
        creation.

        Examples
        --------
        >>> m.summary()
        """
        _mt._get_metric_tracker().track(self.__class__.__module__ + '.summary')
        print self.__repr__()

    def __getitem__(self, key):
        return self.get(key)

    def _get_wrapper(self):
        """Return a lambda function: UnityModel -> M, for constructing model
        class M from a UnityModel proxy."""
        raise NotImplementedError

    def save(self, location):
        """
        Save the model. The model is saved as a directory which can then be
        loaded using the :py:func:`~graphlab.load_model` method.

        Parameters
        ----------
        location : string
            Target destination for the model. Can be a local path or remote URL.

        See Also
        ----------
        graphlab.load_model

        Examples
        ----------
        >>> model.save('my_model_file')
        >>> loaded_model = gl.load_model('my_model_file')

        """
        _mt._get_metric_tracker().track('toolkit.model.save')
        return glconnect.get_unity().save_model(self, make_internal_url(location))

    def __repr__(self):
        raise NotImplementedError

    def __str__(self):
        return self.__repr__()
