"""
Methods for creating and using a linear regression model.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.supervised_learning import SupervisedLearningModel
from graphlab.data_structures.sframe import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.deps import pandas as _pandas, HAS_PANDAS as _HAS_PANDAS
from graphlab import vowpal_wabbit as _vw
from graphlab.toolkits.supervised_learning import _vw_solver_warning


DEFAULT_SOLVER_OPTIONS = {
'convergence_threshold': 1e-2,
'step_size': 1.0,
'lbfgs_memory_level': 3,
'mini_batch_size': 1,
'max_iterations': 10}

DEFAULT_HYPER_PARAMETER_RANGE = {
    'L1_penalty' : [0.0, 0.001, 0.01, 0.1, 1.0, 10.0, 100.0],
    'L2_penalty' : [0.0, 0.001, 0.01, 0.1, 1.0, 10.0, 100.0]
}

def create(dataset, target, features=None, L2_penalty=1e-2, L1_penalty=0.0,
    solver='auto', feature_rescaling=True, solver_options=None, verbose=True):

    """
    Create a :class:`~graphlab.linear_regression.LinearRegressionModel` to
    predict a scalar target variable as a linear function of one or more
    features. In addition to standard numeric and categorical types, features
    can also be extracted automatically from list- or dictionary-type SFrame
    columns.

    The linear regression module can be used for ridge regression, Lasso, and
    elastic net regression (see References for more detail on these methods). By
    default, this model has an L2 regularization weight of 0.01.

    Parameters
    ----------
    dataset : SFrame 
        The dataset to use for training the model.

    target : string
        Name of the column containing the target variable.

    features : list[string], optional
        Names of the columns containing features. 'None' (the default) indicates
        that all columns except the target variable should be used as features.
        
        The features are columns in the input SFrame that can be of the
        following types:

        - *Numeric*: values of numeric type integer or float.

        - *Categorical*: values of type string.

        - *Array*: list of numeric (integer or float) values. Each list element
          is treated as a separate feature in the model.

        - *Dictionary*: key-value pairs with numeric (integer or float) values
          Each key of a dictionary is treated as a separate feature and the
          value in the dictionary corresponds to the value of the feature. 
          Dictionaries are ideal for representing sparse data.

        Columns of type *list* are not supported. Convert such feature
        columns to type array if all entries in the list are of numeric
        types. If the lists contain data of mixed types, separate
        them out into different columns.

    L2_penalty : float, optional
        Weight on the L2-regularizer of the model. The larger this weight, the
        more the model coefficients shrink toward 0. This introduces bias into
        the model but decreases variance, potentially leading to better
        predictions. The default value is 0.01; setting this parameter to 0
        corresponds to unregularized linear regression. See the ridge
        regression reference for more detail.

    L1_penalty : float, optional
        Weight on L1 regularization of the model. Like the L2 penalty, the
        higher the L1 penalty, the more the estimated coefficients shrink toward
        0. The L1 penalty, however, completely zeros out sufficiently small
        coefficients, automatically indicating features that are not useful for
        the model. The default weight of 0 prevents any features from being
        discarded. See the LASSO regression reference for more detail.

    solver : string, optional
        Solver to use for training the model. See the references for more detail
        on each solver.

        - *auto (default)*: automatically chooses the best solver for the data
          and model parameters.
        - *newton*: Newton-Raphson
        - *lbfgs*: limited memory BFGS
        - *gd*: gradient descent
        - *vw*: Vowpal Wabbit
        - *fista*: accelerated gradient descent
        - *sgd*: stochastic gradient descent

        The model is trained using a carefully engineered collection of methods 
        that are automatically picked based on the input data. The ``newton`` 
        method  works best for datasets with plenty of examples and few features
        (long datasets). Limited memory BFGS (``lbfgs``) is a robust solver for 
        wide datasets (i.e datasets with many coefficients).  ``fista`` is the 
        default solver for L1-regularized linear regression. Gradient-descent 
        (GD) is another well tuned method that can work really well on 
        L1-regularized problems.  The solvers are all automatically tuned and 
        the default options should function well. See the solver options guide 
        for setting additional parameters for each of the solvers.


    feature_rescaling: boolean, optional
        Feature rescaling is an important pre-processing step that ensures that
        all features are on the same scale. An L2-norm rescaling is performed
        to make sure that all features are of the same norm. Categorical
        features are also rescaled by rescaling the dummy variables that are
        used to represent them. The coefficients are returned in original scale
        of the problem. This process is particularly useful when features
        vary widely in their ranges. (Not applicable when solver = 'vw')


    solver_options : dict, optional
        Solver options. The options and their default values are as follows.

        +-----------------------+---------+------------------------------------------+
        |      Option           | Default |      Description                         |
        +=======================+=========+==========================================+
        | auto_tuning           |    True | Toggle step-size auto-tuner (sgd only)   |
        +-----------------------+---------+------------------------------------------+
        | convergence_threshold |    1e-2 | Desired training accuracy                |
        +-----------------------+---------+------------------------------------------+
        | lbfgs_memory_level    |     3   | Number of updates to store (lbfgs only)  |
        +-----------------------+---------+------------------------------------------+
        | max_iterations        |     10  | Max number of solver iterations          |
        +-----------------------+---------+------------------------------------------+
        | mini_batch_size       |     1   | Number of mini-batch examples (sgd only) |
        +-----------------------+---------+------------------------------------------+
        | step_size             |     1.0 | Initial solver step size (fista, sgd)    |
        +-----------------------+---------+------------------------------------------+

        Most solvers are all automatically tuned and the default options should
        function well. Changing the default settings may help in some cases.
    

        auto_tuning:

        Toggles wethere the step size in Stochastic Gradient Descent (SGD) is
        automatically in training

        convergence_threshold: 

        Convergence is tested using variation in the training objective. The 
        variation in the training objective is calculated using the difference 
        between the objective values between two steps. Consider reducing this 
        below the default value (0.01) for a more accurately trained model. 
        Beware of overfitting (i.e a model that works well only on the training 
        data) if this parameter is set to a very low value.
        
        lbfgs_memory_level: 

        The L-BFGS algorithm keeps track of gradient information from the 
        previous ``lbfgs_memory_level`` iterations. The storage requirement for 
        each of these gradients is the ``num_coefficients`` in the problem. 
        Increasing the ``lbfgs_memory_level ``can help improve the quality of 
        the model trained. Setting this to more than ``max_iterations`` has the 
        same effect as setting it to ``max_iterations``.     

        max_iterations:

        The maximum number of allowed passes through the data. More passes over 
        the data can result in a more accurately trained model. Consider 
        increasing this (the default value is 10) if the training accuracy is 
        low and the *Grad-Norm* in the display is large.

        mini_batch_size: 

        The number of examples that are considered in Stochastic Gradient 
        Descent (SGD) before a parameter update is made. 

        step_size:

        The starting step size to use for the ``fista`` and ``gd`` solvers. The
        default is set to 1.0, this is an aggressive setting. If the first
        iteration takes a considerable amount of time, reducing this parameter
        may speed up model training. 


    verbose : bool, optional (default True)
        If True, print progress updates.

    Returns
    -------
    out : LinearRegressionModel.
        A trained model of type
        :class:`~graphlab.linear_regression.LinearRegressionModel`.

    See Also
    --------
    LinearRegressionModel

    Notes
    -----
    - Categorical variables are encoded by creating dummy variables. For a
      variable with :math:`K` categories, the encoding creates :math:`K-1` dummy
      variables, while the first category encountered in the data is used as the
      baseline.

    - For prediction and evaluation of linear regression models with sparse
      dictionary inputs, new keys/columns that were not seen during training
      are silently ignored.

    - If the 'vw' solver is used, the model returns only predictions; it does
      not allow inspection of estimated effect sizes.

    - Any 'None' values in the data will result in an error being thrown.

    - A constant term is automatically added for the model intercept. This term
      is not regularized.

    - For ``solver``='vw', the ``step_size`` option is the ``learning_rate`` parameter
      in the  :class:`~graphlab.vowpal_wabbit.VWModel`.
    
    .. warning::
      The solver 'vw' will not be supported in future releases. Use 
      graphlab.vowpal_wabbit.create() for Vowpal Wabbit.


    References
    ----------
    - Hoerl, A.E. and Kennard, R.W. (1970) `Ridge regression: Biased Estimation
      for Nonorthogonal Problems
      <http://amstat.tandfonline.com/doi/abs/10.1080/00401706.1970.10488634>`_.
      Technometrics 12(1) pp.55-67

    - Tibshirani, R. (1996) `Regression Shrinkage and Selection via the Lasso <h
      ttp://www.jstor.org/discover/10.2307/2346178?uid=3739256&uid=2&uid=4&sid=2
      1104169934983>`_. Journal of the Royal Statistical Society. Series B
      (Methodological) 58(1) pp.267-288.

    - Zhu, C., et al. (1997) `Algorithm 778: L-BFGS-B: Fortran subroutines for
      large-scale bound-constrained optimization
      <http://dl.acm.org/citation.cfm?id=279236>`_. ACM Transactions on
      Mathematical Software 23(4) pp.550-560.

    - Barzilai, J. and Borwein, J. `Two-Point Step Size Gradient Methods
      <http://imajna.oxfordjournals.org/content/8/1/141.short>`_. IMA Journal of
      Numerical Analysis 8(1) pp.141-148.

    - Beck, A. and Teboulle, M. (2009) `A Fast Iterative Shrinkage-Thresholding
      Algorithm for Linear Inverse Problems
      <http://epubs.siam.org/doi/abs/10.1137/080716542>`_. SIAM Journal on
      Imaging Sciences 2(1) pp.183-202.

    - Zhang, T. (2004) `Solving large scale linear prediction problems using
      stochastic gradient descent algorithms
      <http://dl.acm.org/citation.cfm?id=1015332>`_. ICML '04: Proceedings of
      the twenty-first international conference on Machine learning p.116.

    - `Vowpal Wabbit website <http://hunch.net/~vw/>`_

    - `Vowpal Wabbit on Github
      <https://github.com/JohnLangford/vowpal_wabbit/wiki>`_

    Examples
    --------

    Given an :class:`~graphlab.SFrame` ``sf`` with a list of columns
    [``feature_1`` ... ``feature_K``] denoting features and a target column
    ``target``, we can create a
    :class:`~graphlab.linear_regression.LinearRegressionModel` as follows:

    >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

    >>> model = graphlab.linear_regression.create(data, target='price', features=['bath', 'bedroom', 'size'])
    

    For ridge regression, we can set the ``L2_penalty`` parameter higher (the
    default is 0.01). For Lasso regression, we set the L1_penalty higher, and
    for elastic net, we set both to be higher.

    .. sourcecode:: python

      # Ridge regression
      >>> model_ridge = graphlab.linear_regression.create(data, 'price', L2_penalty=0.1)
      
      # Lasso
      >>> model_lasso = graphlab.linear_regression.create(data, 'price', L2_penalty=0.,
                                                                   L1_penalty=1.0)

      # Elastic net regression
      >>> model_enet  = graphlab.linear_regression.create(data, 'price', L2_penalty=0.5,
                                                                 L1_penalty=0.5)

    """

    if verbose:
      print "Preparing the data..."
    _mt._get_metric_tracker().track('toolkit.regression.linear_regression.create')

    if not (isinstance(dataset, _SFrame) or (_HAS_PANDAS and isinstance(dataset, _pandas.DataFrame))):
        raise TypeError('Input dataset must be an SFrame or a pandas dataframe.')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)

    # Regression model names.
    model_name = "regression_linear_regression"
    solver = solver.lower()
    if solver == 'vw' or solver == 'vowpal_wabbit':
        _vw_solver_warning()

    # Make sure all keys in the dictionary are lower case.
    # Also keeps a separate copy of solver_options to prevent changes.
    if solver_options is not None:
        _solver_options = {k.lower(): v for k, v in solver_options.items()}
    else:
        _solver_options = {}

    # Extract the target columns into an SFrame.
    target_sframe = dataset.select_columns([target])

    # If features is None, then select all other columns in the table.
    if features is None:
        features = dataset.column_names()
        features.remove(target)

    # Make sure that features are of type strings.
    if not hasattr(features, '__iter__'):
        raise TypeError("Input 'features' must be an iterable")
    if not all([isinstance(x, str) for x in features]):
        raise TypeError("Invalid feature %s: Feature names must be of type str" % x)
    features_sframe = dataset.select_columns(features)

    # Make sure all options are in a flat dictionary.
    opts = {}
    opts.update(_solver_options)
    opts.update({'target'     : target_sframe,
                'features'    : features_sframe,
                'model_name'  : model_name,
                'solver'      : solver,
                'l1_penalty'  : L1_penalty,
                'feature_rescaling'  : feature_rescaling,
                'l2_penalty'  : L2_penalty})

    # Pre-training
    ret = _graphlab.toolkits.main.run("supervised_learning_train_init", opts)
    opts.update(ret)

    # Switching point for vw
    if(solver == 'vw' or solver == 'vowpal-wabbit'):
        # Make the SFrame into VW format
        required_columns = features + [target]
        sf = dataset.select_columns(required_columns)

        # Append options
        vw_options = {}
        for key in ['max_iterations', 'step_size']:
            if key in _solver_options:
                vw_options[key] = _solver_options[key]
            else:
                vw_options[key] = DEFAULT_SOLVER_OPTIONS[key]

        # Call VW
        vw_model = _vw.create(sf, target,
                              l1_penalty= L1_penalty,
                              l2_penalty= L2_penalty,
                              loss_function ='squared',
                              step_size = vw_options['step_size'],
                              verbose = verbose,
                              max_iterations = vw_options['max_iterations'])

        # Save the 'solver' option so that the regression toolkit
        # can query for it when testing which solver is active.
        vw_model._set('solver', 'vw')

        # Use the proxy object for VW
        model = LinearRegressionModel(vw_model.__proxy__, use_vw=True)
        print "\n"

    # Call all our solvers!
    else:

        # Print some output for ipython notebook users
        ret = _graphlab.toolkits.main.run("supervised_learning_train", opts,
                verbose)

        model_proxy = ret['model']
        model = LinearRegressionModel(model_proxy)

    return model


class LinearRegressionModel(SupervisedLearningModel):
    """
    Linear regression is an approach for modeling a scalar target :math:`y` as
    a linear function of one or more explanatory variables denoted :math:`X`.
    An instance of this model can be created using
    :func:`graphlab.linear_regression.create`. Do not construct the model
    directly.

    Given a set of features :math:`x_i`, and a label :math:`y_i`, linear
    regression interprets the probability that the label is in one class as
    a linear function of a linear combination of the features.

        .. math::
          f_i(\\theta) =  \\theta^T x + \epsilon_i

    where :math:`\epsilon_i` is noise.  An intercept term is added by appending
    a column of 1's to the features.  Regularization is often required to
    prevent overfitting by penalizing models with extreme parameter values. The
    linear regression module supports L1 and L2 regularization, which are added
    to the loss function.
    
    The composite objective being optimized for is the following:

        .. math::
           \min_{\\theta} \sum_{i = 1}^{n} (\\theta^Tx - y_i)^2 + \lambda_1 ||\\theta||_1 + \lambda_2 ||\\theta||^{2}_{2} 

    where :math:`\lambda_1` is the ``L1_penalty`` and :math:`\lambda_2` is the
    ``L2_penalty``.


    Examples
    --------
    .. sourcecode:: python

        # Load the data (From an S3 bucket)
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        # Make a linear regression model
        >>> model = graphlab.linear_regression.create(data, target='price', features=['bath', 'bedroom', 'size'])

        # Here, all columns (other than the target) are used as features.
        >>> model = graphlab.linear_regression.create(data, target='price')

        # Extract the coefficients
        >>> coefficients = model['coefficients']   
        
        # Make predictions (as margins, probability, or class)
        >>> predictions = model.predict(data)      

        # Evaluate the model 
        >>> results = model.evaluate(data)         
    
    See Also
    --------
    create

    """
    def __init__(self, model_proxy, use_vw=False):
        '''__init__(self)'''
        self.__proxy__ = model_proxy
        self.__name__ = "regression_linear_regression"
        self.use_vw = use_vw

    def _get_wrapper(self):
        def vw_model_wrapper(model_proxy):
            vw_model = _vw.VWModel(model_proxy)
            return LinearRegressionModel(model_proxy, use_vw=True)
        def model_wrapper(model_proxy):
            return LinearRegressionModel(model_proxy)

        if (self.get('solver') == 'vw'):
            return vw_model_wrapper
        else:
            return model_wrapper

    def __str__(self):
        """
        Return a string description of the model, including a description of
        the training data, training statistics, and model hyper-parameters.

        Returns
        -------
        out : string
            A description of the model.
        """

        return self.__repr__()

    def __repr__(self):
        """
        Return a string description of the model, including a description of
        the training data, training statistics, and model hyper-parameters.

        Returns
        -------
        out : string
            A description of the model.
        """

        solver = self.get('solver')
        width = 24
        key_str = "{:<{}}: {}"
        model_fields = [
            ("L1 penalty", 'l1_penalty'),
            ("L2 penalty", 'l2_penalty'),
            ("Examples", 'num_examples'),
            ("Features", 'num_features'),
            ("Coefficients", 'num_coefficients')]

        solver_fields = [
            ("Solver", 'solver'),
            ("Solver iterations", 'train_iters'),
            ("Solver status", 'solver_status'),
            ("Training time (sec)", 'train_time')]

        train_fields = [
            ("Residual sum of squares", 'train_loss'),
            ("Training RMSE", 'train_rmse')]

        ret = []
        ret.append(key_str.format("Class", width, self.__class__.__name__))

        if solver == 'vw':
            m = _vw.VWModel(self.__proxy__)
            ret.append(m.__repr__())

        else:
            for tranche_fields in [model_fields, solver_fields, train_fields]:
                for k, v in tranche_fields:
                    value = self.get(v)
                    if isinstance(value, float):
                        try:
                            value = round(value, 4)
                        except:
                            pass
                    ret.append(key_str.format(k, width, value))
                ret.append("")

        return '\n'.join(ret)

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.linear_regression.LinearRegressionModel.list_fields`
        method.
        
        +-----------------------+----------------------------------------------+
        |      Field            | Description                                  |
        +=======================+==============================================+
        | auto_tuning           | True if auto-tuning was used during training |
        +-----------------------+----------------------------------------------+
        | coefficients          | Regression coefficients (non-'vw' only)      |
        +-----------------------+----------------------------------------------+
        | convergence_threshold | Desired solver accuracy                      |
        +-----------------------+----------------------------------------------+
        | feature_rescaling     | Bool indicating l2-rescaling of features     |
        +-----------------------+----------------------------------------------+
        | features              | Feature column names                         |
        +-----------------------+----------------------------------------------+
        | l1_penalty            | L1 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | l2_penalty            | L2 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | lbfgs_memory_level    | LBFGS memory level ('lbfgs only')            |
        +-----------------------+----------------------------------------------+
        | max_iterations        | Maximum number of solver iterations          |
        +-----------------------+----------------------------------------------+
        | mini_batch_size       | Size of mini-batches ('sgd' only)            |
        +-----------------------+----------------------------------------------+
        | num_coefficients      | Number of coefficients in the model          |
        +-----------------------+----------------------------------------------+
        | num_examples          | Number of examples used for training         |
        +-----------------------+----------------------------------------------+
        | num_features          | Number of dataset columns used for training  |
        +-----------------------+----------------------------------------------+
        | solver                | Type of solver                               |
        +-----------------------+----------------------------------------------+
        | solver_status         | Solver status after training                 |
        +-----------------------+----------------------------------------------+
        | step_size             | Initial step size for the solver             |
        +-----------------------+----------------------------------------------+
        | target                | Target column name                           |
        +-----------------------+----------------------------------------------+
        | train_iters           | Number of solver iterations                  |
        +-----------------------+----------------------------------------------+
        | train_loss            | Residual sum-of-squares training loss        |
        +-----------------------+----------------------------------------------+
        | train_rmse            | Training root-mean-squared-error (RMSE)      |
        +-----------------------+----------------------------------------------+
        | train_time            | Training time (excludes preprocessing)       |
        +-----------------------+----------------------------------------------+


        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

        See Also
        --------
        list_fields

        Examples
        --------

        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> print model['num_features']
        3
        >>> print model.get('num_features')       # equivalent to previous line
        3

        """

        _mt._get_metric_tracker().track('toolkit.regression.linear_regression.get')
        # Note: For VW as the solver, map the options back to the options
        # stored in VW. This lets us have control over what the options
        # should look like when VW is the solver.
        if self.use_vw:
            if field == 'coefficients':
                raise ValueError("Models trained with 'vw' as the "\
                                "solver cannot provide coefficients.")
            vw_option_map = {'solver'     : 'solver',
                             'target'     : 'target',
                             'step_size'  : 'step_size',
                             'l1_penalty' : 'l1_penalty',
                             'l2_penalty' : 'l2_penalty',
                             'max_iterations'  : 'max_iterations',
                             'train_iters': 'max_iterations',
                             'train_time' : 'elapsed_time',
                             'train_loss' : 'loss_function',
                             'train_rmse' : 'train_rmse'}

            # Only return options in the map
            if field not in vw_option_map:
                raise ValueError, "Key %s does not exist. Use " \
                        "list_fields() for a list of keys that can be " \
                        "queried." % field
            return _vw.VWModel(self.__proxy__).get(vw_option_map[field])
        else:
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__,
                    'field': field}
            response = _graphlab.toolkits.main.run('supervised_learning_get_value',
                opts)

            # Coefficients returns a unity SFrame. Cast to an SFrame.
            if field == 'coefficients':
                return _SFrame(None, _proxy=response['value'])
            else:
                return response['value']

    def summary(self):
        """
        Display a summary of the LinearRegressionModel.

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> model.summary()


        """
        _mt._get_metric_tracker().track('toolkit.regression.linear_regression.summary')
        solver = self.get('solver')
        if solver == 'vw':
            _vw.VWModel(self.__proxy__).summary()
            return
        else:
            coefs = self.get('coefficients')
            top_coefs = coefs.topk('Coefficient', k=6)
            top_coefs = top_coefs[top_coefs['Coefficient'] > 0]

            bottom_coefs = coefs.topk('Coefficient', k=5, reverse=True)
            bottom_coefs = bottom_coefs[bottom_coefs['Coefficient'] < 0]

            print ""
            print "                    Model summary                       "
            print "--------------------------------------------------------"
            print self.__repr__()

            print "             Strongest positive coefficients            "
            print "--------------------------------------------------------"
            if len(top_coefs) > 0:
                print _SFrame(top_coefs)
            else:
                print "[No positive coefficients]"

            print "             Strongest negative coefficients            "
            print "--------------------------------------------------------"
            if len(bottom_coefs) > 0:
                print _SFrame(bottom_coefs)
            else:
                print "[No negative coefficients]"

    def get_default_options(self):
        """
        A dictionary describing the default options for the LinearRegressionModel.

        Returns
        -------
        out : dict
             A dictionary with default option (name, value) pairs.
        
        See Also
        --------
        get_current_options, list_fields, get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> default_options = model.get_default_options()
        """

        _mt._get_metric_tracker().track('toolkit.regression.linear_regression.get_default_options')
        if self.get('solver') == 'vw':
            return {'max_iterations': 10, 'step_size': 1.0}
        else:
            return super(LinearRegressionModel, self).get_default_options()

    def get_current_options(self):
        """
        A dictionary describing the options requested during training.

        Returns
        -------
        out : dict
             A dictionary with option (name, value) pairs requested during
             train time.
        
        See Also
        --------
        get_current_options, list_fields, get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> current_options = model.get_current_options()
        """

        _mt._get_metric_tracker().track('toolkit.regression.linear_regression.get_options')
        if self.get('solver') == 'vw':
            max_iterations = _vw.VWModel(self.__proxy__).get('max_iterations')
            step_size = _vw.VWModel(self.__proxy__).get('step_size')
            return {'max_iterations': max_iterations, 'step_size': step_size}
        else:
            return super(LinearRegressionModel, self).get_current_options()

    def predict(self, dataset, missing_value_action='impute'):
        """
        Return target value predictions for ``dataset``, using the trained
        linear regression model. This method can be used to get fitted values
        for the model by inputting the training dataset.

        Parameters
        ----------
        dataset : SFrame/pandas.Dataframe
            Dataset of new observations. Must include columns with the same
            names as the features used for model training, but does not require
            a target column. Additional columns are ignored.

        missing_value_action: str, optional
            Action to perform when missing values are encountered. This can be 
            one of:

            - 'impute': Proceed with evaluation by filling in the missing
                        values with the mean of the training data. Missing
                        values are also imputed if an entire column of data is
                        missing during evaluation.
            - 'error' : Do not proceed with prediction and terminate with 
                        an error message.


        Returns
        -------
        out : SArray
            Predicted target value for each example (i.e. row) in the dataset.
        
        See Also
        ----------
        create, evaluate

        Examples
        ----------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> results = model.predict(data)
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.linear_regression.predict')

        # Special handling of vw model
        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            predictions = m.predict(dataset)
            return predictions
        else:
            # Get metadata from C++ object
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__,
                    'dataset': dataset,
                    'missing_value_action': missing_value_action,
                    'output_type': ""}

            response = _graphlab.toolkits.main.run('supervised_learning_predict_init', opts)
            opts.update(response)
            response = _graphlab.toolkits.main.run('supervised_learning_predict', opts)
            return _SArray(None, _proxy=response['predicted'])

    def evaluate(self, dataset, missing_value_action = 'impute'):
        r"""Evaluate the model by making target value predictions and comparing
        to actual values.

        Two metrics are used to evaluate linear regression models.  The first
        is root-mean-squared error (RMSE) while the second is the absolute
        value of the maximum error between the actual and predicted values.
        Let :math:`y` and :math:`\hat{y}` denote vectors of length :math:`N`
        (number of examples) with actual and predicted values. The RMSE is
        defined as:

        .. math::

            RMSE = \sqrt{\frac{1}{N} \sum_{i=1}^N (\widehat{y}_i - y_i)^2}

        while the max-error is defined as

        .. math::

            max-error = \max_{i=1}^N \|\widehat{y}_i - y_i\|

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the target and features used for model training. Additional
            columns are ignored.
        
        missing_value_action: str, optional
            Action to perform when missing values are encountered. This can be 
            one of:

            - 'impute': Proceed with evaluation by filling in the missing
                        values with the mean of the training data. Missing
                        values are also imputed if an entire column of data is
                        missing during evaluation.
            - 'error' : Do not proceed with evaluation and terminate with 
                        an error message.

        Returns
        -------
        out : dict
            Results from  model evaluation procedure.

        See Also
        ----------
        create, predict

        Examples
        ----------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> results = model.evaluate(data)
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.linear_regression.evaluate')
        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            evaluation_results = m.evaluate(dataset)
            return evaluation_results

        opts = {'model': self.__proxy__,
                'dataset': dataset,
                'missing_value_action': missing_value_action,
                'model_name': self.__name__}

        response = _graphlab.toolkits.main.run('supervised_learning_evaluate_init', opts)
        opts.update(response)
        response = _graphlab.toolkits.main.run('supervised_learning_evaluate', opts)
        opts.update(response)
        return _graphlab.toolkits.main.run('supervised_learning_get_evaluate_stats', opts)

    def list_fields(self):
        """
        List of fields stored in the model. Each of these fields can be queried
        using the ``get`` function. Note that the list of fields that can be
        queried by the model are different when the chosen solver is ``vw``.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        
        See Also
        --------
        get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> model = graphlab.linear_regression.create(data,
                                             target='price',
                                             features=['bath', 'bedroom', 'size'])
        >>> model.list_fields()
        """
        _mt._get_metric_tracker().track('toolkit.regression.linear_regression.list_fields')
        solver = self.get('solver')
        # Hard code the options exposed from the VW model proxy.
        if solver == 'vw':
            return [
                    'train_iters',
                    'l1_penalty',
                    'l2_penalty',
                    'max_iterations',
                    'solver',
                    'step_size',
                    'target',
                    'train_loss',
                    'train_rmse',
                    'train_time',
                    ]
        else:
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__}
            response = _graphlab.toolkits.main.run('supervised_learning_list_keys', opts)
            return sorted(response.keys())
