"""
Methods for creating and using a logistic regression model.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits.supervised_learning import SupervisedLearningModel
from graphlab.data_structures.sframe import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab import vowpal_wabbit as _vw
from graphlab.deps import pandas as _pandas, HAS_PANDAS as _HAS_PANDAS
import math
from graphlab.toolkits.main import ToolkitError
from graphlab.toolkits.supervised_learning import _vw_solver_warning

DEFAULT_SOLVER_OPTIONS = {
'convergence_threshold': 1e-2,
'step_size': 1.0,
'mini_batch_size': 1,
'max_iterations': 10}

DEFAULT_HYPER_PARAMETER_RANGE = {
    'L1_penalty' : [0.0, 0.001, 0.01, 0.1, 1.0, 10.0, 100.0],
    'L2_penalty' : [0.0, 0.001, 0.01, 0.1, 1.0, 10.0, 100.0]
}

def create(dataset, target, features=None, L2_penalty=0.01, L1_penalty=0.0,
    solver='auto', feature_rescaling=True, solver_options=None, verbose=True):
    """
    Create a :class:`~graphlab.logistic_regression.LogisticRegressionModel` to
    predict the class of a binary target variable based on a model of class
    probability as a logistic function of a linear combination of the features.
    In addition to standard numeric and categorical types, features can also be
    extracted automatically from list- or dictionary-type SFrame columns.

    This model can be regularized with an L1 penalty, an L2 penalty, or both. By
    default this model has an L2 regularization weight of 0.01.

    Parameters
    ----------
    dataset : SFrame
        Dataset for training the model.

    target : string
        Name of the column containing the target variable. The values in this
        column must be 0 or 1, of integer type.

    features : list[string], optional
        Names of the columns containing features. 'None' (the default) indicates
        that all columns except the target variable should be used as features.

        The features are columns in the input SFrame that can be of the
        following types:

        - *Numeric*: values of numeric type integer or float.

        - *Categorical*: values of type string.

        - *Array*: list of numeric (integer or float) values. Each list element
          is treated as a separate feature in the model.

        - *Dictionary*: key-value pairs with numeric (integer or float) values
          Each key of a dictionary is treated as a separate feature and the
          value in the dictionary corresponds to the value of the feature. 
          Dictionaries are ideal for representing sparse data.
          
        Columns of type *list* are not supported. Convert such feature
        columns to type array if all entries in the list are of numeric
        types. If the lists contain data of mixed types, separate
        them out into different columns.

    L2_penalty : float, optional
        Weight on L2 regularization of the model. The larger this weight, the
        more the model coefficients shrink toward 0. This introduces bias into
        the model but decreases variance, potentially leading to better
        predictions. The default value is 0.01; setting this parameter to 0
        corresponds to unregularized logistic regression. See the ridge
        regression reference for more detail.

    L1_penalty : float, optional
        Weight on L1 regularization of the model. Like the L2 penalty, the
        higher the L1 penalty, the more the estimated coefficients shrink toward
        0. The L1 penalty, however, completely zeros out sufficiently small
        coefficients, automatically indicating features that are not useful
        for the model. The default weight of 0 prevents any features from
        being discarded. See the LASSO regression reference for more detail.

    solver : string, optional
        Name of the solver to be used to solve the regression. See the
        references for more detail on each solver. Available solvers are:

        - *auto (default)*: automatically chooses the best solver for the data
          and model parameters.
        - *newton*: Newton-Raphson
        - *lbfgs*: limited memory BFGS
        - *vw*: Vowpal Wabbit
        - *fista*: accelerated gradient descent

        For this model, the Newton-Raphson method is equivalent to the
        iteratively re-weighted least squares algorithm. If the L1_penalty is
        greater than 0, use either the 'fista' or 'vw' solver.

        The model is trained using a carefully engineered collection of methods 
        that are automatically picked based on the input data. The ``newton`` 
        method  works best for datasets with plenty of examples and few features
        (long datasets). Limited memory BFGS (``lbfgs``) is a robust solver for 
        wide datasets (i.e datasets with many coefficients).  ``fista`` is the 
        default solver for L1-regularized linear regression. The solvers are all
        automatically tuned and the default options should function well. See 
        the solver options guide for setting additional parameters for each of 
        the solvers.

    feature_rescaling: boolean, optional

        Feature rescaling is an important pre-processing step that ensures that
        all features are on the same scale. An L2-norm rescaling is performed
        to make sure that all features are of the same norm. Categorical
        features are also rescaled by rescaling the dummy variables that are
        used to represent them. The coefficients are returned in original scale
        of the problem. This process is particularly useful when features
        vary widely in their ranges.  (Not applicable when solver = 'vw')

    solver_options : dict, optional
        Solver options. The options and their default values are as follows:

        +-----------------------+---------+-----------------------------------------+
        |      Option           | Default |        Description                      |
        +=======================+=========+=========================================+
        | convergence_threshold |    1e-2 | Desired training accuracy               |
        +-----------------------+---------+-----------------------------------------+
        | lbfgs_memory_level    |     3   | Number of updates to store (lbfgs only) |
        +-----------------------+---------+-----------------------------------------+
        | max_iterations        |     10  | Max number of solver iterations         |
        +-----------------------+---------+-----------------------------------------+
        | step_size             |    1.0  | Initial solver step size (fista only)   |
        +-----------------------+---------+-----------------------------------------+

        Most solvers are all automatically tuned and the default options should
        function well. Changing the default settings may help in some cases.
    

        convergence_threshold: 

        Convergence is tested using variation in the training objective. The 
        variation in the training objective is calculated using the difference 
        between the objective values between two steps. Consider reducing this 
        below the default value (0.01) for a more accurately trained model. 
        Beware of overfitting (i.e a model that works well only on the training 
        data) if this parameter is set to a very low value.
        
        lbfgs_memory_level: 

        The L-BFGS algorithm keeps track of gradient information from the 
        previous ``lbfgs_memory_level`` iterations. The storage requirement for 
        each of these gradients is the ``num_coefficients`` in the problem. 
        Increasing the ``lbfgs_memory_level ``can help improve the quality of 
        the model trained. Setting this to more than ``max_iterations`` has the 
        same effect as setting it to ``max_iterations``.     

        max_iterations:

        The maximum number of allowed passes through the data. More passes over 
        the data can result in a more accurately trained model. Consider 
        increasing this (the default value is 10) if the training accuracy is 
        low and the *Grad-Norm* in the display is large.

        step_size:

        The starting step size to use for the ``fista`` and ``gd`` solvers. The
        default is set to 1.0, this is an aggressive setting. If the first
        iteration takes a considerable amount of time, reducing this parameter
        may speed up model training. 


    verbose : bool, optional
        If True, print progress updates.

    Returns
    -------
    out : LogisticRegressionModel
        A trained model of type
        :class:`~graphlab.logistic_regression.LogisticRegressionModel`.
    
    See Also
    --------
    LogisticRegressionModel

    Notes
    -----
    - Categorical variables are encoded by creating dummy variables. For a
      variable with :math:`K` categories, the encoding creates :math:`K-1` dummy
      variables, while the first category encountered in the data is used as the
      baseline.

    - For prediction and evaluation of logistic regression models with sparse
      dictionary inputs, new keys/columns that were not seen during training
      are silently ignored.

    - If the 'vw' solver is used, the model returns only predictions; it does
      not allow inspection of estimated effect sizes or standard errors.

    - During model creation, 'None' values in the data will result in an error
      being thrown.

    - A constant term is automatically added for the model intercept. This term
      is not regularized.
    
    - For solver='vw', the "step_size" option is the "learning_rate" parameter
      in the  :class:`~graphlab.vowpal_wabbit.VWModel`. 
    

    References
    ----------
    - `Wikipedia - logistic regression
      <http://en.wikipedia.org/wiki/Logistic_regression>`_

    - Hoerl, A.E. and Kennard, R.W. (1970) `Ridge regression: Biased Estimation
      for Nonorthogonal Problems
      <http://amstat.tandfonline.com/doi/abs/10.1080/00401706.1970.10488634>`_.
      Technometrics 12(1) pp.55-67

    - Tibshirani, R. (1996) `Regression Shrinkage and Selection via the Lasso <h
      ttp://www.jstor.org/discover/10.2307/2346178?uid=3739256&uid=2&uid=4&sid=2
      1104169934983>`_. Journal of the Royal Statistical Society. Series B
      (Methodological) 58(1) pp.267-288.

    - Zhu, C., et al. (1997) `Algorithm 778: L-BFGS-B: Fortran subroutines for
      large-scale bound-constrained optimization
      <http://dl.acm.org/citation.cfm?id=279236>`_. ACM Transactions on
      Mathematical Software 23(4) pp.550-560.

    - Beck, A. and Teboulle, M. (2009) `A Fast Iterative Shrinkage-Thresholding
      Algorithm for Linear Inverse Problems
      <http://epubs.siam.org/doi/abs/10.1137/080716542>`_. SIAM Journal on
      Imaging Sciences 2(1) pp.183-202.

    - `Vowpal Wabbit website <http://hunch.net/~vw/>`_

    - `Vowpal Wabbit on Github
      <https://github.com/JohnLangford/vowpal_wabbit/wiki>`_


    Examples
    --------

    Given an :class:`~graphlab.SFrame` ``sf``, a list of feature columns
    [``feature_1`` ... ``feature_K``], and a target column ``target`` with 0 and
    1 values, create a
    :class:`~graphlab.logistic_regression.LogisticRegressionModel` as follows:

    >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')
    >>> data['is_expensive'] = data['price'] > 30000
    >>> model = graphlab.logistic_regression.create(data, 'is_expensive')

    By default all columns of ``data`` except the target are used as features, but
    specific feature columns can be specified manually.

    >>> model = graphlab.logistic_regression.create(data, 'is_expensive', ['bedroom', 'size'])


    .. sourcecode:: python

      # L2 normalization
      >>> model_ridge = graphlab.logistic_regression.create(data, 'is_expensive', L2_penalty=0.1)
      
      # L1 normalization
      >>> model_lasso = graphlab.logistic_regression.create(data, 'is_expensive', L2_penalty=0.,
                                                                   L1_penalty=1.0)

      # L1 and L2 normalization
      >>> model_enet  = graphlab.logistic_regression.create(data, 'is_expensive', L2_penalty=0.5, L1_penalty=0.5)
    """

    if verbose:
      print "Preparing the data..."
    model_name = "regression_logistic_regression"
    _mt._get_metric_tracker().track(
        'toolkit.regression.logistic_regression.create')

    ## Check data matrix type and convert to SFrame
    if not (isinstance(dataset, _SFrame) or (_HAS_PANDAS and
      isinstance(dataset, _pandas.DataFrame))):
        raise TypeError('Dataset input must be an SFrame.')

    if type(dataset) != _SFrame:
        dataset = _SFrame(dataset)


    # Make the solver name and all solver options lower case. Also keeps a
    # separate copy of solver_options to prevent changes.
    solver = solver.lower()
    if solver == 'vw' or solver == 'vowpal_wabbit':
        _vw_solver_warning()

    if solver_options is not None:
        _solver_options = {k.lower(): v for k,v in solver_options.items()}
    else:
        _solver_options = {}

    # Extract the target vector and check the type and values
    sf_target = dataset.select_columns([target])

    target_type = sf_target[target].dtype()
    if not target_type == int:
        raise TypeError("Target column must be type int")

    # Check that target values are all +1 or 0
    zero_or_one = (sf_target[target] == 0) + (sf_target[target] == 1)
    if not zero_or_one.all():
        raise ValueError("The target column for logistic regression" + \
            " must contain only 0 and 1 values.")

    # Extract the features
    if features is None:
        features = dataset.column_names()
        features.remove(target)

    if not hasattr(features, '__iter__'):
        raise TypeError("Input 'features' must be an iterable")

    if not all([isinstance(x, str) for x in features]):
        raise TypeError("Invalid type: predictor names must be str")

    sf_features = dataset.select_columns(features)


    # Set up options dictionary and initialize the model
    opts = {}
    opts.update(_solver_options)
    opts.update({'target'     : sf_target,
                'features'    : sf_features,
                'model_name'  : model_name,
                'solver'      : solver,
                'feature_rescaling'  : feature_rescaling,
                'l2_penalty'  : L2_penalty,
                'l1_penalty'  : L1_penalty})


    # Model checking
    ret = _graphlab.toolkits.main.run("supervised_learning_train_init", opts)
    opts.update(ret)
    model = LogisticRegressionModel(ret['model'])


    if solver == 'vw' or solver == 'vowpal_wabbit':
        # Make the SFrame into VW format
        required_columns = features + [target]
        sf = dataset.select_columns(required_columns)

        # Translate the 0/1 targets to -1/+1 targets
        sf[target] = sf[target] * 2 - 1

        # Append options
        vw_options = {}
        for key in ['max_iterations', 'step_size']:
            if key in _solver_options:
                vw_options[key] = _solver_options[key]
            else:
                vw_options[key] = DEFAULT_SOLVER_OPTIONS[key]

        # Call VW
        # Note: VW checks that the target columns has +1 or -1
        vw_model = _vw.create(sf, target,
                              l1_penalty = L1_penalty,
                              l2_penalty = L2_penalty,
                              loss_function ='logistic',
                              step_size = vw_options['step_size'],
                              verbose = verbose,
                              max_iterations = vw_options['max_iterations'])

        # Save the 'solver' option so that the regression toolkit
        # can query for it when testing which solver is active.
        vw_model._set('solver', 'vw')

        # Use the proxy object for VW
        model = LogisticRegressionModel(vw_model.__proxy__, use_vw=True)

    else:  # all other solvers beside VW

        try:
            ret = _graphlab.toolkits.main.run("supervised_learning_train", opts, verbose)
        except:
            raise ValueError("Model failed to train")
        else:
            model_proxy = ret['model']
            model = LogisticRegressionModel(model_proxy)

    return model


class LogisticRegressionModel(SupervisedLearningModel):
    """
    Logistic regression models a binary target variable as a function of
    several feature variables. An instance of this model can be created using
    :func:`graphlab.logistic_regression.create`.  Do not construct the model
    directly. A detailed list of options are available in the create function.


    The :class:`~graphlab.logistic_regression.LogisticRegressionModel` uses
    a binary target variable :math:`y` instead of a scalar. For each
    observation, the probability that :math:`y=1` (instead of 0) is modeled as
    the logistic function of a linear combination of the feature values.
    Logistic regression predictions may take the form of the linear combination
    (also known as the margin), the probability that :math:`y = 1`, or a binary
    class value, which is obtained by thresholding the probability at 0.5. The
    :class:`~graphlab.evaluation` toolkit gives you more flexible evaluation
    metrics.
    
    
    Given a set of features :math:`x_i`, and a label :math:`y_i \in \{0,1\}`,
    logistic regression interprets the probability that the label is in one class
    as a logistic function of a linear combination of the features.

        .. math::
          f_i(\\theta) =  p(y_i = 1 | x) = \\frac{1}{1 + \exp(-\\theta^T x)}

    An intercept term is added by appending a column of 1's to the features.
    Regularization is often required to prevent over fitting by penalizing
    models with extreme parameter values. The logistic regression module
    supports L1 and L2 regularization, which are added to the loss function.
    
    The composite objective being optimized for is the following:

        .. math::
           \min_{\\theta} \sum_{i = 1}^{n} f_i(\\theta) + \lambda_1 ||\\theta||_1 + \lambda_2 ||\\theta||^{2}_{2} 

    where :math:`\lambda_1` is the ``L1_penalty`` and :math:`\lambda_2` is the
    ``L2_penalty``.

    Examples
    --------
    .. sourcecode:: python

        # Load the data (From an S3 bucket)
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        # Make sure the target is binary 0/1
        >>> data['is_expensive'] = data['price'] > 30000
        
        # Make a logistic regression model
        >>> model = graphlab.logistic_regression.create(data, target='is_expensive', features=['bath', 'bedroom', 'size'])

        # Extract the coefficients
        >>> coefficients = model['coefficients']
        
        # Make predictions (as margins, probability, or class)
        >>> predictions = model.predict(data)    
        >>> predictions = model.predict(data, output_type='probability')
        >>> predictions = model.predict(data, output_type='margin')       

        # Evaluate the model 
        >>> results = model.evaluate(data)
    
    See Also
    --------
    create


    """
    def __init__(self, model_proxy, use_vw=False):
        '''__init__(self)'''
        self.__proxy__ = model_proxy
        self.__name__ = "regression_logistic_regression"
        self.use_vw = use_vw

    def _get_wrapper(self):

        def vw_model_wrapper(model_proxy):
            vw_model = _vw.VWModel(model_proxy)
            return LogisticRegressionModel(model_proxy, use_vw=True)

        def model_wrapper(model_proxy):
            return LogisticRegressionModel(model_proxy)

        if self.get('solver') == 'vw':
            return vw_model_wrapper
        else:
            return model_wrapper

    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model, when the model name is entered
        in the terminal.
        """

        solver = self.get('solver')
        width = 20
        key_str = "{:<{}}: {}"
        model_fields = [
            ("L1 penalty", 'l1_penalty'),
            ("L2 penalty", 'l2_penalty'),
            ("Examples", 'num_examples'),
            ("Features", 'num_features'),
            ("Coefficients", 'num_coefficients')]

        solver_fields = [
            ("Solver", 'solver'),
            ("Solver iterations", 'train_iters'),
            ("Solver status", 'solver_status'),
            ("Training time (sec)", 'train_time')]

        train_fields = [
            ("Log-likelihood", 'train_loss')]

        ret = []
        ret.append(key_str.format("Class", width, self.__class__.__name__))

        if solver == 'vw':
            m = _vw.VWModel(self.__proxy__)
            ret.append(m.__repr__())

        else:
            for tranche_fields in [model_fields, solver_fields, train_fields]:
                for k, v in tranche_fields:
                    value = self.get(v)
                    if isinstance(value, float):
                        try:
                            value = round(value, 4)
                        except:
                            pass
                    ret.append(key_str.format(k, width, value))
                ret.append("")

        return '\n'.join(ret)

    def get(self, field):
        """
        Return the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically with the
        :func:`~graphlab.logistic_regression.LogisticRegressionModel.list_fields`
        method. Note that some options are not available if the model is trained
        with the ``vw`` solver.

        +-----------------------+----------------------------------------------+
        |      Field            | Description                                  |
        +=======================+==============================================+
        | coefficients          | Regression coefficients (non-'vw' only)      |
        +-----------------------+----------------------------------------------+
        | convergence_threshold | Desired solver accuracy                      |
        +-----------------------+----------------------------------------------+
        | feature_rescaling     | Bool indicating l2-rescaling of features     |
        +-----------------------+----------------------------------------------+
        | features              | Feature column names                         |
        +-----------------------+----------------------------------------------+
        | l1_penalty            | L1 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | l2_penalty            | L2 regularization weight                     |
        +-----------------------+----------------------------------------------+
        | lbfgs_memory_level    | LBFGS memory level                           |
        +-----------------------+----------------------------------------------+
        | max_iterations        | Maximum number of solver iterations          |
        +-----------------------+----------------------------------------------+
        | num_coefficients      | Number of coefficients in the model          |
        +-----------------------+----------------------------------------------+
        | num_examples          | Number of examples used for training         |
        +-----------------------+----------------------------------------------+
        | num_features          | Number of dataset columns used for training  |
        +-----------------------+----------------------------------------------+
        | solver                | Type of solver                               |
        +-----------------------+----------------------------------------------+
        | solver_status         | Solver status after training                 |
        +-----------------------+----------------------------------------------+
        | step_size             | Initial step size for the solver             |
        +-----------------------+----------------------------------------------+
        | target                | Target column name                           |
        +-----------------------+----------------------------------------------+
        | train_iters           | Number of solver iterations                  |
        +-----------------------+----------------------------------------------+
        | train_loss            | Maximized Log-likelihood                     |
        +-----------------------+----------------------------------------------+
        | train_time            | Training time (excludes preprocessing)       |
        +-----------------------+----------------------------------------------+
        | trained               | Indicates whether model is trained           |
        +-----------------------+----------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out
            Value of the requested fields.

        See Also
        --------
        list_fields

        Examples
        --------

        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])
        >>> print model['num_features']
        3
        >>> print model.get('num_features')       # equivalent to previous line
        3


        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.get')

        # Note: For VW as the solver, map the options back to the options
        # stored in VW. This lets us have control over what the options
        # should look like when VW is the solver.
        if self.use_vw:
            if field == 'coefficients':
                raise ValueError("Models trained with 'vw' as the" + \
                                "solver cannot provide coefficients.")
            vw_option_map = {'solver'         : 'solver',
                             'target'         : 'target',
                             'step_size'      : 'step_size',
                             'l1_penalty'     : 'l1_penalty',
                             'l2_penalty'     : 'l2_penalty',
                             'max_iterations'      : 'max_iterations',
                             'train_iters'    : 'max_iterations',
                             'train_time'     : 'elapsed_time',
                             'train_loss'     : 'loss_function',
                             'train_accuracy' : 'train_accuracy'}

            # Only return options in the map
            if field not in vw_option_map:
                raise ValueError, "Field '{}' does not exist. Use".format(field) + \
                    " list_fields() for a list of fields that can be queried."
            return _vw.VWModel(self.__proxy__).get(vw_option_map[field])

        else:
            opts = {'model': self.__proxy__, 'model_name': self.__name__,
                    'field': field}
            response = _graphlab.toolkits.main.run('supervised_learning_get_value', opts)

            # Coefficients returns a unity SFrame. Cast to an SFrame.
            if field == 'coefficients':
                return _SFrame(None, _proxy=response['value'])
            else:
                return response['value']

    def summary(self):
        """
        Display a summary of the LogisticRegressionModel.

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])
        >>> model.summary()
        """
        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.summary')
        solver = self.get('solver')
        if solver == 'vw':
            _vw.VWModel(self.__proxy__).summary()
            return
        else:
            coefs = self.get('coefficients')
            top_coefs = coefs.topk('Coefficient', k=6)
            top_coefs = top_coefs[top_coefs['Coefficient'] > 0]

            bottom_coefs = coefs.topk('Coefficient', k=5, reverse=True)
            bottom_coefs = bottom_coefs[bottom_coefs['Coefficient'] < 0]

            print ""
            print "                    Model summary                       "
            print "--------------------------------------------------------"
            print self.__repr__()

            print "             Strongest positive coefficients            "
            print "--------------------------------------------------------"
            if len(top_coefs) > 0:
                print _SFrame(top_coefs)
            else:
                print "[No positive coefficients]"

            print "             Strongest negative coefficients            "
            print "--------------------------------------------------------"
            if len(bottom_coefs) > 0:
                print _SFrame(bottom_coefs)
            else:
                print "[No negative coefficients]"

    def get_default_options(self):
        """
        Return a dictionary with the model's default options.

        Returns
        -------
        out : dict
            Dictionary with model default options.
        
        See Also
        --------
        get_current_options, list_fields, get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])
        >>> default_options = model.get_default_options()
        """
        _mt._get_metric_tracker().track('toolkit.regression.logistic_regression.get_default_options')
        if self.get('solver') == 'vw':
            return {'max_iterations': 10, 'step_size': 1.0}
        else:
            return super(LogisticRegressionModel, self).get_default_options()

    def get_current_options(self):
        """
        Return a dictionary with the options used to define and train the model.

        Returns
        -------
        out : dict
            Dictionary with options used to define and train the model.

        See Also
        --------
        get_default_options, list_fields, get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])
        >>> current_options = model.get_current_options()
        """
        if self.get('solver') == 'vw':
            max_iterations = _vw.VWModel(self.__proxy__).get('max_iterations')
            step_size = _vw.VWModel(self.__proxy__).get('step_size')
            return {'max_iterations': max_iterations, 'step_size': step_size}
        else:
            return super(LogisticRegressionModel, self).get_current_options()

    def predict(self, dataset, output_type='probability',
        missing_value_action='impute'):
        """
        Return predictions for ``dataset``, using the trained logistic
        regression model. Predictions can be generated as class labels (0 or 1),
        probabilities that the target value is 1 (rather than 0), or margins
        (i.e. the distance of the observations from the hyperplane separating
        the classes). By default, the predict method returns probabilities.

        For each new example in ``dataset``, the margin---also known as the
        linear predictor---is the inner product of the example and the model
        coefficients. The probability is obtained by passing the margin through
        the logistic function. Predicted classes are obtained by thresholding
        the predicted probabilities at 0.5. If you would like to threshold
        predictions at a different probability level, you can use the 
        Graphlab evaluation toolkit.

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the features used for model training, but does not require
            a target column. Additional columns are ignored.

        output_type : {'probability', 'margin', 'class'}, optional
            Form of the predictions.
        
        missing_value_action: str, optional
            Action to perform when missing values are encountered. This can be 
            one of:

            - 'impute': Proceed with evaluation by filling in the missing
                        values with the mean of the training data. Missing
                        values are also imputed if an entire column of data is
                        missing during evaluation.
            - 'error' : Do not proceed with prediction and terminate with 
                        an error message.

        Returns
        -------
        out : SArray
            An SArray with model predictions.

        See Also
        ----------
        create, evaluate

        Examples
        ----------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])

        >>> probability_predictions = model.predict(data)
        >>> margin_predictions = model.predict(data, output_type='margin')
        >>> class_predictions = model.predict(data, output_type='class')

        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.predict')

        if output_type not in ['margin', 'probability', 'class']:
            raise ValueError("Output type '{}' is not supported.".format(output_type) + \
                             " Please select 'probability', 'margin', or 'class'.")

        if _HAS_PANDAS and type(dataset) == _pandas.DataFrame:
            dataset = _SFrame(dataset)

        # special handling of vw model
        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            target_column = self.get('target')
            dataset_copy = dataset[dataset.column_names()]  # SFrame copy by reference

            if target_column in dataset_copy.column_names():
                dataset_copy.remove_column(target_column)

            predictions = m.predict(dataset_copy)

            # VW predictions are raw "margins" (i.e., w^T * x + b)
            if output_type == "margin":
                return predictions

            else:
                # We need to transform it back into the [0,1] range through
                # the logistic link function.
                predictions = predictions.apply(lambda x : 1 - 1/(1+math.exp(x)))

                if output_type == "probability":
                    return predictions
                else:
                    predictions = predictions.apply(lambda x: x >= 0.5)
                    return predictions

        else:
            # Get metadata from C++ object
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__,
                    'dataset': dataset,
                    'missing_value_action': missing_value_action,
                    'output_type': output_type}

            ## Compute the predictions
            init_opts = _graphlab.toolkits.main.run('supervised_learning_predict_init', opts)
            opts.update(init_opts)
            target = _graphlab.toolkits.main.run('supervised_learning_predict', opts)
            return _SArray(None, _proxy=target['predicted'])

    def evaluate(self, dataset, missing_value_action = 'impute'):
        """
        Evaluate the model by making predictions of target values and comparing
        these to actual values.

        Two metrics are used to evaluate logistic regression. The confusion
        table contains the cross-tabulation of actual and predicted classes for
        the target variable. Classification accuracy is the fraction of examples
        whose predicted and actual classes match.

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the target and features used for model training. Additional
            columns are ignored.

        missing_value_action: str, optional
            Action to perform when missing values are encountered. This can be 
            one of:

            - 'impute': Proceed with evaluation by filling in the missing
                        values with the mean of the training data. Missing
                        values are also imputed if an entire column of data is
                        missing during evaluation.
            - 'error' : Do not proceed with evaluation and terminate with 
                        an error message.

        Returns
        -------
        out : dict
            Dictionary of evaluation results. The dictionary keys are *accuracy*
            and *confusion_table*.

        See Also
        ----------
        create, predict

        Examples
        ----------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])

        >>> results = model.evaluate(data)
        >>> print results['accuracy']

        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.evaluate')

        target_column = self.get('target')
        if target_column not in dataset.column_names():
            raise ToolkitError, \
                "Input dataset must contain a target column for " \
                "evaluation of prediction quality."

        target_type = dataset[target_column].dtype()
        if not target_type == int:
            raise TypeError("Target column must be type int")

        if self.get('solver') == 'vw':
            m = _vw.VWModel(self.__proxy__)
            dataset_copy = dataset[dataset.column_names()]  # SFrame copy by reference

            # If the dataset contains a column with the same heading as the
            # target column seen during training, then VW dictates that
            # this column must contain only +1/-1's.  Otherwise VW throws
            # an error.  To be consistent with the rest of the logistic
            # regression API, we'll handle the case where the target column
            # contains 0s or 1s. If not, we leave it as is, and allow VW to
            # fail.
            if set(dataset_copy[target_column]) == set([0, 1]):
                dataset_copy[target_column] = dataset_copy[target_column] * 2 - 1

            evaluation_results = m.evaluate(dataset_copy)
            return evaluation_results
        else:
            opts = {'model': self.__proxy__,
                    'dataset': dataset,
                    'missing_value_action': missing_value_action,
                    'model_name': self.__name__}

            response = _graphlab.toolkits.main.run('supervised_learning_evaluate_init', opts)
            opts.update(response)
            response = _graphlab.toolkits.main.run("supervised_learning_evaluate", opts)
            opts.update(response)
            results =  _graphlab.toolkits.main.run("supervised_learning_get_evaluate_stats",
                    opts)
            keys = ['true_positive', 'true_negative', 'false_positive',
                    'false_negative']
            confusion_matrix = {k: results[k] for k in keys}
            accuracy = results['accuracy']
            return {'accuracy': accuracy,
                    'confusion_table': confusion_matrix}


    def list_fields(self):
        """
        List the fields stored in the model, including data, model, and training
        options. Each field can be queried with the ``get`` method. Note: the
        list of queryable fields is different when the solver is ``vw``.

        Returns
        -------
        out : list
            List of fields queryable with the ``get`` method.
        
        See Also
        --------
        get

        Examples
        --------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.logistic_regression.create(data,
                                             target='is_expensive',
                                             features=['bath', 'bedroom', 'size'])

        >>> model.list_fields()
        """

        _mt._get_metric_tracker().track(
            'toolkit.regression.logistic_regression.list_fields')

        solver = self.get('solver')

        if solver == 'vw':
            return ['train_iters',
                    'l1_penalty',
                    'l2_penalty',
                    'max_iterations',
                    'solver',
                    'step_size',
                    'target',
                    'train_time']
        else:
            opts = {'model': self.__proxy__,
                    'model_name': self.__name__}
            response = _graphlab.toolkits.main.run('supervised_learning_list_keys', opts)
            return sorted(response.keys())

