"""
Methods for creating a topic model and predicting the topics of new documents.
"""

import graphlab as _graphlab
import graphlab.connect as _mt
from graphlab.toolkits.model import Model as _Model
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.data_structures.sarray import SArray as _SArray
from util import _check_input
from util import perplexity
from itertools import izip

def create(dataset,
           num_topics=10,
           initial_topics=None,
           alpha=None, beta=.1,
           num_iterations=10,
           associations=None,
           verbose=False,
           print_interval=10,
           method='auto'):
    """
    Create a topic model from the given data set. A topic model assumes each
    document is a mixture of a set of topics, where for each topic some words
    are more likely than others. One statistical approach to do this is called a
    "topic model". This method learns a topic model for the given document
    collection.

    Parameters
    ----------
    dataset : SArray of type dict or SFrame with a single column of type dict
        A bag of words representation of a document corpus.
        Each element is a dictionary representing a single document, where
        the keys are words and the values are the number of times that word
        occurs in that document.

    num_topics : int, optional
        The number of topics to learn.

    initial_topics : SFrame, optional
        An SFrame with a column of unique words representing the vocabulary
        and a column of dense vectors representing
        probability of that word given each topic. When provided,
        these values are used to initialize the algorithm.

    num_iterations : int, optional
        The number of iterations to perform.

    alpha : float, optional
        Hyperparameter that controls the diversity of topics in a document.
        Smaller values encourage fewer topics per document.
        Provided value must be positive. Default value is 50/num_topics.

    beta : float, optional
        Hyperparameter that controls the diversity of words in a topic.
        Smaller values encourage fewer words per topic. Provided value
        must be positive.

    verbose : bool, optional
        When True, print most probable words for each topic while printing
        progress. 

    print_interval : int, optional
        The number of iterations to wait between progress reports.

    associations : SFrame, optional
        An SFrame with two columns named "word" and "topic" containing words
        and the topic id that the word should be associated with. These words
        are not considered during learning.

    method : {'cgs'}, optional
        The algorithm used for learning the model. Currently only 'cgs' is
        available.

        - *cgs:* Collapsed Gibbs sampling

    Returns
    -------
    out : TopicModel
        A fitted topic model. This can be used with
        :py:func:`~TopicModel.get_topics()` and
        :py:func:`~TopicModel.predict()`. While fitting is in progress, several
        metrics are shown, including:

        +------------------+---------------------------------------------------+
        |      Field       | Description                                       |
        +==================+===================================================+
        | tokens/second    | The number of unique words processed per second   |
        +------------------+---------------------------------------------------+
        | seconds/iter     | The number of seconds elapsed per iteration       |
        +------------------+---------------------------------------------------+
        | perplexity       | An estimate of the model's ability to model the   |
        |                  | training data. See the documentation on evaluate. |
        +------------------+---------------------------------------------------+
 
    See Also
    --------
    TopicModel, TopicModel.get_topics, TopicModel.predict, 
    graphlab.SArray.dict_trim_by_keys

    References
    ----------
    - `Wikipedia - Latent Dirichlet allocation
      <http://en.wikipedia.org/wiki/Latent_Dirichlet_allocation>`_

    Examples
    --------
    The following example includes an SArray of documents, where
    each element represents a document in "bag of words" representation
    -- a dictionary with word keys and whose values are the number of times
    that word occurred in the document:

    >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nytimes')

    Once in this form, it is straightforward to learn a topic model.

    >>> from graphlab.toolkits.text import topic_model
    >>> m = topic_model.create(docs)

    It is also easy to create a new topic model from an old one  -- whether
    it was created using GraphLab Create or another package.

    >>> m2 = topic_model.create(docs,
                                initial_topics=m['topics'])

    To manually fix several words to always be assigned to a topic, use
    the `associations` argument. The following will ensure that topic 0
    has the most probability for each of the provided words:

    >>> from graphlab import SFrame
    >>> associations = SFrame({'word':['hurricane', 'wind', 'storm'],
                               'topic': [0, 0, 0]})
    >>> m = topic_model.create(docs,
                               associations=associations)

    More advanced usage allows you  to control aspects of the model and the
    learning method.

    >>> m = topic_model.create(docs,
                               num_topics=20,       # number of topics
                               num_iterations=10,   # algorithm parameters
                               alpha=.01, beta=.1)  # hyperparameters
    """
    _mt._get_metric_tracker().track('toolkit.text.topic_model.create')

    dataset = _check_input(dataset)

    

    # If associations are provided, check they are in the proper format
    if associations is None:
        associations = _graphlab.SFrame({'word': [], 'topic': []})
    if isinstance(associations, _graphlab.SFrame) and \
       associations.num_rows() > 0:
        assert set(associations.column_names()) == set(['word', 'topic']), \
            "Provided associations must be an SFrame containing a word column\
             and a topic column."
        assert associations['word'].dtype() == str, \
            "Words must be strings."
        assert associations['topic'].dtype() == int, \
            "Topic ids must be of int type."
    if alpha is None:
        alpha = float(50) / num_topics

    opts = {'data': dataset,
            'verbose': verbose,
            'num_topics': num_topics,
            'num_iterations': num_iterations,
            'print_interval': print_interval,
            'alpha': alpha,
            'beta': beta,
            'associations': associations}

    # Initialize the model with basic parameters
    response = _graphlab.toolkits.main.run("text_topicmodel_init", opts)
    m = TopicModel(response['model'])

    # If initial_topics provided, load it into the model
    if isinstance(initial_topics, _graphlab.SFrame):
        assert set(['vocabulary', 'topic_probabilities']) ==              \
               set(initial_topics.column_names()),                        \
            "The provided initial_topics does not have the proper format, \
             e.g. wrong column names."
        observed_topics = initial_topics['topic_probabilities'].apply(lambda x: len(x))
        assert all(observed_topics == num_topics),                        \
            "Provided num_topics value does not match the number of provided initial_topics."

        # Rough estimate of total number of words
        weight = dataset.size() * 1000

        opts = {'model': m.__proxy__,
                'topics': initial_topics['topic_probabilities'],
                'vocabulary': initial_topics['vocabulary'],
                'weight': weight}
        response = _graphlab.toolkits.main.run("text_topicmodel_set_topics", opts)
        m = TopicModel(response['model'])

    # Train the model on the given data set and retrieve predictions
    opts = {'model': m.__proxy__,
            'data': dataset,
            'method': method,
            'verbose': verbose}
    # opts.update(method_options)
    response = _graphlab.toolkits.main.run("text_topicmodel_train", opts)
    m = TopicModel(response['model'])

    return m


class TopicModel(_Model):
    """
    TopicModel objects can be used to predict the underlying topic of a
    document. Use the `create` method in this module to construct a TopicModel
    instance.
    """

    def __init__(self, model_proxy):
        self.__proxy__ = model_proxy

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return TopicModel(model_proxy)
        return model_wrapper

    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model when the model name is entered
        in the terminal.
        """

        s = "Topic Model\n"
        s += "  Data:\n"
        s += "      Vocabulary size:     %i\n" % len(self['vocabulary'])
        s += "  Settings:\n"
        s += "      Number of topics:    {0}\n".format(self['num_topics'])
        s += "      alpha:               {0}\n".format(self['alpha'], 3)
        s += "      beta:                {0}\n".format(self['beta'], 3)
        s += "      Iterations:          {0}\n".format(self['num_iterations'])
        s += "      Verbose:             {0}\n".format(self['verbose'] == 1)
        s += "  Accessible fields:\n"
        s += "      m['topics']          An SFrame containing the topics.\n"
        s += "      m['vocabulary']      An SArray containing the topics.\n"
        s += "  Useful methods:\n"
        s += "      m.get_topics()       Get the most probable words per topic.\n"
        s += "      m.predict(new_docs)  Make predictions for new documents.\n"

        return s

    def summary(self):
        """
        Display a summary of the TopicModel.

        Examples
        --------

        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> m.summary()
        Topic Model
            Data:
                Vocabulary size:     12375
            Settings:
                Number of topics:    10
                alpha:               5.0
                beta:                0.1
                Iterations:          10
                Verbose:             False
            Accessible attributes:
                m['topics']          An SFrame containing the topics.
                m['vocabulary']      An SArray containing the topics.
            Useful methods:
                m.get_topics()       Get the most probable words per topic.
                m.predict(new_docs)  Make predictions for new documents.

        """

        _mt._get_metric_tracker().track('toolkit.text.topic_model.summary')
        print self.__repr__()

    def list_fields(self):
        """
        List the fields stored in the model, including data, model, and training
        options. Each field can be queried with the :py:func:`~TopicModel.get`
        method.

        Returns
        -------
        out : list
            List of fields queryable with the :py:func:`~TopicModel.get` method.

        See Also
        --------
        get

        Examples
        --------

        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> m.list_fields()
        ['training_complete',
        'verbose',
        'num_topics',
        'vocabulary',
        'num_iterations',
        'topics',
        'beta',
        'print_interval',
        'perplexity',
        'alpha',
        'minibatch_size',
        'num_burnin']

        """

        _mt._get_metric_tracker().track('toolkit.text.topic_model.list_fields')
        opts = {'model': self.__proxy__}
        response = _graphlab.toolkits.main.run("text_topicmodel_list_fields", opts)
        return response.keys()

    def get(self, field):
        """
        Return the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained with the
        :py:func:`~TopicModel.list_fields` method.

        +-----------------------+----------------------------------------------+
        |      Field            | Description                                  |
        +=======================+==============================================+
        | topics                | An SFrame containing a column with the unique|
        |                       | words observed during training, and a column |
        |                       | of arrays containing the probability values  |
        |                       | for each word given each of the topics.      |
        +-----------------------+----------------------------------------------+
        | vocabulary            | An SArray containing the words used. This is |
        |                       | same as the vocabulary column in the topics  |
        |                       | field above.                                 |
        +-----------------------+----------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out
            Value of the requested field.

        See Also
        --------
        list_fields

        Examples
        --------
        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> m.get('topics')
        +--------------------------------+------------+
        |      topic_probabilities       | vocabulary |
        +--------------------------------+------------+
        | array('d', [0.000514752462 ... |  limited   |
        | array('d', [6.120718939647 ... |  consider  |
        | array('d', [0.000337251613 ... | represent  |
        | array('d', [0.000104664293 ... |    lack    |
        | array('d', [6.120718939647 ... | desirable  |
        | array('d', [6.120718939647 ... |   focus    |
        | array('d', [6.120718939647 ... | generaliza |
        | array('d', [6.120718939647 ... | generalize |
        | array('d', [6.120718939647 ... |    row     |
        | array('d', [6.120718939647 ... |   depend   |
        |              ...               |    ...     |
        +--------------------------------+------------+

        You may also do m['topics'].
        """

        _mt._get_metric_tracker().track('toolkit.text.topic_model.get')
        opts = {'model': self.__proxy__, 'field': field}
        response = _graphlab.toolkits.main.run("text_topicmodel_get_value", opts)
        if field == 'vocabulary':
            return _SArray(None, _proxy=response['value'])
        elif field == 'topics':
            return _SFrame(None, _proxy=response['value'])
        return response['value']

    def get_default_options(self):
        """
        Return a dictionary with the default options for the TopicModel.

        Returns
        -------
        out : dict
            Dictionary with the default options.

        See Also
        --------
        get_current_options, list_fields, get

        Examples
        --------

        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> m.get_default_options()
        {'alpha': 5.0,
        'beta': 0.1,
        'minibatch_size': 500,
        'num_burnin': 5,
        'num_iterations': 10,
        'num_topics': 10,
        'print_interval': 10,
        'verbose': 1}

        """

        _mt._get_metric_tracker().track(
            'toolkit.topic_model.get_default_options')

        opts = {'model': self.__proxy__}

        return _graphlab.toolkits.main.run(
            'text_topicmodel_get_default_options', opts)

    def get_current_options(self):
        """
        Return a dictionary with the options used to define and create the
        current TopicModel instance.

        Returns
        -------
        out : dict
            Dictionary of options used to train the current instance of the
            TopicModel.

        See Also
        --------
        get_default_options, list_fields, get

        Examples
        --------
        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> m.get_current_options()
        {'alpha': 5.0,
        'beta': 0.1,
        'minibatch_size': 500,
        'num_burnin': 5,
        'num_iterations': 10,
        'num_topics': 10,
        'print_interval': 10,
        'verbose': 1}

        """

        _mt._get_metric_tracker().track(
            'toolkit.topic_model.get_current_options')

        opts = {'model': self.__proxy__}

        return _graphlab.toolkits.main.run(
            'text_topicmodel_get_current_options', opts)

    def get_topics(self, topic_ids=None, num_words=5, cdf_cutoff=1.0):
        """
        Get the words associated with a given topic. The score column is the
        probability of choosing that word given that you have chosen a
        particular topic.

        Parameters
        ----------
        topic_ids : list of int, optional
            The topics to retrieve words. Topic ids are zero-based.
            Throws an error if greater than or equal to m['num_topics'], or
            if the requested topic name is not present.

        num_words : int, optional
            The number of words to show.

        cdf_cutoff : float, optional
            Allows one to only show the most probable words whose cumulative
            probability is below this cutoff. For example if there exist
            three words where
               p(word_1 | topic_k) = .1
               p(word_2 | topic_k) = .2
               p(word_3 | topic_k) = .05
            then setting cdf_cutoff=.3 would return only word_1 and word_2
            since p(word_1 | topic_k) + p(word_2 | topic_k) <= cdf_cutff.

        Returns
        -------
        out : SFrame
            An SFrame with a column of words ranked by a column of scores for
            each topic.

        Examples
        --------
        Get the highest ranked words for all topics.

        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs,
                                            num_iterations=50)
        >>> m.get_topics()
        +-------+----------+-----------------+
        | topic |   word   |      score      |
        +-------+----------+-----------------+
        |   0   |   cell   |  0.028974400831 |
        |   0   |  input   | 0.0259470208503 |
        |   0   |  image   | 0.0215721599763 |
        |   0   |  visual  | 0.0173635081992 |
        |   0   |  object  | 0.0172447874156 |
        |   1   | function | 0.0482834508265 |
        |   1   |  input   | 0.0456270024091 |
        |   1   |  point   | 0.0302662839454 |
        |   1   |  result  | 0.0239474934631 |
        |   1   | problem  | 0.0231750116011 |
        |  ...  |   ...    |       ...       |
        +-------+----------+-----------------+

        Get the highest ranked words for topics 0 and 1. Show 15 words.

        >>> m.get_topics([0, 1], num_words=15)
        +-------+----------+------------------+
        | topic |   word   |      score       |
        +-------+----------+------------------+
        |   0   |   cell   |  0.028974400831  |
        |   0   |  input   | 0.0259470208503  |
        |   0   |  image   | 0.0215721599763  |
        |   0   |  visual  | 0.0173635081992  |
        |   0   |  object  | 0.0172447874156  |
        |   0   | response | 0.0139740298286  |
        |   0   |  layer   | 0.0122585145062  |
        |   0   | features | 0.0115343177265  |
        |   0   | feature  | 0.0103530459301  |
        |   0   | spatial  | 0.00823387994361 |
        |  ...  |   ...    |       ...        |
        +-------+----------+------------------+

        If one wants to rearrange this into dictionaries, one can use
        :py:func:`~graphlab.SFrame.unstack`:

        >>> topics = m.get_topics()
        >>> topics.unstack(['word','score'], 'word')['word'].dict_keys()
        dtype: list
        Rows: 10
        [['cell', 'image', 'input', 'object', 'visual'],
         ['algorithm', 'data', 'learning', 'method', 'set'],
         ['function', 'input', 'point', 'problem', 'result'],
         ['model', 'output', 'pattern', 'set', 'unit'],
         ['action', 'learning', 'net', 'problem', 'system'],
         ['error', 'function', 'network', 'parameter', 'weight'],
         ['information', 'level', 'neural', 'threshold', 'weight'],
         ['control', 'field', 'model', 'network', 'neuron'],
         ['hidden', 'layer', 'system', 'training', 'vector'],
         ['component', 'distribution', 'local', 'model', 'optimal']]
        """
        _mt._get_metric_tracker().track('toolkit.text.topic_model.get_topics')


        if topic_ids is None:
            topic_ids = range(self.get('num_topics'))

        assert isinstance(topic_ids, list), \
            "The provided topic_ids is not a list."

        if any([type(x) == str for x in topic_ids]):
            raise ValueError, \
                "Only integer topic_ids can be used at this point in time."
        if not all([x >= 0 and x < self['num_topics']]):
            raise ValueError, \
                "Topic id values must be non-negative and less than the " + \
                "number of topics used to fit the model."

        opts = {'model': self.__proxy__,
                'topic_ids': topic_ids,
                'num_words': num_words,
                'cdf_cutoff': cdf_cutoff}
        response = _graphlab.toolkits.main.run('text_topicmodel_get_topic',
                                               opts)
        sf =_SFrame(None, _proxy=response['top_words'])
        return sf

    def predict(self, dataset, output_type='assignment'):
        """
        Use the model to predict topics for each document. The provided
        `dataset` should be an SArray object where each element is a dict
        representing a single document in bag-of-words format, where keys
        are words and values are their corresponding counts. If `dataset` is
        an SFrame, then it must contain a single column of dict type.

        The current implementation will make inferences about each document
        given its estimates of the topics learned when creating the model.
        This is done via Gibbs sampling.

        Parameters
        ----------
        dataset : SArray, SFrame of type dict
            A set of documents to use for making predictions.

        output_type : str, optional
            The type of output desired. This can either be

            - assignment: the returned values are integers in [0, num_topics)
            - probability: each returned prediction is a vector with length
              num_topics, where element k prepresents the probability that
              document belongs to topic k.

        Returns
        -------
        out : SArray

        See Also
        --------
        evaluate

        Examples
        --------
        Make predictions about which topic each document belongs to.

        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> m = graphlab.topic_model.create(docs)
        >>> pred = m.predict(docs)

        If one is interested in the probability of each topic

        >>> pred = m.predict(docs, output_type='probability')

        Notes
        -----
        For each unique word w in a document d, we sample an assignment to
        topic k with probability proportional to

        .. math::
            p(z_{dw} = k) \propto (n_{d,k} + \\alpha) * \Phi_{w,k}

        where

        - :math:`W` is the size of the vocabulary,
        - :math:`n_{d,k}` is the number of other times we have assigned a word in
          document to d to topic :math:`k`,
        - :math:`\Phi_{w,k}` is the probability under the model of choosing word
          :math:`w` given the word is of topic :math:`k`. This is the matrix
          returned by calling `m['topics']`.

        This represents a collapsed Gibbs sampler for the document assignments
        while we keep the topics learned during training fixed.
        This process is done in parallel across all documents, five times per
        document.

        """
        _mt._get_metric_tracker().track('toolkit.text.topic_model.predict')

        dataset = _check_input(dataset)

        opts = {'model': self.__proxy__,
                'data': dataset}
        response = _graphlab.toolkits.main.run("text_topicmodel_predict", opts)
        preds = _SArray(None, _proxy=response['predictions'])

        # Get most likely topic if probabilities are not requested
        if output_type not in ['probability', 'probabilities', 'prob']:
            # equivalent to numpy.argmax(x)
            preds = preds.apply(lambda x: max(izip(x, xrange(len(x))))[1])

        return preds


    def evaluate(self, train_data, test_data=None):
        """
        Estimate the model's ability to predict new data. Imagine you have a
        corpus of books. One common approach to evaluating topic models is to
        train on the first half of all of the books and see how well the model
        predicts the second half of each book.

        This method returns a metric called perplexity, which  is related to the
        likelihood of observing these words under the given model. See
        :py:func:`~graphlab.text.util.perplexity` for more details.

        The provided `train_data` and `test_data` must have the same length,
        i.e., both data sets must have the same number of documents; the model
        will use train_data to estimate which topic the document belongs to, and
        this is used to estimate the model's performance at predicting the
        unseen words in the test data.

        See :py:func:`~graphlab.text.topic_model.TopicModel.predict` for details
        on how these predictions are made, and see
        :py:func:`~graphlab.text.util.random_split` for a helper function
        that can be used for making train/test splits.

        Parameters
        ----------
        train_data : SArray or SFrame
            A set of documents to predict topics for.

        test_data : SArray or SFrame, optional
            A set of documents to evaluate performance on.
            By default this will set to be the same as train_data.

        Returns
        -------
        out : float
            An estimated perplexity value.

        See Also
        --------
        predict, graphlab.text.util.random_split

        Examples
        --------
        >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
        >>> train_data, test_data = graphlab.text.util.random_split(docs)
        >>> m = graphlab.topic_model.create(train_data)
        >>> m.evaluate(train_data, test_data)
        2467.530370396021

        """
        _mt._get_metric_tracker().track('toolkit.text.topic_model.evaluate')

        train_data = _check_input(train_data)

        if test_data is None:
            test_data = train_data
        else:
            test_data = _check_input(test_data)

        predictions = self.predict(train_data, output_type='probability')
        topics = self.get('topics')

        perp = perplexity(test_data,
                          predictions,
                          topics['topic_probabilities'],
                          topics['vocabulary'])
        return perp
