"""
This module defines PredictiveServiceClient that consumes service provided
by GraphLab Predictive Service.

To use the client, use one of the following mechanism:

a. Pass appropriate parameters to the constructor

    >>> client = PredictiveServiceClient(endpoint=endpoint, api_key=api_key, ...)
    >>> result = client.query('model_name', 'method_name', query_data)
    >>> result = client.feedback('some key', feedback_data)

b. Create a configuration file and pass it to PredictiveServiceClient

    >>> client = PredictiveServiceClient(config_file = path_to_file)
    >>> result = client.query('model_name', 'method_name', query_data)
    >>> result = client.feedback('some key', feedback_data)


The configuration file is expected to be in a format that is similar to Microsoft
Windows INI file and can be consumed by python package ConfigParser, it consists
multipe sections and a list of key/value pairs inside each section, this is a
sample file:

[Service Info]
endpoint = http://service-dns-name
api key = api-key-string
"""


import os
import requests
import json
import logging
import urllib
from ConfigParser import ConfigParser

def _setup_logger():
  logging.basicConfig(level=logging.DEBUG)
  logger = logging.getLogger(__name__)
  return logger

__logger__ = _setup_logger()

class PredictiveServiceClient(object):
    def __init__(self, endpoint = None, api_key = None, should_verify_certificate = None, config_file = None):
        '''Constructs a new PredictiveServiceClient
        '''
        if config_file != None:
            self._read_config(config_file)
        else:
            self.endpoint = endpoint
            self.api_key = api_key
            self.should_verify_certificate = should_verify_certificate

        self._ping()

    def __str__(self):
        '''String representation of the PredictiveServiceClient'''
        s = ""
        s += 'Predictive Service Client:\n'
        s += "\tendpoint: %s\n" % self.endpoint

        s += 'List of models:\n'
        models = self._models
        if len(models) == 0:
            s += 'No models deployed\n'
        else:
            for (uri, version) in models.iteritems():
                s += '\tname: %s, version:%s\n' %(uri, version)
        return s

    def __repr__(self):
        '''String representation of the PredictiveServiceClient'''
        return self.__str__()

    def query(self, uri, data):
        '''Query a Predictive Service object

        Parameters
        ----------
        uri : str
            The model uri, must have been deployed in server side

        data : dict
            The data that is passed to the query method of the predictive object,
            must be in he format that is accepted by the object.

        Examples
        --------

            >>> client = PredictiveServiceClient(config_file='some file')
            >>> data = {'method':'predict', 'dataset':{'user_id':175343, 'movie_id':1011}}
            >>> client.query('recommender', data)

            >>> data = {'method':'predict', 'dataset':[
                {'user_id':175343, 'movie_id':1011},
                {'user_id':175344, 'movie_id':1012}
                ]}
            >>> client.query('recommender', data)

        Returns
        -------
        out : dict
            Returns the query result in dictionary format

        '''
        if type(uri) is not str:
            raise TypeError("'uri' has to be a string")

        if type(data) is not dict:
            raise TypeError("'data' has to be a dictionary")

        # convert to valid url
        uri = urllib.quote(uri)

        internal_data = {'api key':self.api_key, 'data':data}
        (success, response) = self._post('data/%s' % uri, internal_data)
        if success:
            __logger__.debug(response)

            # todo: may want to extract query_id here from response and return
            # to user as (query_id, response_data)
            if response['type'] == 'QuerySuccessful':
                return response['response']
            elif response['type'] == 'QueryFailed':
                raise RuntimeError('Query error: %s' % response['error'])
            elif response['type'] == 'UnknownURI':
                raise RuntimeError('Model "%s" is unknown.' % uri)
            else:
                raise RuntimeError('Query error: %s' % response)
        else:
            raise RuntimeError("Error encountered: %s" % response)

    def feedback(self, key, data):
        '''Provide feedback to the query result. This is a free format feedback.

        Parameters
        ----------
        key : str

        data : dict

        Examples
        --------

            >>> client = PredictiveServiceClient(config_file='some file')
            >>> client.feedback('user123', data)

        Returns
        -------
        out : dict
            Returns the server response.


        '''
        if type(key) != str:
            raise RuntimeError("Expect key to be a string type")
        if type(data) != dict:
            raise RuntimeError("Feedback 'data' needs to be dictionary type")

        data = {'api key': self.api_key, 'data': data, 'id': key}
        response = self._post('feedback', data)
        return response

    def _ping(self):
        try:
            __logger__.info("Connecting to Predictive Service at %s" % self.endpoint)
            response = requests.get(self.endpoint, verify=self.should_verify_certificate)
            if response.status_code == 200:
                __logger__.info("Successfully connected to %s" % (self.endpoint))
            else:
                raise RuntimeError("Error responding from service: response: %s" % str(response.text))
        except RuntimeError as e:
            raise e
        except Exception as e:
             raise RuntimeError("Cannot connect to end point, exception: %s" % e.message)

    def _post(self, path, data = None):
        try:
            headers = {'content-type': 'application/json' }
            url = os.path.join(self.endpoint, path)

            if data != None:
                data = json.dumps(data)

            response = requests.post(url = url, data=data, headers=headers, verify=self.should_verify_certificate, timeout=10)
            if response.status_code < 299 and response.status_code >= 200:
                return (True, response.json())
            else:
                __logger__.info("Error querying endpoint %s" % response)
                return (False, response.text)
        except Exception as e:
            raise e

    def _read_config(self, config_file):
        config_file = os.path.abspath(os.path.expanduser(config_file))
        if not os.path.isfile(config_file):
            raise RuntimeError("Path '%s' is not pointing to a file %s" % config_file)

        config = ConfigParser()
        config.read(config_file)

        service_info_section_name = "Service Info"
        if service_info_section_name not in config.sections():
            raise RuntimeError("Cannot find %s section in config file %s" % (service_info_section_name, config_file))

        options = config.options(service_info_section_name)
        self.endpoint = config.get(service_info_section_name, 'endpoint')
        self.api_key = config.get(service_info_section_name, 'api key')
        if config.has_option(service_info_section_name, 'verify certificate'):
            self.should_verify_certificate = config.getboolean(service_info_section_name, 'verify certificate')
        else:
            self.should_verify_certificate = True
        __logger__.info("Read configuration, endpoint: %s" % self.endpoint)

