'''
Predictive Service exposed methods
'''
from logging import getLogger as _getLogger
import time as _time
from re import compile as _compile
from uuid import uuid4 as _random_uuid
import os as _os
import datetime as _datetime

from boto.ec2.elb import connect_to_region as _lb_connection

from graphlab.connect.aws._ec2 import get_credentials as _get_credentials
import graphlab as _gl
import graphlab.connect as _mt

# since _predictive_service_environment imports these, need to have them defined first
_MAX_CREATE_TIMEOUT_SECS = 600 # 10m

from _predictive_service._predictive_service_environment import Ec2PredictiveServiceEnvironment as _Ec2PredictiveServiceEnvironment
from _predictive_service._file_util import parse_s3_path as _parse_s3_path, s3_recursive_delete as _s3_recursive_delete, s3_delete_key as _s3_delete_key
from _predictive_service._predictive_service import PredictiveService as _PredictiveService

_logger = _getLogger(__name__)
_name_checker = _compile('^[a-zA-Z-]+$')

def create(name, environment, state_path, description = None, api_key = None, admin_key = None,
           ssl_credentials = None, num_hosts=3):
    '''
    Launch a Predictive Services cluster. This cluster can currently be launched
    on EC2 by specifying an EC2 environment.

    Parameters
    ----------
    name : str
        The name of the Predictive Service that will be launched.

        This string can only contain: a-z, A-Z and hyphens.

    environment : :class:`~graphlab.deploy.environment.EC2` or str
        Must be either an EC2 Environment object or the name of EC2 environment.
        This is the environment where the Predictive Service will be deployed.

    state_path :  str
        S3 path used to manage state for the Predictive Service. This path can
        also be used to create the Predictive Service object on another
        computer.

    description : str, optional
        Description of this Predictive Service.

    api_key : str, optional
        An API key that client must included with requests. If an api_key is
        not specified, one will be auto generated. The API Key can be retrieved
        from the return PredictiveService object.

    admin_key : str, optional
        An API key used for control operations (i.e. anything other than client
        requests). If an admin_key is not specified, one will be auto
        generated. The API Key can be retrieved from the return
        PredictiveService object.

    ssl_credentials : tuple of len three, with types: str, str, bool.
        The first string is the path to the private key file. The second string
        is the path to public key certificate. The third denotes whether the
        certificates are self signed (and any client should not verify the
        certificate).

        These files must be in the precise format AWS expects. Such a private
        key and a self-signed certificate can be generated using openssl with
        following commands:

        openssl genrsa 1024 > privatekey.pem
        openssl req -new -key privatekey.pem -out csr.pem
        openssl x509 -req -days 365 -in csr.pem -signkey privatekey.pem -out server.crt

        If a tuple is not given, requests will be served over HTTP rather than
        HTTPS (i.e.  encryption will not be used).

    num_hosts : int, optional
        Number of hosts to launch for this Predictive Service. Predictive
        Services require at least 3 hosts.

    Returns
    -------
    out : :py:class:`~graphlab.deploy._predictive_service._predictive_service.PredictiveService`
        A Predictive Service object which can be used to manage the deployment.

    Examples
    --------
        >>> ec2 = graphlab.deploy.environments['ec2']
        >>> ps = graphlab.deploy.create('name', ec2, 's3://mybucket/pred_services/name')
        >>> print ps
        >>> ps.get_status()

    See Also
    --------
    load
    '''

    # local test hook
    if (environment is None):
        # load locally
        ret = _gl.deploy.predictive_service.load(state_path, override=True)
        ret._environment = ret._environment.launch(predictive_service_path = ret._s3_state_path, log_path = ret._environment.log_path, num_hosts = 3, admin_key = ret._environment.admin_key)
        return ret

    # Validate inputs for current session
    if not _gl.deploy._default_session.is_name_unique(name):
        # found another predictive service or predictive service endpoint with the same name
        raise RuntimeError("Validation Error: Predictive Service already exists with the name '%s', please rename or delete the existing Predictive Service." % name)

    # Validate 'environment' value
    if not isinstance(environment, _gl.deploy.environment.EC2) and not isinstance(environment, str):
        raise TypeError('Unsupported type given for environment parameter. Must be either an EC2' \
                          ' environment or the name of an EC2 environment.')
    if isinstance(environment, str):
        # Get the EC2 config object
        environment_name = environment
        environment = _gl.deploy.environments[environment_name]
        if not environment:
            raise TypeError("No environment named '%s'." % environment_name)
        if not isinstance(environment, _gl.deploy.environment.EC2):
            raise TypeError("%s is not an EC2 environment." % environment_name)

    # Save AWS config
    if(hasattr(environment, 'aws_access_key') and hasattr(environment, 'aws_secret_key')):
        aws_access_key = environment.aws_access_key
        aws_secret_key = environment.aws_secret_key
    else:
        try:
            aws_access_key, aws_secret_key = _get_credentials()
        except:
            raise IOError('No AWS credentials set. Credentials must either be set in the ' \
                              'environment parameter or set globally using ' \
                              'graphlab.aws.set_credentials(...).')
    aws_credentials = {
        'aws_access_key_id': aws_access_key,
        'aws_secret_access_key': aws_secret_key
        }

    # Validate 'name' value
    if not _name_checker.match(name):
        raise IOError('Names can only can only contain: a-z, A-Z and hyphens.')
    conn = _lb_connection(environment.region, **aws_credentials)
    for lb in conn.get_all_load_balancers():
        if lb.name == name:
            raise IOError('There is already a load balancer with that name. Load balancer names' \
                              ' must be unique in their region. Please choose a different name.')
    if not isinstance(num_hosts, int):
        raise TypeError('Invalid parameter for num_hosts, must be an integer')
    if num_hosts < 3:
        _logger.error("Predictive Services require a minimum of 3 hosts in a cluster. Please set num_hosts to 3 or more when launching.")
        return

    # clone the environment, so not mutating existing environment
    environment = environment.clone()
    environment.num_hosts = num_hosts

    tracker = _mt._get_metric_tracker()
    tracker.track('graphlab.deploy.predictive_service.create', value=1,
            properties={'num_hosts':num_hosts, 'instance_type':environment.instance_type})

    _logger.info("Launching Predictive Service with %d hosts, as specified by num_hosts parameter" % (num_hosts))

    # Add a tag for all EC2 instances to indicate they are related to this PredictiveService
    if not environment.tags:
        environment.tags = {}
    environment.tags.update({'Name':name, 'predictive_service':name})

    # Set defaults values, if needed
    if not api_key:
        api_key = str(_random_uuid())
    if not admin_key:
        admin_key = str(_random_uuid())

    result = None
    env = None
    try:
        starttime = _datetime.datetime.now()
        _logger.info("Launching Predictive Service, with name: %s" % name)

        _logger.info("[Step 0/5]: Initializing S3 locations.")
        # Create the predictive service object. It writes init state to S3.
        result = _PredictiveService(name, state_path, description, api_key, aws_credentials)

        # Launch the environment.
        env = _Ec2PredictiveServiceEnvironment.launch(name, environment, state_path, admin_key,
                                                     ssl_credentials, aws_credentials, started=starttime)

        # Attach the launched environment and write all service state to S3.
        result._environment = env
        result._save_state_to_s3()

        _logger.info("[Step 4/5]: Waiting for Load Balancer to put all instances into service.")
        while ((_datetime.datetime.now() - starttime).total_seconds() < _MAX_CREATE_TIMEOUT_SECS):
            # query status, verify all InService
            nodes = env.get_status(_show_errors = False)
            statuses = []
            for node in nodes:
                statuses.append(node['state'] == 'InService')
            if all(statuses):
                _logger.info("Cluster is fully operational, [%d/%d] instances currently in service." %
                        (statuses.count(True), len(statuses)))
                break
            else:
                _logger.info("Cluster not fully operational yet, [%d/%d] instances currently in service." %
                        (statuses.count(True), len(statuses)))
                _time.sleep(15)
        else:
            _logger.error("Instances failed to be ready within 10 minutes. Tearing down.")
            raise RuntimeError("Cluster configuration not successful in time, timing out.")

        _logger.info("[Step 5/5]: Finalizing Configuration.")

        _gl.deploy._default_session.register(result)
        result.save()

        return result

    except:
        # any exceptions we should gracefully terminate / tear down what we've created
        _logger.warning("Tearing down Predictive Service due to error launching")

        # default behavior deletes the log files in tear down.
        # To preserve the logs set GRAPHLAB_DEBUG in environment, and the logs will be preserved
        delete_logs = True
        if 'GRAPHLAB_DEBUG' in _os.environ:
            _logger.info("Preserving Log Files for debugging purposes")
            delete_logs = False

        if env:
            env.terminate(delete_logs)

        if result and delete_logs:
            _logger.info('Deleting model data.')
            try:
                _s3_recursive_delete(result._get_predictive_object_path_prefix(), aws_credentials)
                _s3_delete_key(result._s3_bucket_name, result._s3_state_key, aws_credentials)
            except:
                _logger.error("Could not delete model data. Please manually delete data under: %s" %
                              result._get_predictive_object_path_prefix())

        # re-raise exception so object not returned etc
        raise

def load(s3_path, aws_access_key_id = None, aws_secret_access_key = None, override = True):
    '''
    Loads a Predictive Service object from the s3_path.

    This will have the effect of instantiating an object locally, which contains
    all the relevant metadata associated with a Predictive Service. This is
    useful for getting a handle on a Predictive Service created from another
    machine, allowing you to query it and report its status, for example.

    Parameters
    -----------
    s3_path : str
        S3 path used to manage state for the Predictive Service. This corresponds
        to the state path specified when the Predictive Service was initially
        created.

    aws_access_key_id : str
        The AWS Access Key to use to load the Predictive Service. If this is not specified,
        your system environment variables will be used. You may also use
        :py:func:`~graphlab.aws.set_credentials` to set the credentials.

    aws_secret_access_key : str
        The AWS secret Key to use to load the Predictive Service. If this is not specified,
        your system environment variables will be used. You may also use
        :py:func:`~graphlab.aws.set_credentials` to set the credentials.

    override: boolean, optional
        If override is True, the created predictive service from s3_path would override any
        local predictive service endpoint that has same name. False otherwise.

    Returns
    -------
    out : :py:class:`~graphlab.deploy._predictive_service._predictive_service.PredictiveService`
        The Predictive Service handle that can be used to access and manage the
        deployed Predictive Service

    See Also
    --------
    create

    Examples
    --------

    >>> ps = graphlab.deploy.predictive_service.load('s3://mybucket/pred_services/name')
    >>> ps.get_status()

    '''
    # Save the credentials.
    if bool(aws_access_key_id) != bool(aws_secret_access_key):
        raise IOError('Either both aws_access_key_id and aws_secret_access_key should be' \
                          ' specified or neither should be specified.')
    if not aws_access_key_id and not aws_secret_access_key:
        try:
            aws_access_key, aws_secret_key = _get_credentials()
        except:
            raise IOError('No AWS credentials set. Credentials must either be passed in,' \
                              ' or set globally using graphlab.aws.set_credentials(...).')
    aws_credentials = {
            'aws_access_key_id': aws_access_key_id,
            'aws_secret_access_key': aws_secret_access_key
        }

    s3_bucket_name, s3_key_name =  _parse_s3_path(s3_path)
    config = _PredictiveService._get_s3_state_config(s3_bucket_name, s3_key_name + '/state.ini',
                                                    aws_credentials)

    name = config.get(_PredictiveService._SERVICE_INFO_SECTION_NAME, 'Name')
    description = config.get(_PredictiveService._SERVICE_INFO_SECTION_NAME, 'Description')
    api_key = config.get(_PredictiveService._SERVICE_INFO_SECTION_NAME, 'API Key')

    result = _PredictiveService(name, s3_path, description, api_key, aws_credentials,
                               _new_service = False)

    if not _gl.deploy._default_session.is_name_unique(name, typename="PredictiveService"):
        if override is True:
            # delete the object in the session before overriding
            _gl.deploy.predictive_services.delete(name)
        else:
            # found another predictive service or predictive service endpoint with the same name
            raise RuntimeError("Validation Error: Predictive Service already exists with the name '%s', please rename or delete the existing Predictive Service." % name)


    _mt._get_metric_tracker().track('predictive_service.load', value = 1, properties={'path': s3_path})

    # register and save to session
    _gl.deploy._default_session.register(result)
    result.save()

    return result
