import views as __views
import graphlab.connect as _mt
from graphlab.canvas import inspect as _inspect

"""
This package contains the implementation of the server component of GraphLab
Canvas, the visualization engine for GraphLab data structures.
"""

_active_target = None
def set_target(target):
    """
    Set the target for GraphLab Canvas view output. By default, view output is set to
    'browser'. To change output target, use graphlab.canvas.set_target(target). Specifying
    'ipynb' will attempt to render to an output cell in the IPython Notebook. If any other 
    value is specified, the target will not be changed.

    To set the output target to browser and use GraphLab Canvas as an interactive web application (default):

    >>> graphlab.canvas.set_target('browser')

    To set the output target to cells within the IPython Notebook:

    >>> graphlab.canvas.set_target('ipynb')

    To disable Canvas output:

    >>> graphlab.canvas.set_target('none')

    Parameters
    ----------
    target : "browser" | "ipynb"
    """
    import target as __target
    global _active_target

    # track metrics on target
    _mt._get_metric_tracker().track('canvas.set_target.%s' % target)

    if target == 'browser' and not isinstance(_active_target, __target.InteractiveTarget):
        _active_target = __target.InteractiveTarget()
    elif target == 'ipynb' and not isinstance(_active_target, __target.IPythonTarget):
        _active_target = __target.IPythonTarget()
    elif target == 'none' and not isinstance(_active_target, __target.NoneTarget):
        _active_target = __target.NoneTarget()

def get_target():
    """
    Get the active target for Canvas. If none has been set, this will set
    the default target ("browser") as the active target and return it.
    """
    import target as __target
    global _active_target
    if _active_target is None:
        set_target('browser')
    return _active_target

@_inspect.find_vars
def show(variable=None):
    """
    show()
    Re-launch and resume the prior GraphLab Canvas session in default browser.
    This method is useful if the GraphLab Create Python session is still active
    but the GraphLab Canvas browser session has ended.

    """
    if variable is not None:
        get_target().state.set_selected_variable(variable)
        variable.validate_js_component_name(variable.get_js_component())
    return get_target().show()

def _get_id(ref):
    import graphlab
    import graphlab.data_structures.gframe
    # use real content identity of underlying SArray to
    # determine whether two are the same.
    # will err on the side of correctness (it's possible two
    # objects that really are the same seems different according
    # to this function) due to lazy evaluation behavior with
    # __get_content_identifier__.
    # for other known types (SFrame and SGraph) use proxy object.
    if isinstance(ref, graphlab.SArray):
        # use content identity (goes all the way to real storage, so
        # two C++ objects w/ two corresponding Python objects could
        # still be the same.
        return ref.__get_content_identifier__()

    if type(ref) == graphlab.SFrame or \
       type(ref) == graphlab.SGraph:
         return long(hash(str(ref.__proxy__)))

    if type(ref) == graphlab.data_structures.gframe.GFrame:
        # hash a tuple of SGraph proxy and GFrame type
        return long(hash((str(ref.__graph__.__proxy__), ref.__type__)))

    if isinstance(ref, (graphlab.deploy._artifact.Task,
                        graphlab.deploy._artifact.Pipeline,
                        graphlab.deploy._artifact.RuntimeTask,
                        graphlab.deploy.environment.Environment,
                        graphlab.deploy._job.Job)):
        return long(hash(str(ref.name)))

    # for all other types, we don't really know how to compare
    # so just use the Python id
    return long(id(ref))

def _same_object(ref1, ref2):
    if type(ref1) != type(ref2):
        return False
    return _get_id(ref1) == _get_id(ref2)

def _profile(func):
    import functools
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        import cProfile, pstats, StringIO, tempfile
        import time
        pr = cProfile.Profile()
        pr.enable()
        time1 = time.time()
        try:
            return func(*args, **kwargs)
        finally:
            time2 = time.time()
            pr.disable()
            s = StringIO.StringIO()
            ps = pstats.Stats(pr, stream=s).sort_stats('cumulative')
            ps.print_stats()
            with tempfile.NamedTemporaryFile(delete=False) as f:
                f.write(s.getvalue())
                print 'function took %0.3f ms' % ((time2-time1)*1000.0)
                print 'log file at %s' % f.name
    return wrapper
