"""
This package contains the Gradient Boosted Trees model class and the create function.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits._model import _get_default_options_wrapper
from graphlab.toolkits._supervised_learning import SupervisedLearningModel as _SupervisedLearningModel
import graphlab.toolkits._supervised_learning as _sl
from graphlab.toolkits._internal_utils import _toolkit_repr_print
from graphlab.toolkits._internal_utils import _SGraphFromJsonTree
from graphlab.toolkits._internal_utils import _raise_error_evaluation_metric_is_valid
from graphlab.toolkits._internal_utils import _raise_error_if_column_exists


_BOOSTED_TREES_MODEL_PARAMS_KEYS = ['num_trees', 'step_size', 'max_depth',
'max_iterations', 'min_child_weight', 'min_loss_reduction', 'row_subsample']
_BOOSTED_TREE_TRAINING_PARAMS_KEYS = ['objective', 'training_time',
'training_error', 'validation_error', 'evaluation_metric']
_BOOSTED_TREE_TRAINING_DATA_PARAMS_KEYS = ['target', 'features',
'num_features', 'num_examples', 'num_validation_examples']


DEFAULT_HYPER_PARAMETER_RANGE = {
    'max_depth': [6, 8, 10],
    'step_size': 0.3,
    'min_loss_reduction': [0, 1, 10],
    'min_child_weight': 0.1,
    'row_subsample': 1,
    'column_subsample': 1,
    'max_iterations': [10, 50, 100]
}

get_default_options = _get_default_options_wrapper(
                          'boosted_trees_regression', 
                          'boosted_trees_regression', 
                          'BoostedTreesRegression')


class BoostedTreesRegression(_SupervisedLearningModel):
    """
    Encapsulates gradient boosted trees for regression tasks.

    The prediction is based on a collection of base learners, `regression trees
    <http://en.wikipedia.org/wiki/Decision_tree_learning>`_.


    Different from linear models, e.g. linear regression,
    the gradient boost trees model is able to model non-linear interactions
    between the features and the target using decision trees as the subroutine.
    It is good for handling numerical features and categorical features with
    tens of categories but is less suitable for highly sparse feautres such as
    text data.

    An instance of this model can be created using
    :py:func:`~graphlab.boosted_trees_regression.create`. 

    Do NOT construct the model directly.

    See Also
    --------
    create
    """
    def __init__(self, proxy):
        """__init__(self)"""
        self.__proxy__ = proxy
        self.__name__ = "boosted_trees_regression"
    
    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return BoostedTreesRegression(model_proxy)
        return model_wrapper


    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model, when the model name is entered
        in the terminal.
        """
        
        data_fields = [
            ('Number of examples', 'num_examples'),
            ('Number of feature columns', 'num_features'),
            ('Number of unpacked features', 'num_unpacked_features')]

        training_fields = [
            ("Number of trees", 'num_trees'),
            ("Max tree depth", 'max_depth'),
            ("Train RMSE", 'training_rmse'),
            ("Validation RMSE", 'validation_rmse'),
            ("Training time (sec)", 'training_time')]

        return _toolkit_repr_print(self, [data_fields, training_fields], width=30)

    def summary(self):
        """
        Display a summary of :class:`~graphlab.boosted_trees_regression.BoostedTreesRegression`.

        See Also
        ----------
        show

        Examples
        --------
        >>> m.summary()
        """
        _mt._get_metric_tracker().track('toolkit.regression.boosted_trees_regression.summary')
        print ""
        print "                    Model summary                       "
        print "--------------------------------------------------------"
        print self.__repr__()

    def get(self, field):
        """
        Get the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained programmatically using the
        :func:`~graphlab.boosted_trees_regression.list_fields` method.

        +-------------------------+--------------------------------------------------------------------------------+
        | Field                   | Description                                                                    |
        +=========================+================================================================================+
        | column_subsample        | Percentage of the columns for training each individual tree                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | features                | Names of the feature columns                                                   |
        +-------------------------+--------------------------------------------------------------------------------+
        | history                 | A list of string for the training history                                      |
        +-------------------------+--------------------------------------------------------------------------------+
        | max_depth               | The maximum depth of individual trees                                          |
        +-------------------------+--------------------------------------------------------------------------------+
        | max_iterations          | Number of iterations, equals to the number of trees                            |
        +-------------------------+--------------------------------------------------------------------------------+
        | min_child_weight        | Minimun weight required on the leave nodes                                     |
        +-------------------------+--------------------------------------------------------------------------------+
        | min_loss_reduction      | Minimun loss reduction required for splitting a node                           |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_features            | Number of features in the model                                                |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_unpacked_features   | Number of features in the model (including unpacked dict/list type columns)    |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_examples            | Number of training examples                                                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_validation_examples | Number of validation examples                                                  |
        +-------------------------+--------------------------------------------------------------------------------+
        | row_subsample           | Percentage of the rows for training each individual tree                       |
        +-------------------------+--------------------------------------------------------------------------------+
        | step_size               | Step_size used for combining the weight of individual trees                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | target                  | Name of the target column                                                      |
        +-------------------------+--------------------------------------------------------------------------------+
        | training_error          | Error on training data                                                         |
        +-------------------------+--------------------------------------------------------------------------------+
        | training_time           | Time spent on training the model in seconds                                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | trees_json              | Tree encoded using JSON                                                        |
        +-------------------------+--------------------------------------------------------------------------------+
        | valiation_error         | Error on validation data                                                       |
        +-------------------------+--------------------------------------------------------------------------------+
        | unpacked_features       | Feature names (including expanded list/dict features)                          |
        +-------------------------+--------------------------------------------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

        See Also
        --------
        list_fields

        Examples
        --------
        >>> m.get('training_error')
        """
        _mt._get_metric_tracker().track('toolkit.regression.boosted_trees_regression.get')
        return super(BoostedTreesRegression, self).get(field)

    def show(self, tree_id=0, vlabel_hover=False):
        """
        Plot the tree as an SGraph in canvas.

        Parameters
        ----------
        tree_id : int
            The id of the tree to show. Starting from 0 to max_iterations-1.

        vlabel_hover : bool, optional
            If True, hide the label on vertex, and only show the label
            with mouse hovering.

        See Also
        ----------
        show

        Examples
        --------
        >>> m.show(0)
        """
        tree_json = self.get('trees_json')[tree_id]
        g = _SGraphFromJsonTree(tree_json)

        # Make the label for each vertex based on its type
        def get_vlabel(record):
            if record['type'] == 'leaf':
                return str(record['value'])
            elif record['type'] == 'indicator':
                return str(record['name'])
            else:
                return str(record['name']) + '<' + str(record['value'])
        g.vertices['__repr__'] = g.vertices.apply(get_vlabel, str)

        fuchsia = [0.69, 0., 0.48]
        blue = [0.039, 0.55, 0.77]
        orange = [1.0, 0.33, 0.]
        green = [0.52, 0.74, 0.]

        # Make the color for leaf nodes
        def get_leaf_vcolor(record):
            return orange

        leaf_vertices = g.vertices[g.vertices['type'] == 'leaf']
        root_vertex_color = {0: fuchsia}
        leaf_vertex_color = dict([(x['__id'], get_leaf_vcolor(x)) for x in leaf_vertices])
        highlight = {}
        highlight.update(root_vertex_color)
        highlight.update(leaf_vertex_color)
        elabel = 'value' if len(g.edges) > 0 else None

        # Hack: we want the canvas to show tree_{i} instead of g, so here is how we do it.
        graph_name = 'tree_' + str(tree_id)
        locals().update({graph_name: g})
        del g
        locals()[graph_name].show(vlabel='__repr__', elabel=elabel,
                                  vlabel_hover=vlabel_hover, 
                                  highlight=highlight, arrows=True)

    def evaluate(self, dataset, metric='auto'):
        """
        Evaluate the model on the given dataset.


        Parameters
        ----------
        dataset : SFrame
            Dataset in the same format used for training. The columns names and
            types of the dataset must be the same as that used in training.

        metric : str, optional
            Name of the evaluation metric.  Possible values are:
            'auto'      : Compute all metrics.
            'rmse'      : Rooted mean squared error.
            'max_error' : Maximum error.

        Returns
        -------
        out : dict
            A dictionary containing the evaluation result.

        See Also
        ----------
        create, predict

        Examples
        --------
        >>> results = model.evaluate(test_data, 'rmse')

        Notes
        -----
        When evaluating for classifier metrics (e.g. auc,
        confusion_matrix), the classifier threshold is set to 0.5. 
        """        
        _mt._get_metric_tracker().track('toolkit.regression.boosted_trees_regression.evaluate')
        _raise_error_evaluation_metric_is_valid(metric,
                                      ['auto', 'rmse', 'max_error'])

        results = {}
        if metric in ['rmse', 'auto']:
          results = super(BoostedTreesRegression, self).evaluate(dataset, metric = metric)

        if metric in ['max_error', 'auto']:
          predictions = self.predict(dataset)
          target = self.get('target')
          _raise_error_if_column_exists(dataset, predictions, 'dataset', target + '(target column)')
          results['max_error'] = _graphlab.evaluation.max_error(\
                                         predictions, dataset[target])
        return results

    def predict(self, dataset):
        """
        Predict the target column of the given dataset.

        The target column is provided during
        :func:`~graphlab.boosted_trees_regression.create`. If the target column is in the
        `dataset` it will be ignored.

        Parameters
        ----------
        dataset : SFrame
          A dataset that has the same columns that were used during training.
          If the target column exists in ``dataset`` it will be ignored
          while making predictions.

        Returns
        -------
        out : SArray
           Predicted target value for each example (i.e. row) in the dataset.

        See Also
        ----------
        create, predict

        Examples
        --------
        >>> m.predict(testdata)
        """
        _mt._get_metric_tracker().track('toolkit.regression.boosted_trees_regression.predict')
        return super(BoostedTreesRegression, self).predict(dataset, 
                                                        output_type='margin');
    
    @classmethod
    def _get_queryable_methods(cls):
        '''Returns a list of method names that are queryable through Predictive
        Service'''
        return {'predict':{'dataset':'sframe'}}


def create(dataset, target, 
           features=None, max_iterations=10,
           validation_set=None,
           verbose=True, 
           **kwargs):
    """
    Create a :class:`~graphlab.boosted_trees_regression.BoostedTreesRegression` to predict
    a scalar target variable using one or more features. In addition to standard
    numeric and categorical types, features can also be extracted automatically
    from list- or dictionary-type SFrame columns.


    Parameters
    ----------
    dataset : SFrame
        A training dataset containing feature columns and a target column.
        Only numerical typed (int, float) target column is allowed.

    target : str
        The name of the column in ``dataset`` that is the prediction target.
        This column must have a numeric type.

    features : list[str], optional
        A list of columns names of features used for training the model.
        Defaults to None, using all columns.

    max_iterations : int, optional
        The number of iterations for boosting. It is also the number of trees
        in the model.

    max_depth : float, optional
        Maximum depth of a tree.

    step_size : float, [0,1],  optional 
        Step size (shrinkage) used in update to prevents overfitting.  It
        shrinks the prediction of each weak learner to make the boosting
        process more conservative.  The smaller, the more conservative the
        algorithm will be. Smaller step_size is usually used together with
        larger max_iterations.

    min_loss_reduction : float, optional
        Minimum loss reduction required to make a further partition on a
        leaf node of the tree. The larger it is, the more conservative the
        algorithm will be.

    min_child_weight : float, optional
        This controls number of instances needed at least for each leaf.
        The larger it is, the more conservative the algorithm will be.  Set
        it larger when you want to prevent overfitting.  Formally, this is
        minimum sum of instance weight(hessian) in each leaf.  If the tree
        partition step results in a leaf node with the sum of instance
        weight less than min_child_weight, then the building process will
        give up further partitioning. For a regression task, this simply
        corresponds to minimum number of instances needed to be in each
        node.

    row_subsample : float, optional
        Subsample the ratio of the training set in each iteration of tree
        construction.  This is called the bagging trick and usually can help
        prevent overfitting.  Setting it to 0.5 means that model randomly
        collected half of the examples (rows) to grow each tree.

    column_subsample : float, optional
        Subsample ratio of the columns in each iteration of tree
        construction.  Like row_subsample, this also usually can help
        prevent overfitting.  Setting it to 0.5 means that model randomly
        collected half of the columns to grow each tree.

    validation_set : SFrame, optional
        The validation set that is used to watch the validation result as
        boosting progress.

    verbose : boolean, optional
        If True, print progress information during training.

    Returns
    -------
      out : BoostedTreesRegression
          A trained gradient boosted trees model

    References
    ----------
    - `Wikipedia - Gradient tree boosting
      <http://en.wikipedia.org/wiki/Gradient_boosting#Gradient_tree_boosting>`_
    - `Trevor Hastie's slides on Boosted Trees and Random Forest
      <http://jessica2.msri.org/attachments/10778/10778-boost.pdf>`_

    See Also
    --------
    BoostedTreesRegression, graphlab.linear_regression.LinearRegression, graphlab.regression.create

    Examples
    --------

    Setup the data:

    >>> url = 'http://s3.amazonaws.com/gl-testdata/xgboost/mushroom.csv'
    >>> data = graphlab.SFrame.read_csv(url)
    >>> data['label'] = data['label'] == 'p'

    Split the data into training and test data:

    >>> train, test = data.random_split(0.8)

    Create the model:

    >>> model = graphlab.boosted_trees_regression.create(train, target='label')

    Make predictions and evaluate the model:

    >>> predictions = model.predict(test)
    >>> results = model.evaluate(test)

    """
    _mt._get_metric_tracker().track('toolkit.regression.boosted_trees_regression.create')
    model = _sl.create(dataset = dataset,
                        target = target,
                        features = features,
                        model_name = 'boosted_trees_regression',
                        max_iterations = max_iterations,
                        validation_set = validation_set,
                        verbose = verbose, **kwargs)
    return BoostedTreesRegression(model.__proxy__)
