from ConfigParser import ConfigParser as _ConfigParser
from copy import copy as _copy
import json as _json
from logging import getLogger as _getLogger
from tempfile import NamedTemporaryFile as _NamedTemporaryFile
import time as _time
from re import compile as _compile
from requests import ConnectionError as _ConnectionError
from uuid import uuid4 as _random_uuid
import os as _os
import datetime as _datetime

from boto import connect_s3 as _connect_s3
from boto.s3.key import Key as _s3_key

from _file_util import parse_s3_path as _parse_s3_path, s3_recursive_delete as _s3_recursive_delete, s3_delete_key as _s3_delete_key
import graphlab as _gl
import graphlab.canvas as _canvas
import graphlab.connect as _mt

# since _predictive_service_environment imports these, need to have them defined first
MAX_CREATE_TIMEOUT_SECS = 600 # 10m

from _predictive_service_environment import Ec2PredictiveServiceEnvironment
from _predictive_service_environment import predictive_service_environment_factory
from _model_predictive_object import ModelPredictiveObject
from _predictive_service_endpoint import PredictiveServiceEndpoint as _PredictiveServiceEndpoint
from _predictive_object import PredictiveObject as _PredictiveObject

_logger = _getLogger(__name__)
_name_checker = _compile('^[a-zA-Z-]+$')

class PredictiveService(object):

    # S3 Config Section Names
    _DEPLOYMENT_SECTION_NAME = 'Predictive Objects Service Versions'
    _PREDICTIVE_OBJECT_DOCSTRING = 'Predictive Objects Docstrings'
    _ENVIRONMENT_SECTION_NAME = 'Environment Info'
    _SERVICE_INFO_SECTION_NAME = 'Service Info'
    _META_SECTION_NAME = 'Meta'


    def __repr__(self):
        return self.__str__()

    def __str__(self):
        ret = ""
        ret += 'Name: %s' % self.name + '\n'
        ret += 'S3 Path: %s' % self._s3_state_path + '\n'
        ret += 'Description: %s' % self._description + '\n'
        ret += 'API Key: %s' % self._api_key + '\n'
        if self._environment is not None:
            ret += 'Load Balancer DNS Name: %s' % self._environment.load_balancer_dns_name + '\n'

        ret += "\nDeployed predictive objects:\n"
        for (po_name, po_info) in self._predictive_objects.iteritems():
            ret += '\tname: %s, version: %s\n' % (po_name, po_info['version'])

        if len(self._local_changes) == 0:
            ret += "No Pending changes.\n"
        else:
            ret += "Pending changes: \n"
            for (po_name, po_info) in self._local_changes.iteritems():
                version = po_info[1]['version'] if po_info[1] else None
                if version:
                    if version == 1:
                        ret += '\tAdding: %s\n' % po_name
                    else:
                        ret += '\tUpdating: %s to version: %s\n' % (po_name, version)
                else:
                    ret += "\tRemoving: %s\n" % po_name

        return ret

    @property
    def _all_predictive_objects(self):
        '''
        Return all predictive objects that are currently registered with a
        Predictive Service, including both deployed and pending objects.

        Returns
        --------
        out : a dictionary with predictive object names as keys and the current
        state of the predictive object as the correspond value.

        Examples
        --------

        >>> ps.predictive_objects
        {'recommender_one': {'version': 1, 'state': 'deployed', 'docstring':'this is my first recommender'},
         'recommender_two': {'version': 2, 'state': 'pending', 'docstring':'this is my second recommender'},
        '''
        ret = {}
        for (po_name, po_info) in self._predictive_objects.iteritems():
            ret[po_name] = {'state': 'deployed', 'version':po_info['version'], 'docstring': po_info['docstring']}

        # Add local changes on top of deployed ones
        for po_name in self._local_changes:
            (po_obj, po_info) = self._local_changes[po_name]
            if po_obj is not None:
                ret[po_name] = {'state': 'pending', 'version':po_info['version'], 'docstring': po_info['docstring']}
            else:  # delete
                del ret[po_name]

        return ret

    @property
    def deployed_predictive_objects(self):
        '''
        Return all deployed Predictive Objects for a Predictive Service.

        Returns
        --------
        out : dict
            One entry for each Predictive Object. The keys are the names of the
            objects and the values are the corresponding versions.

        Examples
        --------

        >>> ps.deployed_predictive_objects
        {'recommender_one': 2, 'recommender_two': 1}
        '''
        return _copy(self._predictive_objects)

    @property
    def pending_changes(self):
        '''
        Return all currently pending updates for a Predictive Service.

        Returns
        --------
        out: dict
            Keys are the changed Predictive Object names, values are a
            dictionary with two keys: 'action' and 'version'. 'action' is one of
            'add', 'update', or 'remove' and 'version' is the new version
            number or None if action is 'remove'

        Examples
        --------

        >>> ps.pending_changes
        Out:
            {'recommender_one': {'action': 'remove', 'version': None},
             'recommender_three': {'action': 'add', 'version': 1},
             'recommender_two': {'action': 'update', 'version': 2}}
        '''
        ret = {}
        for po_name in self._local_changes:
            (po_obj, po_info) = self._local_changes[po_name]
            po_version = po_info['version'] if po_info is not None else None
            if po_version == 1:
                ret[po_name] = {'action': 'add', 'version':po_version}
            elif po_obj == None:  # deleted case
                ret[po_name] = {'action': 'remove', 'version':None}
            else:
                ret[po_name] = {'action': 'update', 'version':po_version}
        return ret

    @staticmethod
    def _get_s3_state_config(s3_bucket_name, s3_key_name, credentials):
        conn = _connect_s3(**credentials)
        bucket = conn.get_bucket(s3_bucket_name)
        key = bucket.get_key(s3_key_name)

        if not key:
            raise IOError("No Predictive Service at the specified location.")

        with _NamedTemporaryFile() as temp_file:
            key.get_contents_to_file(temp_file)
            temp_file.flush()
            config = _ConfigParser(allow_no_value=True)
            config.optionxform = str
            config.read(temp_file.name)
            temp_file.close()  # deletes temp file

        return config


    def __init__(self, name, s3_state_path, description, api_key, aws_credentials,
                 _new_service = True):
        '''
        Initialize a new Predictive Service object

        Notes
        -----
        Do not call this method directly.

        To create a new Predictive Service, use:
             graphlab.deploy.predictive_service.create(...)

        To load an existing Predictive Service, use
            graphlab.deploy.predictive_service.load(<ps-s3-path>)
        '''
        if type(name) != str:
            raise TypeError("Name of Predictive Service needs to be a string")

        self.name = name
        self._s3_bucket_name, self._s3_key_name = _parse_s3_path(s3_state_path)
        self._s3_state_key = self._s3_key_name + '/state.ini'
        self._description = description
        self._api_key = api_key

        self._local_changes = {}
        self._predictive_objects = {}
        self._s3_state_path = s3_state_path
        self.aws_credentials = aws_credentials
        self._session = _gl.deploy._default_session

        if _new_service:
            # Verify we're not overriding another predictive service.
            bucket = _connect_s3(**self.aws_credentials).get_bucket(self._s3_bucket_name)
            key = bucket.get_key(self._s3_state_key)
            if key:
                raise IOError("There is already a Predictive Service at the specified location. Use"
                              " a different S3 path. If you want to load an existing Predictive"
                              " Service, call 'load(...)'.")

            # Init version data
            self._revision_number = 0

            # No environment yet. A launched one must be attached later.
            self._environment = None

            # Write init data to S3
            self._save_state_to_s3()
        else:
            # Read version data
            self._update_from_s3()

    def get_status(self):
        '''
        Gets the status of each host in the environment.
        '''
        result = '\nSERVICE\n'
        result += self.__str__() + '\n'

        result += 'HOSTS\n'
        for host in self._get_status():
            result += "%s (id: %s)\n" % (host['dns_name'], host['id'])
            result +=  "State: %s" % host['state']
            if host['state'] != 'InService':
                result += " - %s" % host['reason']
            result += "\nServing Models:\n"
            result += _json.dumps(host['models'], sort_keys=True, indent=4, separators=(',', ':'))
            result += '\n'

        print result


    def _get_status(self):
        '''
        Gets the status of each host in the environment.
        '''
        return self._environment.get_status()


    def query(self, po_name, data):
        '''
        Queries the operation of the Predictive Object.

        Same as the query operation the client uses, useful for testing.

        Parameters
        ----------
        po_name : str
            The name of the Predictive Object to query

        data : dict
            Any additional data/parameters to the query

        Examples
        ---------

        If there is a Predictive Object named 'book recommender' and we want to
        recommend some books for users, the following query may be used:

          >>> ps.query('book recommender', {
                'method':'recommend',
                'data': {
                    'users': [
                        {'user_id':12345, 'book_id':2},
                        {'user_id':12346, 'book_id':3},
                    ],
                    'k': 5
                }
            })


        '''
        return self._environment.query(po_name, data, self._api_key)


    def add(self, name, obj):
        '''
        Adds a new model or customized predictive object to predictive service

        The name of the predictive object must be unique to the Predictive
        Service. This operation will not take effect until `apply_changes` is
        called.

        Parameters
        ----------
        name : str
            A unique identifier for the predictive object.

        obj : Model | PredictiveObject
            The model or custom PredictiveObject to add to the predictive service.

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        See Also
        --------
        apply_changes
        '''
        if not isinstance(name, str):
            raise TypeError("'name' must be a string")

        if name == None or name == '':
            raise TypeError("'name' cannot be empty")

        if name in self.deployed_predictive_objects.keys():
            raise ValueError("There is already a predictive object with name '%s'." % name)

        predictive_object = obj

        if isinstance(obj, _gl.Model):
            predictive_object = ModelPredictiveObject(model=obj)
        elif isinstance(obj, _PredictiveObject):
            pass
        else:
            raise TypeError("'obj' parameter has to be either an instance of GraphLab model or an instance of GraphLab PredictiveObject")

        # extract doc string
        docstring = predictive_object.get_doc_string().strip()
        self._local_changes[name] = (predictive_object, {'version':1, 'docstring':docstring})

        _logger.info("New predictive object '%s' added, use apply_changes() to deploy all pending changes, or continue other modification." % name)

    def update(self, name, obj):
        '''
        Updates the version of the predictive object being served.

        Parameters
        ----------
        name : str
            The unique identifier for the model.

        obj : Model | PredictiveObject
            The new version of the model being updated

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        See Also
        --------
        apply_changes
        '''
        new_version = 1
        if name not in self.deployed_predictive_objects.keys():
            if name not in self.pending_changes.keys():
                raise ValueError("Cannot find a predictive object with name '%s'." % name)
            else:
                new_version = 1
        else:
            new_version = 1 + self._predictive_objects[name]['version']

        predictive_object = obj
        if isinstance(obj, _gl.Model):
            predictive_object = ModelPredictiveObject(model=obj)
        elif isinstance(obj, _PredictiveObject):
            pass
        else:
            raise TypeError("'obj' parameter has to be either an instance of GraphLab model or an instance of GraphLab PredictiveObject")

        docstring = predictive_object.get_doc_string().strip()

        self._local_changes[name] = (predictive_object, {'version':new_version, 'docstring':docstring})

        _logger.info("Predictive object '%s' updated to version '%s', use apply_changes() to deploy all pending changes, or continue other modification." % (name, new_version))


    def remove(self, name):
        '''
        Removes a deployed Predictive Object

        Parameters
        ----------
        name : str
            The name of the Predictive Object to be removed

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        See Also
        --------
        apply_changes
        '''
        if name not in self.deployed_predictive_objects.keys():
            if name in self.pending_changes.keys():
                del self._local_changes[name]
            else:
                raise ValueError("Cannot find a model with name '%s'." % name)
        else:
            self._local_changes[name] = (None, None)

        _logger.info("Predictive object '%s' removed, use apply_changes() to deploy all pending changes, or continue other modification." % name)

    def apply_changes(self):
        '''
        Used to apply all pending changes to the Predictive Service.

        User may use add, remove, and update to make changes to the current
        Predictive Service, but these changes will be applied only when apply_changes
        is called.

        If apply_changes returns success, then all pending changes have been
        staged to the S3 bucket associated with the Predictive Service. Each of
        the nodes in the Predictive Service will pick up the state eventually.
        User may use get_status() to check the status of each node.

        See Also
        --------
        add, update, remove
        '''
        self._save_state_to_s3()
        try:
            self._environment.poke()
        except _ConnectionError as e:
            _logger.warn("Unable to connect to running Predictive Service: %s" %
                         (e.message))


    def clear_pending_changes(self):
        '''
        Clears all changes which have not been applied. Clears all actions, done using `add`,
        `update` and `remove`, since the last time `apply_changes` was called.
        '''
        _logger.info('Clearing all pending changes.')
        self._local_changes = {}


    def save_client_config(self, file_path, predictive_service_cname):
        '''
        Create the config file that can be used by applications accessing the
        Predictive Service.

        Parameters
        ----------
        file_path : str
            The path where the config file will be saved.

        predictive_service_cname : str
            The CNAME for the Predictive Service endpoint. It is *highly recommended*
            that all client connect through a CNAME record that you have created.
            If this value is set to None, the "A" record will be used.
            Using the "A" record is only advisable for testing, since this value
            may change over time.
        '''
        out = _ConfigParser(allow_no_value=True)
        out.add_section(PredictiveService._SERVICE_INFO_SECTION_NAME)
        out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'api key', self._api_key)

        if self._environment.certificate_name:
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'verify certificate',
                    not(self._environment.certificate_is_self_signed))
            schema = 'https://'
        else:
            schema = 'http://'

        if predictive_service_cname:
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'endpoint',
                    schema + predictive_service_cname)
        else:
            _logger.warn('Creating client config using the "A" Record name. You\'re crazy to use' \
                             ' this for anything other than testing!')
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'endpoint',
                    schema + self._environment.load_balancer_dns_name)

        with open(file_path, 'w') as f:
            out.write(f)


    def _has_state_changed_on_s3(self):
        '''
        Returns weather any changes have been uploaded to S3 since this Predictive Service was
        loaded or created.
        '''
        if self._revision_number == 0:
            return False
        config = PredictiveService._get_s3_state_config(self._s3_bucket_name, self._s3_state_key,
                                                        self.aws_credentials)
        saved_version_num = config.getint(self._META_SECTION_NAME, 'Revision Number')
        return (self._revision_number != saved_version_num)


    def terminate_service(self, remove_predictive_objects = False,
                          remove_logs = False):
        '''
        Terminates the Predictive Service. The Predictive Service object is not
        usable after terminate_service is called.

        This operation can not be undone.

        Parameters
        ----------
        remove_predictive_objects : bool
            Delete all Predictive Objects associated with this Predictive Service
            from S3.

        remove_logs : bool
            Delete all logs associated with this Predictive Service from S3.
        '''
        if not self._environment:
            _logger.error('Predictive service has already been terminated.')
            return

        # Terminate hosts and delete load balancer.
        self._environment.terminate(remove_logs)
        self._environment = None

        # Delete model data?
        if remove_predictive_objects:
            _logger.info('Deleting model data.')
            try:
                _s3_recursive_delete(self._get_predictive_object_path_prefix(), self.aws_credentials)
                _s3_delete_key(self._s3_bucket_name, self._s3_state_key, self.aws_credentials)
            except:
                _logger.error("Could not delete predictive object data from S3. Please manually delete data under: %s" %
                              self._get_predictive_object_path_prefix())

        # Delete local predictive service endpoint
        _gl.deploy.predictive_services.delete(self.name)

    def _update_from_s3(self):
        '''
        Update the predictive objects from S3.

        Returns
        -------
        out : dict
            Returns a summary of what has changed. There is one entry in the dictionary for each
            predictive object that has changed. The keys are the name of the predictive object. The
            values are tuples: (model_version_number, s3_path_to_new_version).
        '''
        # TODO: make sure we're not "ahead" of S3 due to eventual consistency

        s3_config = PredictiveService._get_s3_state_config(self._s3_bucket_name, self._s3_state_key,
                                                           self.aws_credentials)
        updated_deployment_versions = {}
        model_names = s3_config.options(PredictiveService._DEPLOYMENT_SECTION_NAME)
        for cur_model_name in model_names:
            cur_model_version = s3_config.getint(PredictiveService._DEPLOYMENT_SECTION_NAME,
                                              cur_model_name)
            updated_deployment_versions[cur_model_name] = {'version':cur_model_version, 'docstring': ''}

        if s3_config.has_section(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING):
            for cur_model_name in model_names:
                cur_model_docstring = s3_config.get(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING,
                                                  cur_model_name)
                updated_deployment_versions[cur_model_name]['docstring'] = cur_model_docstring.decode('string_escape')

        if s3_config.has_section(PredictiveService._ENVIRONMENT_SECTION_NAME):
            # Create and attach the environment.
            environment_info = dict(s3_config.items(PredictiveService._ENVIRONMENT_SECTION_NAME))
            environment_info['aws_credentials'] = self.aws_credentials
            self._environment = predictive_service_environment_factory(environment_info)
        else:
            self._environment = None

        diff = {}
        for (cur_po_name, cur_po_info) in updated_deployment_versions.items():
            if(cur_po_name not in self._predictive_objects or
               cur_po_info['version'] != self._predictive_objects[cur_po_name]['version']):
                # Either a new predictive object or new predictive object version
                s3_path_to_new_version = self._get_predictive_object_save_path(cur_po_name,
                                                                               cur_po_info['version'])
                diff[cur_po_name] = (cur_po_info['version'], s3_path_to_new_version)

        # add removed models too
        for (po_name, po_info) in self._predictive_objects.iteritems():
            if po_name not in updated_deployment_versions.keys():
                diff[po_name] = (None, None)

        self._revision_number = s3_config.getint(PredictiveService._META_SECTION_NAME, 'Revision Number')
        self._predictive_objects = updated_deployment_versions
        self._local_changes = {}

        return diff

    def get_metrics_url(self):
        '''
        Return a URL to Amazon CloudWatch for viewing the metrics assoicated with
        the service

        Returns
        -------
        out : str
            A URL for viewing Amazon CloudWatch metrics
        '''
        _logger.info("retrieving metrics from predictive service...")
        try:
            return self._environment.get_metrics_url()
        except Exception as e:
            _logger.error("Error retrieving metrics: %" % e.message)

    def get_metrics(self, start_date=None, end_date=None):
        '''
        Get the metrics associated with the Predictive Service instance.

        Parameters
        -----------
        start_date : datetime, optional
            The begin time (in utc) to query metrics.

        end_date : datetime, optional
            The end time (in utc) to query metrics

        Returns
        -------
        out : dict
            Return a dictionary containing aggregated metric data as a time
            series from the specified date range.

        Notes
        -----
        If start_date and end_date are not given, metrics for the past 7 days
        are retrieved.

        '''
        return self._environment.get_metrics(start_date, end_date)

    @_canvas.inspect.find_vars
    def show(self):
        """
        Visualize the Predictive Service with GraphLab Canvas. This function
        starts Canvas if it is not already running.

        Returns
        -------
        view: graphlab.canvas.view.View
            An object representing the GraphLab Canvas view

        See Also
        --------
        canvas
        """
        _canvas.get_target().state.set_selected_variable(('Predictive Services', self.name))
        return _canvas.show()


    def save(self):
        """
        Saves the Predictive Service information to disk. Information saved
        contains: Name, S3 path, and AWS credentials. Note that only metadata
        of the Predictive Service is stored, it does not impact the actual
        deployed Predictive Service.

        This information can be useful later to access the Predictive Service,
        for example:

            >>> import graphLab
            >>> my_ps = graphlab.deploy.predictive_services[<name>]

        """
        self._session.save(self, typename="PredictiveService")


    def _save_state_to_s3(self):
        # Dump immutable state data to a config
        state = _ConfigParser(allow_no_value=True)
        state.optionxform = str
        state.add_section(PredictiveService._SERVICE_INFO_SECTION_NAME)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'Name', self.name)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'Description', self._description)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'API Key', self._api_key)

        # Save environment, if we have one
        if self._environment:
            state.add_section(PredictiveService._ENVIRONMENT_SECTION_NAME)
            for (key, value) in self._environment._get_state().iteritems():
                state.set(PredictiveService._ENVIRONMENT_SECTION_NAME, key, value)

        # Save deployment version data to config
        state.add_section(PredictiveService._DEPLOYMENT_SECTION_NAME)
        current_predictive_objects = _copy(self._all_predictive_objects)
        for (model_name, info) in current_predictive_objects.iteritems():
            state.set(PredictiveService._DEPLOYMENT_SECTION_NAME, model_name, info['version'])

        state.add_section(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING)
        for (model_name, info) in current_predictive_objects.iteritems():
            state.set(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING, model_name, info['docstring'].encode('string_escape'))

        if self._has_state_changed_on_s3():
            raise IOError("Can not save changes. The Predictive Service has changed on S3. Please "
                          "reload from S3.")

        # Update the revision number
        self._revision_number += 1
        state.add_section(self._META_SECTION_NAME)
        state.set(self._META_SECTION_NAME, 'Revision Number', self._revision_number)

        # Save any new predictive objects to S3.
        for predictive_object_name in self._local_changes:
            (predictive_object, po_info) = self._local_changes[predictive_object_name]
            if predictive_object != None:
                save_path = self._get_predictive_object_save_path(predictive_object_name, po_info['version'])
                predictive_object.save(save_path, self.aws_credentials)

        # Write state file to S3
        with _NamedTemporaryFile() as temp_file:
            state.write(temp_file)
            temp_file.flush()
            conn = _connect_s3(**self.aws_credentials)
            bucket = conn.get_bucket(self._s3_bucket_name)
            key = _s3_key(bucket)
            key.key = self._s3_state_key
            key.set_contents_from_filename(temp_file.name)
            temp_file.close()  # deletes temp file

        # Update our state
        self._local_changes = {}
        self._predictive_objects = dict(zip(current_predictive_objects.keys(),
            [{'version':info['version'], 'docstring': info['docstring']} for info in current_predictive_objects.values()]))


    def _get_predictive_object_save_path(self, predictive_object_name, version_number):
        return "/".join([self._get_predictive_object_path_prefix(), predictive_object_name,
                         str(version_number)])

    def _get_predictive_object_path_prefix(self):
        return "s3://%s/%s/predictive_objects/" % (self._s3_bucket_name, self._s3_key_name)
