import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab import SArray as _SArray
from graphlab import SFrame as _SFrame
import math
import random

def count_words(sa, to_lower=True):
    """
    Count words in the SArray. Return an SArray of dictionary type where
    each element contains the word count for each word that appeared in the
    corresponding input element. The words are split on all whitespace and
    punctuation characters. Only works if this SArray is of string type.

    Parameters
    ----------
    to_lower : bool, optional
        If True, all words are converted to lower case before counting.

    Returns
    -------
    out : SArray
        The SArray of dictionary type, where each element contains the word
        count for each word that appeared in corresponding input element.

    See Also
    --------
    count_ngrams

    Examples
    --------
    >>> sa = graphlab.SArray(["The quick brown fox jumps.",
                             "Word word WORD, word!!!word"])
    >>> count_words(sa)
    dtype: dict
    Rows: 2
    [{'quick': 1, 'brown': 1, 'jumps': 1, 'fox': 1, 'the': 1}, {'word': 5}]
    """

    return sa._count_words(to_lower)

def count_ngrams(sa, n=2, method="word", to_lower=True, ignore_space=True):
    """
    Return an SArray of ``dict`` type where each element contains the count
    for each of the n-grams that appear in the corresponding input element.
    The n-grams can be specified to be either character n-grams or word
    n-grams.  The input SArray must contain strings.

    Parameters
    ----------
    n : int, optional
      The number of words in each n-gram. An ``n`` value of 1 returns word
      counts.

    method : {'word', 'character'}, optional
      If "word", the function performs a count of word n-grams. If
      "character", does a character n-gram count.

    to_lower : bool, optional
      If True, all words are converted to lower case before counting.

    ignore_space : bool, optional
      If method is "character", indicates if spaces between words are
      counted as part of the n-gram. For instance, with the input SArray
      element of "fun games", if this parameter is set to False one
      tri-gram would be 'n g'. If ``ignore_space`` is set to True, there
      would be no such tri-gram (there would still be 'nga'). This
      parameter has no effect if the method is set to "word".

    Returns
    -------
    out : SArray
      An SArray of dictionary type, where each key is the n-gram string
      and each value is its count.

    See Also
    --------
    count_words

    Notes
    -----
    - Ignoring case (with ``to_lower``) involves a full string copy of the
    SArray data. To increase speed for large documents, set ``to_lower``
    to False.

    - Punctuation and spaces are both delimiters when counting word n-grams.
    When counting character n-grams, punctuation is always ignored.

    References
    ----------
    - `N-gram wikipedia article <http://en.wikipedia.org/wiki/N-gram>`_

    Examples
    --------
    Counting word n-grams:

    >>> from graphlab import SArray
    >>> sa = SArray(['I like big dogs. I LIKE BIG DOGS.'])
    >>> count_ngrams(sa, 3)
    dtype: dict
    Rows: 1
    [{'big dogs i': 1, 'like big dogs': 2, 'dogs i like': 1, 'i like big': 2}]

    Counting character n-grams:

    >>> sa = SArray(['Fun. Is. Fun'])
    >>> count_ngrams(sa, 3, "character")
    dtype: dict
    Rows: 1
    {'fun': 2, 'nis': 1, 'sfu': 1, 'isf': 1, 'uni': 1}]
    """
    return sa._count_ngrams(n=n, method=method,
                            to_lower=to_lower, ignore_space=ignore_space)


def tf_idf(dataset, term_frequency_weight='linear'):
    """
    Compute the TF-IDF scores for each word in each document. The collection
    of documents must be in bag-of-words format.

    .. math::
        \mbox{TF-IDF}(w, d) = tf(w, d) * log(N / f(w))

    where :math:`tf(w, d)` is the number of times word :math:`w` appeared in
    document :math:`d`, :math:`f(w)` is the number of documents word :math:`w`
    appeared in, and :math:`N` is the number of documents.

    Parameters
    ----------
    dataset : SArray of type dict, SFrame with columns of type dict
        A bag of words representation of a document corpus.
        Each element is a dictionary representing a single document, where
        the keys are words and the values are the number of times that word
        occurs in that document.

    term_frequency_weight : str, optional
        Possible values: 'linear' and 'log'. If 'log', then a log
        transformation is applied to each term frequency prior to
        computing TF-IDF.

    Returns
    -------
    out : SArray of type dict
        The same document corpus where each score has been replaced by the
        TF-IDF transformed.

    References
    ----------
    - `Wikipedia - TF-IDF <https://en.wikipedia.org/wiki/TFIDF>`_

    Examples
    --------
    >>> docs = graphlab.SArray('http://s3.amazonaws.com/GraphLab-Datasets/nips-text')
    >>> docs_tfidf = graphlab.text.tf_idf(docs)
    """
    _mt._get_metric_tracker().track('toolkit.text_analytics.tf_idf')

    if type(dataset) != _graphlab.SArray:
        raise TypeError, 'tf_idf requires an SArray of dict type, where each '+\
            'dictionary whose keys are words and whose values are word frequency.'

    if len(dataset) == 0:
        return _graphlab.SArray()

    # Convert into 'stacked' format
    d = _graphlab.SFrame({'x': dataset})
    d = d.add_row_number('doc_id').stack('x', ['word', 'tf'])

    # Compute number of docs each word appears in
    doc_freq = d.groupby('word', {'doc_freq': _graphlab.aggregate.COUNT})

    # For each word, get document frequency
    d = d.join(doc_freq, on='word')

    # Compute tf-idf
    N = dataset.size()

    if term_frequency_weight == 'log':
        d['tf'] = d['tf'].apply(lambda x: math.log(x+1))

    d['tfidf'] = d['tf'] * d['doc_freq'].apply(lambda x: math.log(N/x))

    # Return SArray of documents
    tf_sa = d[['doc_id', 'word', 'tfidf']].unstack(['word', 'tfidf'], 'scores').sort('doc_id')['scores']
    return tf_sa.fillna({})


def bm25(dataset, query, k1=1.5, b=.75):
    """
    For a given query and set of documents, compute the BM25 score for each
    document. If we have a query with words q_1, ..., q_n the BM25 score for
    a document is:

        .. math:: \sum_{i=1}^N IDF(q_i)\\frac{f(q_i) * (k_1+1)}{f(q_i + k_1 * (1-b+b*|D|/d_avg))}

    where

    * :math:`\mbox{IDF}(q_i) = log((N - n(q_i) + .5)/(n(q_i) + .5)`
    * :math:`f(q_i)` is the number of times q_i occurs in the document
    * :math:`n(q_i)` is the number of documents containing q_i
    * :math:`|D|` is the number of words in the document
    * :math:`d_avg` is the average number of words per document in the corpus
    * :math:`k_1` and :math:`b` are free parameters.

    Parameters
    ----------
    dataset : SArray of type dict
        An SArray where each element represents a document in bag-of-words
        format, where each key is a word and each value is the number of times
        that word occurs in the document.

    query : SArray of type str
        An SArray where each element is a word.

    k1 : float, optional
        Free parameter which controls the relative importance of term
        frequencies. Recommended values are [1.2, 2.0].

    b : float, optional
        Free parameter which controls how much to downweight scores for long
        documents. Recommended value is 0.75.

    Returns
    -------
    out : SFrame
        An SFrame containing the BM25 score for each document containing one of
        the query words. The doc_id column is the row number of the document.

    Example
    -------

        >>> dataset = graphlab.SArray([
          {'a':5, 'b':7, 'c':10},
          {'a':3, 'c':1, 'd':2},
          {'a':10, 'b':3, 'e':5},
          {'a':1},
          {'f':5}])

        >>> query = ['a', 'b', 'c']
        >>> graphlab.text_analytics.bm25(dataset, query)


    References
    ----------
    .. [BM25] `"Okapi BM-25" <http://en.wikipedia.org/wiki/Okapi_BM25>`_
    """
    _mt._get_metric_tracker().track('toolkit.text_analytics.bm25')

    if type(dataset) != _graphlab.SArray:
        raise TypeError, 'bm25 requires an SArray of dict type, where each '+\
            'dictionary whose keys are words and whose values are word frequency.'

    if type(query) not in [_graphlab.SArray, dict, list]:
        raise TypeError, 'The query must either be an SArray of str type, '+\
           ' a list of strings, or a dictionary.'

    if type(query) == dict:
        query = query.keys()
    if type(query) == list:
        query = _graphlab.SArray(query)
    query = _SFrame({'word': query})

    # Add row ids
    N = len(dataset)
    d = _SFrame({'bow':dataset})
    d = d.add_row_number('doc_id')

    # Compute document length and average doc length
    d['doc_length'] = d['bow'].dict_values().apply(lambda x: sum(x))
    avg_doc_length = d['doc_length'].mean()

    # Convert into 'stacked' format
    d_stacked = d.stack('bow', new_column_name=['word', 'tf'], drop_na=False)
    query = query.join(d_stacked)

    # Compute number of docs in which each word appears
    doc_freq = query.groupby('word', {'doc_freq': _graphlab.aggregate.COUNT})
    query = query.join(doc_freq, on='word')

    # Compute IDF for each word in the query
    query['idf'] = query['doc_freq'].apply(lambda n: math.log((N - n + .5) / (n + .5)))

    # Compute the score of each word for each document
    query['word_score'] = query['idf'] * \
        (query['tf'] * (k1 + 1.0)) / \
        (query['tf'] + k1 * 1.0 * (1.0 - b + b * query['doc_length'] / avg_doc_length))

    # Compute BM25
    result = query.groupby('doc_id', {'bm25': _graphlab.aggregate.SUM('word_score')})
    return result.sort('doc_id')


def parse_sparse(filename, vocab_filename):
    """
    Parse a file that's in libSVM format. In libSVM format each line of the text
    file represents a document in bag of words format:

    num_unique_words_in_doc word_id:count another_id:count

    The word_ids have 0-based indexing, i.e. 0 corresponds to the first
    word in the vocab filename.

    Parameters
    ----------
    filename : str
        The name of the file to parse.

    vocab_filename : str
        A list of words that are used for this data set.

    Returns
    -------
    out : SArray
        Each element represents a document in bag-of-words format.

    Examples
    --------
    If we have two documents:
    1. "It was the best of times, it was the worst of times"
    2. "It was the age of wisdom, it was the age of foolishness"

    Then the vocabular file might contain the unique words, with a word
    on each line, in the following order:
    it, was, the, best, of, times, worst, age, wisdom, foolishness

    In this case, the file in libSVM format would have two lines:
    7 0:2 1:2 2:2 3:1 4:2 5:1 6:1
    7 0:2 1:2 2:2 7:2 8:1 9:1 10:1

    The following command will parse the above two files into an SArray
    of type dict.

    >>> file = 'http://s3.amazonaws.com/GraphLab-Datasets/text/ap.dat'
    >>> vocab = 'http://s3.amazonaws.com/GraphLab-Datasets/text/ap.vocab.txt'
    >>> docs = graphlab.text.parse_sparse(file, vocab)
    """
    _mt._get_metric_tracker().track('toolkit.text_analytics.parse_sparse')

    vocab = _graphlab.SFrame.read_csv(vocab_filename, header=None)['X1']
    vocab = list(vocab)

    docs = _graphlab.SFrame.read_csv(filename, header=None)

    # Remove first word
    docs = docs['X1'].apply(lambda x: x.split(' ')[1:])

    # Helper function that checks whether we get too large a word id
    def get_word(word_id):
        assert int(word_id) < len(vocab), \
                "Text data contains integers that are larger than the \
                 size of the provided vocabulary."
        return vocab[word_id]

    def make_dict(pairs):
        pairs = [z.split(':') for z in pairs]
        ret = {}
        for k, v in pairs:
            ret[get_word(int(k))] = int(v)
        return ret

    # Split word_id and count and make into a dictionary
    docs = docs.apply(lambda x: make_dict(x))

    return docs


def parse_docword(filename, vocab_filename):
    """
    Parse a file that's in "docword" format. This consists of a 3-line header
    comprised of the document count, the vocabulary count, and the number of
    tokens, i.e. unique (doc_id, word_id) pairs. After the header, each line
    contains a space-separated triple of (doc_id, word_id, frequency), where
    frequency is the number of times word_id occurred in document doc_id.

    This format assumes that documents and words are identified by a positive
    integer (whose lowest value is 1).
    Thus, the first word in the vocabulary file has word_id=1.

    2
    272
    5
    1 5 1
    1 105 3
    1 272 5
    2 1 3
    ...

    Parameters
    ----------
    filename : str
        The name of the file to parse.

    vocab_filename : str
        A list of words that are used for this data set.

    Returns
    -------
    out : SArray
        Each element represents a document in bag-of-words format.

    Examples
    --------
    >>> textfile = 'http://s3.amazonaws.com/GraphLab-Datasets/text/docword.nips.txt')
    >>> vocab = 'http://s3.amazonaws.com/GraphLab-Datasets/text/vocab.nips.txt')
    >>> docs = graphlab.text.parse_docword(textfile, vocab)
    """
    _mt._get_metric_tracker().track('toolkit.text_analytics.parse_docword')

    vocab = _graphlab.SFrame.read_csv(vocab_filename, header=None)['X1']
    vocab = list(vocab)

    sf = _graphlab.SFrame.read_csv(filename, header=False)
    sf = sf[3:]
    sf['X2'] = sf['X1'].apply(lambda x: [int(z) for z in x.split(' ')])
    del sf['X1']
    sf = sf.unpack('X2', column_name_prefix='', column_types=[int,int,int])
    docs = sf.unstack(['1', '2'], 'bow').sort('0')['bow']
    docs = docs.apply(lambda x: {vocab[k-1]:v for (k, v) in x.iteritems()})

    return docs


def random_split(dataset, prob=.5):
    """
    Utility for performing a random split for text data that is already in
    bag-of-words format. For each (word, count) pair in a particular element,
    the counts are uniformly partitioned in either a training set or a test
    set.

    Parameters
    ----------
    dataset : SArray of type dict, SFrame with columns of type dict
        A data set in bag-of-words format.

    prob : float, optional
        Probability for sampling a word to be placed in the test set.

    Returns
    -------
    train, test : SArray
        Two data sets in bag-of-words format, where the combined counts are
        equal to the counts in the original data set.

    Examples
    --------
    >>> docs = graphlab.SArray([{'are':5, 'you':3, 'not': 1, 'entertained':10}])
    >>> train, test = graphlab.text.random_split(docs)
    >>> print train
    [{'not': 1.0, 'you': 3.0, 'are': 3.0, 'entertained': 7.0}]
    >>> print test
    [{'are': 2.0, 'entertained': 3.0}]
    """
    _mt._get_metric_tracker().track('toolkit.text_analytics.random_split')


    def grab_values(x, train=True):
        if train:
            ix = 0
        else:
            ix = 1
        return dict([(key, value[ix]) for key, value in x.iteritems() \
                    if value[ix] != 0])

    def word_count_split(n, p):
        num_in_test = 0
        for i in range(n):
            if random.random() < p:
                num_in_test += 1
        return [n - num_in_test, num_in_test]

    # Get an SArray where each word has a 2 element list containing
    # the count that will be for the training set and the count that will
    # be assigned to the test set.
    data = dataset.apply(lambda x: dict([(key, word_count_split(int(value), prob)) \
                                for key, value in x.iteritems()]))

    # Materialize the data set
    data.__materialize__()

    # Grab respective counts for each data set
    train = data.apply(lambda x: grab_values(x, train=True))
    test  = data.apply(lambda x: grab_values(x, train=False))

    return train, test


def stopwords(lang='en'):
    """
    Get common words that are often removed during preprocessing of text data,
    i.e. "stopwords". Currently only English stop words are provided.

    Parameters
    ----------
    lang : str, optional
        The desired language. Default: 'en' (English).

    Returns
    -------
    out : set
        A set of strings.

    Examples
    --------
    You may remove stopwords from an SArray as follows:

    >>> docs = graphlab.SArray([{'are': 1, 'you': 1, 'not': 1, 'entertained':1}])
    >>> docs.dict_trim_by_keys(graphlab.text.stopwords(), True)
    dtype: dict
    Rows: 1
    [{'entertained': 1}]
    """
    if lang=='en' or lang=='english':
        return set(['a', 'able', 'about', 'above', 'according', 'accordingly', 'across', 'actually', 'after', 'afterwards', 'again', 'against', 'all', 'allow', 'allows', 'almost', 'alone', 'along', 'already', 'also', 'although', 'always', 'am', 'among', 'amongst', 'an', 'and', 'another', 'any', 'anybody', 'anyhow', 'anyone', 'anything', 'anyway', 'anyways', 'anywhere', 'apart', 'appear', 'appreciate', 'appropriate', 'are', 'around', 'as', 'aside', 'ask', 'asking', 'associated', 'at', 'available', 'away', 'awfully', 'b', 'be', 'became', 'because', 'become', 'becomes', 'becoming', 'been', 'before', 'beforehand', 'behind', 'being', 'believe', 'below', 'beside', 'besides', 'best', 'better', 'between', 'beyond', 'both', 'brief', 'but', 'by', 'c', 'came', 'can', 'cannot', 'cant', 'cause', 'causes', 'certain', 'certainly', 'changes', 'clearly', 'co', 'com', 'come', 'comes', 'concerning', 'consequently', 'consider', 'considering', 'contain', 'containing', 'contains', 'corresponding', 'could', 'course', 'currently', 'd', 'definitely', 'described', 'despite', 'did', 'different', 'do', 'does', 'doing', 'done', 'down', 'downwards', 'during', 'e', 'each', 'edu', 'eg', 'eight', 'either', 'else', 'elsewhere', 'enough', 'entirely', 'especially', 'et', 'etc', 'even', 'ever', 'every', 'everybody', 'everyone', 'everything', 'everywhere', 'ex', 'exactly', 'example', 'except', 'f', 'far', 'few', 'fifth', 'first', 'five', 'followed', 'following', 'follows', 'for', 'former', 'formerly', 'forth', 'four', 'from', 'further', 'furthermore', 'g', 'get', 'gets', 'getting', 'given', 'gives', 'go', 'goes', 'going', 'gone', 'got', 'gotten', 'greetings', 'h', 'had', 'happens', 'hardly', 'has', 'have', 'having', 'he', 'hello', 'help', 'hence', 'her', 'here', 'hereafter', 'hereby', 'herein', 'hereupon', 'hers', 'herself', 'hi', 'him', 'himself', 'his', 'hither', 'hopefully', 'how', 'howbeit', 'however', 'i', 'ie', 'if', 'ignored', 'immediate', 'in', 'inasmuch', 'inc', 'indeed', 'indicate', 'indicated', 'indicates', 'inner', 'insofar', 'instead', 'into', 'inward', 'is', 'it', 'its', 'itself', 'j', 'just', 'k', 'keep', 'keeps', 'kept', 'know', 'knows', 'known', 'l', 'last', 'lately', 'later', 'latter', 'latterly', 'least', 'less', 'lest', 'let', 'like', 'liked', 'likely', 'little', 'look', 'looking', 'looks', 'ltd', 'm', 'mainly', 'many', 'may', 'maybe', 'me', 'mean', 'meanwhile', 'merely', 'might', 'more', 'moreover', 'most', 'mostly', 'much', 'must', 'my', 'myself', 'n', 'name', 'namely', 'nd', 'near', 'nearly', 'necessary', 'need', 'needs', 'neither', 'never', 'nevertheless', 'new', 'next', 'nine', 'no', 'nobody', 'non', 'none', 'noone', 'nor', 'normally', 'not', 'nothing', 'novel', 'now', 'nowhere', 'o', 'obviously', 'of', 'off', 'often', 'oh', 'ok', 'okay', 'old', 'on', 'once', 'one', 'ones', 'only', 'onto', 'or', 'other', 'others', 'otherwise', 'ought', 'our', 'ours', 'ourselves', 'out', 'outside', 'over', 'overall', 'own', 'p', 'particular', 'particularly', 'per', 'perhaps', 'placed', 'please', 'plus', 'possible', 'presumably', 'probably', 'provides', 'q', 'que', 'quite', 'qv', 'r', 'rather', 'rd', 're', 'really', 'reasonably', 'regarding', 'regardless', 'regards', 'relatively', 'respectively', 'right', 's', 'said', 'same', 'saw', 'say', 'saying', 'says', 'second', 'secondly', 'see', 'seeing', 'seem', 'seemed', 'seeming', 'seems', 'seen', 'self', 'selves', 'sensible', 'sent', 'serious', 'seriously', 'seven', 'several', 'shall', 'she', 'should', 'since', 'six', 'so', 'some', 'somebody', 'somehow', 'someone', 'something', 'sometime', 'sometimes', 'somewhat', 'somewhere', 'soon', 'sorry', 'specified', 'specify', 'specifying', 'still', 'sub', 'such', 'sup', 'sure', 't', 'take', 'taken', 'tell', 'tends', 'th', 'than', 'thank', 'thanks', 'thanx', 'that', 'thats', 'the', 'their', 'theirs', 'them', 'themselves', 'then', 'thence', 'there', 'thereafter', 'thereby', 'therefore', 'therein', 'theres', 'thereupon', 'these', 'they', 'think', 'third', 'this', 'thorough', 'thoroughly', 'those', 'though', 'three', 'through', 'throughout', 'thru', 'thus', 'to', 'together', 'too', 'took', 'toward', 'towards', 'tried', 'tries', 'truly', 'try', 'trying', 'twice', 'two', 'u', 'un', 'under', 'unfortunately', 'unless', 'unlikely', 'until', 'unto', 'up', 'upon', 'us', 'use', 'used', 'useful', 'uses', 'using', 'usually', 'uucp', 'v', 'value', 'various', 'very', 'via', 'viz', 'vs', 'w', 'want', 'wants', 'was', 'way', 'we', 'welcome', 'well', 'went', 'were', 'what', 'whatever', 'when', 'whence', 'whenever', 'where', 'whereafter', 'whereas', 'whereby', 'wherein', 'whereupon', 'wherever', 'whether', 'which', 'while', 'whither', 'who', 'whoever', 'whole', 'whom', 'whose', 'why', 'will', 'willing', 'wish', 'with', 'within', 'without', 'wonder', 'would', 'would', 'x', 'y', 'yes', 'yet', 'you', 'your', 'yours', 'yourself', 'yourselves', 'z', 'zero'])
    else:
        raise NotImplementedError, 'Only English stopwords are currently available.'



def _check_input(dataset):
    if isinstance(dataset, _SFrame):
        assert dataset.num_columns() == 1, \
        "The provided SFrame contains more than one column. It should have " +\
        "only one column of type dict."
        colname = dataset.column_names()[0]
        dataset = dataset[colname]

    assert isinstance(dataset, _SArray), \
    "Provided data must be an SArray."

    assert dataset.dtype() == dict, \
    "Provided data must be of type dict, representing the documents in " + \
    "bag-of-words format. Please consult the documentation."

    return dataset

