import graphlab.connect as _mt
from _job import Job as _Job
from graphlab.deploy import Task as _Task, _Pipeline, _default_session
from graphlab.connect.aws._ec2 import LicenseValidationException

import environment as _environment
import _executionenvironment as _env
import graphlab as _gl

import inspect as _inspect
import logging as _logging
import tempfile as _tempfile
import datetime as _datetime
import time as _time
import os as _os
import pwd as _pwd
import types as _types
import subprocess as _subprocess
import traceback as _traceback

__LOGGER__ = _logging.getLogger(__name__)


def create(tasks=None, name=None, environment=None, function=None, function_arguments=None, required_packages=None):
    """
    Creates a Job and begins executing it. The Job can be defined by either
    specifying a list of tasks, with optional bindings, or with a function,
    with arguments defined. Each Job also needs to know where to run, and that
    is specified by the environment.

    By default, this method will kick off asynchronous work, and return a Job
    object to monitor/manage that work.

    Parameters
    ----------
    tasks : list [Task | str | tuple [ str, dict ]] | str
        List of Tasks to run.

    name : str
        Name for this execution (names the returned Job). Default is environment name + timestamp.

    environment : :class:`~graphlab.deploy.environment.EC2` |
                  :class:`~graphlab.deploy.environment.Hadoop` |
                  :class:`~graphlab.deploy.environment.LocalAsync`
        Optional environment for execution. This would commonly hold access
        keys, launch locations etc.  Also included in the environment object is
        a dictionary for associated metadata to pass to the execution. Default
        is 'LocalAsync', which will have the execution occur in the background
        locally.

    function : function
        Function to be executed in this Job, with arguments to pass to this
        function specified by function_arguments. If a function is specified,
        then tasks cannot be specified. Specifying a function makes it easy to
        get code running in a remote environment.

        If the function returns a dict then it will be collated into the
        results. If the function returns something other than a dict, it will
        be cast to an str and that will be collated into the results. See the
        examples below for more information. See the example below for more
        information.

    function_arguments : dict | list [ dict ] | :class:`~graphlab.SFrame`
        Arguments to pass to the specified function as kwargs. To run multiple
        invocations of the function, simply specify a list of arguments or an
        SFrame. Each element in the list will result in invoking the function
        once. Each row of the SFrame will be used to invoke the function.

    required_packages : list [ str ] | set [ str ]
        List of package requirements (same as disutils.requires) format for
        packages required for running this Job. This is most useful to specify
        any non-standard Python packages required to run the function
        specified.

    Returns
    -------
    job : :py:class:`~graphlab.deploy._job.Job`
        Used for monitoring and managing the execution of the Job.

    Notes
    -----
    - When this method is invoked, each Task specified is cloned and a snapshot
      of it is used for execution. This snapshot can then be queried by
      inspecting the Job object returned.

    Examples
    --------
    Creating a Job using a function instead of Tasks is easy. Just define a
    function and then use it when calling job.create.

    Using a list of dicts to specify arguments:
        >>> def sum_four(one, two, three, four):
        >>>     return {'sum': one + two + three + four}
        >>>
        >>> job = graphlab.deploy.job.create(
        >>>             function=sum_four,
        >>>             function_arguments=[{'one':1, 'two':2,
        >>>                 'three':3, 'four':4}])
        >>>
        >>> results = job.get_results() # SFrame with aggregated results

    Using an SFrame to specify multiple sets of arguments:
        >>> def mult_three(one, two, three):
        >>>     return {'product': one * two * three}
        >>>
        >>> sf = graphlab.SFrame(data={'one':[1,5], 'two':[2,6], 'three':[3,7]})
        >>> job = graphlab.deploy.job.create(function=mult_three,
        >>>                                  function_arguments=sf)
        >>>
        >>> +----+--------------------------------+------------------+-----------+
        >>> | id |             input              |      result      |   status  |
        >>> +----+--------------------------------+------------------+-----------+
        >>> | 0  | {'three': 3, 'two': 2, 'on ... |  {'product': 6}  | Completed |
        >>> | 1  | {'three': 7, 'two': 6, 'on ... | {'product': 210} | Completed |
        >>> +----+--------------------------------+------------------+-----------+
        >>> +---------------------------+---------------------------+---------+
        >>> |         start_time        |          end_time         | message |
        >>> +---------------------------+---------------------------+---------+
        >>> | 2014-11-17 11:06:38+00:00 | 2014-11-17 11:06:38+00:00 |         |
        >>> | 2014-11-17 11:06:40+00:00 | 2014-11-17 11:06:40+00:00 |         |
        >>> +---------------------------+---------------------------+---------+
        >>> [2 rows x 7 columns]

    Each entry in the tasks list could be a pair with a dictionary of bindings
    for that entry. For example:

        >>> tasks = [('task1', {'input':'s3://big-file'}),
        >>>          ('task2', {'input':'/localfile'})]
        >>> graphlab.deploy.job.create(tasks, name='with-bindings')

    """
    tracker = _mt._get_metric_tracker()
    _session = _gl.deploy._default_session

    if tasks is None and function is None:
        raise TypeError('tasks or function needs to be defined')
    if tasks is not None and function is not None:
        raise TypeError('Cannot specify BOTH tasks and function')

    if environment is None:
        environment = _gl.deploy.environments['async']
        if environment is None:
            environment = _environment.LocalAsync('async')
            environment.save()
    else:
        if isinstance(environment, str):
            __LOGGER__.debug("Loading environment: %s" % environment)
            environment = _gl.deploy.environments[environment]

        elif not isinstance(environment, _environment.Environment):
            raise Exception("Unknown type of environment")

        if environment is None:
            raise TypeError("Environment cannot be loaded correctly with name '%s', please confirm this environment exists by calling graphlab.deploy.environments." % environment)

    # always clone the environment, so not mutating existing
    environment = environment.clone()
    __LOGGER__.info("Preparing using environment: %s" % environment.name)

    if name is not None:
        if type(name) is not str:
            raise TypeError("The name you gave for this job is not a string.")

    __LOGGER__.info("Beginning Job Validation.")

    # special handling for function= parameter
    combiner = None
    if function is not None:
        # clobber tasks specified and create a Task for the execution of the function
        tasks = []
        if not _inspect.isfunction(function):
            raise TypeError('Invalid function, must be a Python function.')

        bindings = function_arguments
        if bindings is None:
            bindings = [{}]
        elif not isinstance(bindings, list) and not isinstance(bindings, _gl.SFrame):
            bindings = [bindings]

        # if no name specified make sure the Task names are prefixed with Job name to ensure uniqueness
        if name is None or name == '':
            name = 'job-%s-%s-%s' % (function.__name__, environment.name, _time.time())

        combiner = _Task(name + '-combiner')
        combiner.set_code(_combiner_task)
        idx = -1

        for binding in bindings:
            idx = idx + 1
            task = _Task('%s-%s-%d' % (name, function.__name__, idx))
            task.set_code(_wrapper_task)

            # validate that no GL data structures being passed in function_arguments
            for key, value in binding.items():
                if _Task._is_valid_data_structure(value):
                    raise RuntimeError("Validation Failed: Unsupported type for function_arguments. Function arguments must be basic types that can be serialized into JSON. Invalid function_argument: '%s', type: '%s'" % (key, type(value)))

            task.set_params({'params':binding, 'function':function})
            task.set_outputs(['output'])

            # create dependency for output from task to combiner task
            combiner.set_inputs({'in-%d' % idx : (task, 'output')})
            tasks.append(task)
            _gl.deploy.tasks.delete(task)

        combiner.set_params({'num' : len(bindings)})
        tasks.append(combiner)
        _gl.deploy.tasks.delete(combiner)
        tracker.track('deploy.job.create.fn', value=1)

    # now make the artifacts a list of objects
    if not isinstance(tasks, list):
        # not a list, let's turn it into a list
        tasks = [tasks]

    # if Environment object missing num_hosts attribute, set to 1
    if not hasattr(environment, 'num_hosts'):
        environment.num_hosts = 1

    # If environment.num_hosts > 1 and not using model_parameter_search or parallel_for_each then
    # reset num_host = 1, since multiple hosts will not be used.
    if environment.num_hosts != 1 and \
            all(map(lambda x : isinstance(x, _Task), tasks)):
                __LOGGER__.warn("Ignoring Environment.num_hosts value since execution will occur only on one host. Using num_hosts=1 for this execution.")
                environment.num_hosts = 1
    
    # add required packages to first task in execution
    # ensures the packages will be present on execution
    if required_packages is not None:
        packages = tasks[0].get_required_packages()
        tasks[0].set_required_packages(packages | set(required_packages))

    if name is None or name == '':
        task = tasks[0]
        if isinstance(task, tuple):
            task = task[0]
        if isinstance(task, str):
            names = task
        else:
            names = task.name
        name = 'job-%s-%s-%s' % (names, environment.name, _time.time())

    # if using fn= parameter, we need to wait until name has been determined to
    # set the results_path, so now that the name is settled, set results_path
    if combiner is not None:
        results_path = _get_results_path(environment, name)
        __LOGGER__.info("Job Results SFrame stored: %s" % results_path)
        combiner.set_outputs({'results':results_path})

    validation_msgs = []

    # verify job name is unique
    if _gl.deploy.jobs[name] is not None:
        # found another job same name, fail
        raise RuntimeError("Validation Error: Job already exists with the name '%s', please rename or delete the existing job." % name)

    # Create artifact from their names, if necessary. Clone all artifacts. Add any bindings.
    cloned_artifacts = []
    using_pipeline = False
    for steps in tasks:

        # handle pipeline differently then task
        if isinstance(steps, _Pipeline):
            using_pipeline = True
            binding = None
            if isinstance(steps, tuple):
                (cur_artifact, binding) = steps
            else:
                cur_artifact = steps
            if not isinstance(cur_artifact, _Task) and not isinstance(cur_artifact, _Pipeline):
                cur_artifact = _session._open(cur_artifact, {}, check_cache=True, typename='Task')

            clone = cur_artifact._clone(cur_artifact.name, session_aware=False)

            # apply bindings if paired with task
            if binding is not None:
                _apply_binding_to_task(clone, binding)
            cloned_artifacts.append(clone)
            continue

        if not isinstance(steps, list):
            steps = [steps]

        cloned_step = []
        for step in steps:
            binding = None
            if isinstance(step, tuple):
                (cur_artifact, binding) = step
            else:
                cur_artifact = step
            if not isinstance(cur_artifact, _Task) and not isinstance(cur_artifact, _Pipeline):
                cur_artifact = _session._open(cur_artifact, {}, check_cache=True, typename='Task')

            if cur_artifact is None:
                raise TypeError('Unable to find Task to try to run')

            clone = cur_artifact._clone(cur_artifact.name, session_aware=False)

            # apply bindings if paired with task
            if binding is not None:
                _apply_binding_to_task(clone, binding)

            # if environment is not local then write out any outputs not bound to a location to an
            # intermediate location, so any subsequent steps can find the output
            _validate_output_to_environment(clone, environment, validation_msgs)

            cloned_step.append(clone)

        cloned_artifacts.append(cloned_step)

    num_tasks = len(cloned_artifacts)
    if isinstance(environment, _environment.Local):
        tracker.track('deploy.job.create.local', value=1, properties={'num_tasks':num_tasks})
        env = _env.LocalExecutionEnvironment()
    elif isinstance(environment, _environment.LocalAsync):
        tracker.track('deploy.job.create.localasync', value=1, properties={'num_tasks':num_tasks})
        env = _env.LocalAsynchronousEnvironment()
    elif isinstance(environment, _environment.EC2):
        tracker.track('deploy.job.create.ec2', value=1, properties={'num_tasks':num_tasks})
        # name the ec2 instance the job name
        if not environment.tags:
            environment.tags = {}
        if not 'Name' in environment.tags:
            environment.tags['Name'] = name
        environment.tags['Job'] = name
        env = _env.Ec2ExecutionEnvironment()
    elif isinstance(environment, _environment.Hadoop):
        tracker.track('deploy.job.create.hadoop', value=1, properties={'num_tasks':num_tasks})
        env = _env.HadoopExecutionEnvironment()
    else:
        raise Exception("Validation Failed: Unknown execution environment.")

    if len(validation_msgs) > 0:
        for msg in validation_msgs:
            __LOGGER__.error(msg)
        raise RuntimeError("Validation Failed: output(s) not set to appropriate location for execution environment. See logs for more details.")
    
    try:
        __LOGGER__.info("Validation complete. Job: '%s' ready for execution" % name)
        job = env.run(_session, cloned_artifacts, name, environment)
        _session.register(job)
        job.save() # save the job once prior to returning.

        # add a .get_results() method to this job.
        if function is not None:
            job.get_results = _types.MethodType(_get_results, job)

        return job
    except LicenseValidationException as e:
        # catch exception and print license check hint message here instead of raise
        __LOGGER__.info(e)

def _get_results_path(environment, job_name, folder_name = 'stored_results'):
    """
    Decide on a place to store the results of the job based on the environment passed in.
    """
    dirname = job_name
    results_path = _os.path.join(_tempfile.gettempdir(), dirname, folder_name)
    if isinstance(environment, _environment.EC2):

        # get credentials
        if not hasattr(environment, 'aws_access_key'):
            # no access key in environment, try to get from get_credentials
            try:
                access_key, secret_key = _gl.aws.get_credentials()
            except KeyError:
                # not in get_credentials and not in environment, fail validation
                raise RuntimeError('Validation Failed: Unable to retrieve AWS credentials for S3, needed for writing results from this execution. Use graphlab.aws.set_credentials() or set the credentials in the EC2 Environment object.')
        else:
            access_key = environment.aws_access_key
            secret_key = environment.aws_secret_key

        path = _os.path.join(environment.s3_bucket, environment.s3_log_folder_path, dirname, folder_name)
        results_path = "s3://%s:%s:%s" % (access_key, secret_key, path)
    elif isinstance(environment, _environment.Hadoop):
        # get the root path in hdfs land, to support actually launching in HDFS
        if environment.get_config_dir() is not None:
            root = _subprocess.check_output(
                        ["hdfs", "--config", _os.path.expanduser(environment.get_config_dir()),
                            "getconf", "-confKey", "fs.defaultFS"]).rstrip()
        else:
            root = _subprocess.check_output(["hdfs", "getconf", "-confKey", "fs.defaultFS"]).rstrip()

        # get the username for running the Hadoop job.
        user = _pwd.getpwuid(_os.getuid()).pw_name
        if 'HADOOP_USER_NAME' in _os.environ:
            user = _os.environ['HADOOP_USER_NAME']
        dirname = dirname + '-' + folder_name
        results_path = _os.path.join(root, 'user', user, dirname)

    return results_path

def _get_results(self):
    """
    Retrieve the results of the job execution by retrieving the output
    of the 'collector' Task.
    """
    status = self.get_status()
    if status not in ['Failed', 'Completed', 'Canceled']:
        __LOGGER__.info("Current status: %s. Results are not available until the Job is in a final state: [Completed, Failed, Canceled]." % status)
        return None

    if status == 'Failed':
        __LOGGER__.info("Job Failed, so results are not available, please see the logs to see why the Job failed.")
        return None

    combiner = self._sequence[-1][0] # combiner task
    location = combiner.get_outputs()['results']
    results = _gl.SFrame(location)
    results.add_row_number(column_name='index')
    return results

def _wrapper_task(task):
    function = task.params['function']
    params = task.params['params']
    result = {}
    message = ''
    try:
        start = _datetime.datetime.now()
        result = function(**params)
        print 'params: %s, result: %s' % (params, result)
        end = _datetime.datetime.now()
        status = 'Completed'

        if not isinstance(result, dict) and not isinstance(result, str):
            message = "Unable to persist function return. Function should return dict or str type. Function return type: '%s'" % type(result)
            result = {}
            status = 'Failed'

    except Exception as e:
        message = _traceback.format_exc(0)
        status = 'Failed'
        end = _datetime.datetime.now()
        raise e
    finally:
        task.outputs['output'] = _gl.SFrame({'input':[params], 'result':[result], 'start_time':[start], 'end_time':[end], 'status':[status], 'message':[message]})

def _combiner_task(task):
    import graphlab as gl
    print 'Combining %d tasks' % task.params['num']
    results_list = []
    for result in task.inputs.itervalues():
        # iterate one row SFrame to get dict for each row
        for row in result:
            results_list.append(row)
    results_sa = gl.SArray(results_list)
    results = results_sa.unpack(column_name_prefix=None)
    results = results[['input', 'result', 'status', 'start_time', 'end_time', 'message']]
    results = results.add_row_number()
    task.outputs['results'] = results

def _apply_binding_to_task(task, binding, allow_object=False):
    """
    Helper method to apply bindings to a given task. This function modifies the given task object and returns
    it modified.

    When specifying bindings, there is no qualification for a binding being an input, output, or param, so when
    trying to apply the binding we need to try each. A TypeError is thrown whenever the name for an input is already
    applied to another type of input (ex. set_input('foo') called, then later set_param('foo','bar') will throw a
    TypeError. So it is expected that the name for each binding throw a TypeError for the other types of slot names
    (so if param named 'foo', then TypeError should be thrown for set_input('foo') and set_output('foo')
    """
    for (param_name, param_value) in binding.iteritems():
        bound = False

        if param_name in task.get_inputs():
            try:
                # special-case when task.run() is specifying input binding with object
                # this way when _realize_input is called it will find the object dependency
                if allow_object is True and _Task._is_valid_data_structure(param_value):
                    if not hasattr(task, '_local_binding') or task._local_binding is None:
                        task._local_binding = {}
                    task._local_binding[param_name] = param_value
                else:
                    task.set_inputs({param_name : param_value})
                bound = True
                __LOGGER__.debug("Applied binding named: '%s' as input, with value: '%s'" % (param_name, param_value))
            except TypeError:
                pass

        if param_name in task.get_outputs():
            try:
                task.set_outputs({param_name : param_value})
                bound = True
                __LOGGER__.debug("Applied binding named: '%s' as output, with value: '%s'" % (param_name, param_value))
            except TypeError:
                pass

        if param_name in task.get_params():
            try:
                task.set_params({param_name:param_value})
                bound = True
                __LOGGER__.debug("Applied binding named: '%s' as param, with value: '%s'" % (param_name, param_value))
            except TypeError:
                pass

        if bound is False:
            __LOGGER__.warning("Binding not applied since not found in input, output, or params. Name: '%s', value: '%s'" % (param_name, param_value))

def _validate_output_to_environment(task, environment, validation_msgs):
    """
    Validate that output paths match the environment - if Ec2/Hadoop environment then paths should be S3/HDFS etc.
    Also emit warnings if params begin with local paths and environment is not local
    """
    if isinstance(environment, _environment.LocalAsync) or isinstance(environment, _environment.Local):
        # nothing to validate here, since the paths might be correct when running locally
        return
    elif isinstance(environment, _environment.Hadoop):
        __validate_path_prefixes(task, 'hdfs://', validation_msgs)
    elif isinstance(environment, _environment.EC2):
        __validate_path_prefixes(task, 's3://', validation_msgs)
    else:
        raise TypeError("Invalid environment encountered, unable to create Job, Supported environments are of type: HadoopEnvironment, EC2Environment, LocalEnvironment, or LocalAsynchronous")

def __validate_path_prefixes(task, prefix, validation_msgs):
    """
    Validate that task outputs all begin with specified prefix, otherwise raise TypeError.
    """
    for name, value in task.get_outputs().iteritems():
        if value is not None and not str(value).strip().lower().startswith(prefix):
           validation_msgs.append("Validation error: Task: '%s', output named: '%s' not being saved to expected prefix: '%s', being saved to: '%s'" % (task.name, name, prefix, value))

