import graphlab.connect as _mt
import graphlab as _graphlab
import random as _random
import array as _array
import logging as _logging
from graphlab.toolkits._model import Model as _Model

import graphlab.aggregate as _Aggregate
from graphlab.data_structures.sarray import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.deps import numpy as _numpy, HAS_NUMPY as _HAS_NUMPY
_has_pyplot = False
try:
    import matplotlib.pyplot as _pp
    from matplotlib import rcParams as _rcParams
    _has_pyplot = True
except:
    pass


def _create(observation_data,
           user_id='user_id', item_id='item_id', target=None,
           user_data=None, item_data=None,
           ranking=True,
           verbose=True):
    r"""
    A unified interface for training recommender models. The trained model can
    be used to predict ratings and make recommendations.

    To use specific options of a desired model, use the ``create`` function
    of the corresponding model.

    Parameters
    ----------
    observation_data : SFrame
        The dataset to use for training the model. It must contain a column of
        user ids and a column of item ids. Each row represents an observed
        interaction between the user and the item.  The (user, item) pairs
        are stored with the model so that they can later be excluded from
        recommendations if desired. It can optionally contain a target ratings
        column. All other columns are interpreted by the underlying model as
        side features for the observations.

        The user id and item id columns must be of type 'int' or 'str'. The
        target column must be of type 'int' or 'float'.

    user_id : string, optional
        The name of the column in `observation_data` that corresponds to the
        user id.

    item_id : string, optional
        The name of the column in `observation_data` that corresponds to the
        item id.

    target : string, optional
        The `observation_data` can optionally contain a column of scores
        representing ratings given by the users. If present, the name of this
        column may be specified variables `target`.

    user_data : SFrame, optional
        Side information for the users.  This SFrame must have a column with
        the same name as what is specified by the `user_id` input parameter.
        `user_data` can provide any amount of additional user-specific
        information.

    item_data : SFrame, optional
        Side information for the items.  This SFrame must have a column with
        the same name as what is specified by the `item_id` input parameter.
        `item_data` can provide any amount of additional item-specific
        information.

    ranking : bool, optional
        Determine whether or not the goal is to rank items for each user.

    verbose : bool, optional
        Enables verbose output.

    Returns
    -------
    out :  A trained model, one of
        :class:`graphlab.recommender.item_similarity_recommender.ItemSimilarityRecommender`,
        :class:`graphlab.recommender.factorization_recommender.FactorizationRecommender`,
        :class:`graphlab.recommender.ranking_factorization_recommender.RankingFactorizationRecommender`,
        or
        :class:`graphlab.recommender.popularity_recommender.PopularityRecommender`.

    Examples
    --------
    **Basic usage**

    Given basic user-item observation data, an
    :class:`~graphlab.recommender.item_similarity_recommender.ItemSimilarityRecommender` is created:

    >>> sf = graphlab.SFrame({'user_id': ['0', '0', '0', '1', '1', '2', '2', '2'],
    ...                       'item_id': ['a', 'b', 'c', 'a', 'b', 'b', 'c', 'd']})
    >>> m = graphlab.recommender.create(sf)
    >>> recs = m.recommend()

    **Creating a model for ratings data**

    This trains a :class:`~graphlab.recommender.factorization_recommender.FactorizationRecommender` that
    can predict target ratings:

    >>> sf2 = graphlab.SFrame({'user_id': ['0', '0', '0', '1', '1', '2', '2', '2'],
    ...                        'item_id': ['a', 'b', 'c', 'a', 'b', 'b', 'c', 'd'],
    ...                        'rating': [1, 3, 2, 5, 4, 1, 4, 3]})
    >>> m2 = graphlab.recommender.create(sf2, target="rating")

    **Creating specific models**

    Specific models allow for a number of additional options during create.
    The available recommenders are all in the graphlab.recommender namespace.
    For the complete list of acceptable options, please refer to the documentation
    for individual models. Such options can be passed to the underlying model
    just like any other parameter. For example, the following code creates
    an :class:`~graphlab.recommender.ItemSimilarityRecommender` with a space-saving
    option called `only_top_k`. The returned model stores only the 25 most
    similar items for item:

    >>> from graphlab.recommender import item_similarity_recommender
    >>> item_similarity_recommender.create(sf, only_top_k=2)
    """

    if not ranking:
        _mt._get_metric_tracker().track('toolkit.recsys.create.rating')
    else:
        _mt._get_metric_tracker().track('toolkit.recsys.create.ranking')

    from item_similarity_recommender import ItemSimilarityRecommender
    from factorization_recommender import FactorizationRecommender
    from ranking_factorization_recommender import RankingFactorizationRecommender

    if not (isinstance(observation_data, _SFrame)):
        raise TypeError('observation_data input must be a SFrame')

    # Check whether target is binary
    binary_target = False
    if target is not None:
        if observation_data[target].dtype() == int:
            if set(observation_data[target].unique()) == set([0,1]):
                binary_target = True

    side_data = (user_data is not None) or (item_data is not None)
    if user_data is not None:
        if not isinstance(user_data, _SFrame):
            raise TypeError('Provided user_data must be an SFrame.')
    if item_data is not None:
        if not isinstance(item_data, _SFrame):
            raise TypeError('Provided item_data must be an SFrame.')

    if target is None:
        if ranking:
            if side_data:
                method = 'ranking_factorization_recommender'
            else:
                method = 'item_similarity'
        else:
            if side_data:
                method = 'ranking_factorization_recommender'
            else:
                method = 'item_similarity'
    else:
        if ranking:
            if side_data:
                method = 'ranking_factorization_recommender'
            else:
                method = 'ranking_factorization_recommender'
        else:
            if side_data:
                method = 'factorization_recommender'
            else:
                method = 'factorization_recommender'

    opts = {'model_name': method}

    response = _graphlab.toolkits._main.run("recsys_init", opts)
    model_proxy = response['model']

    if user_data is None:
        user_data = _graphlab.SFrame()
    if item_data is None:
        item_data = _graphlab.SFrame()

    opts = {'dataset': observation_data,
            'user_id': user_id,
            'item_id': item_id,
            'target': target,
            'user_data': user_data,
            'item_data': item_data,
            'nearest_items': _graphlab.SFrame(),
            'model': model_proxy,
            'random_seed': 1}

    if method == "ranking_factorization_recommender":
        opts['ranking_regularization'] = 0.25

    plot = False

    # Train model with default options
    response = _graphlab.toolkits._main.run('recsys_train', opts, verbose, plot)

    if method == "item_similarity":
        m = ItemSimilarityRecommender(response['model'])
    elif method == "factorization_recommender":
        m = FactorizationRecommender(response['model'])
    elif method == "ranking_factorization_recommender":
        m = RankingFactorizationRecommender(response['model'])
    else:
        raise RuntimeError("Provided method not recognized.")
    return m


def __plot_histogram(measurements, means, names=None, metric_name=None):
    """
    Plot histograms of the measurements, overlaid with vertical lines
    representing the means of the measurements.

    Parameters
    ----------
    measurements : list
        List of measurements (recall, precision or RMSE).

    means : list
        List of doubles, intended to be the mean of each list in
        'measurements'.

    names : list
        List of model name strings.

    metric_name : string
        Name of the metric.
    """
    num_measurements = len(measurements)

    # A list of colors for plotting
    COLORS_LIST = ['b', 'g', 'r', 'k', 'm', 'c']
    colors = [COLORS_LIST[i % len(COLORS_LIST)] for i in range(num_measurements)]

    hist_handle = _pp.hist(measurements, bins=20,
                           color=colors,
                           label=names, hold=True)
    _pp.legend()
    # hist() returns a list of two lists, the first is a list of all the counts,
    # the second is a list of the bin centers.  We need the maximum count so we know
    # how tall the vertical line should be.  So we take the max of the max of the
    # first list of lists
    max_count = max([max(hcount) for hcount in hist_handle[0]])
    _pp.vlines(means, 0, max_count, colors=colors)
    _pp.xlabel(metric_name)
    _pp.ylabel('Counts')


def __plot_overlap_hists(results, label, names, bins=20, alpha=0.3):
    """
    Plot overlapping (un-normalized) histograms for a list of one-dimensional
    series.

    Parameters
    ----------
    results : list
        List of list-like objects. Each element is plotted as a separate histogram.

    label : string
        Label for the x-axis of the histogram.

    names : list
        Names for each series in `results'.

    bins : int
        Number of bins. Default is 20.

    alpha : float
        Opacity of the histogram patches. Default is 0.4.
    """
    assert _HAS_NUMPY, 'Cannot find numpy'
    fig, ax = _pp.subplots()

    # plot first series to fix the bins
    counts, bins, patches = ax.hist(results[0], bins=bins, alpha=alpha, lw=0.1,
                                    label=names[0])
    clr = patches[0].get_facecolor()
    counts = _numpy.insert(counts, 0, 0)
    ax.step(bins, counts, color=clr, lw=5)

    # plot the rest of the series
    if len(results) > 1:
        for series, name in zip(results[1:], names[1:]):
            counts, bins, patches = ax.hist(series, bins=bins, alpha=alpha, lw=0.03,
                                            label=name, fill=True)
            clr = patches[0].get_facecolor()
            counts = _numpy.insert(counts, 0, 0)
            ax.step(bins, counts, color=clr, lw=4)

    ax.ticklabel_format(style='sci', scilimits=(0, 0), axis='y')
    ax.set_xlabel(label)
    ax.set_ylabel('Frequency')
    ax.legend()
    fig.show()


def _compare_results_precision_recall(results, model_names=None):
    """
    Compare models that output precision/recall. Display the per-user
    precision/recall scatter plot, the histograms of precision, and the
    histograms of recall.

    Parameters
    ----------
    results : list
        List of SFrames. Each SFrame describes the evaluation results for a
        separate model.

    model_names : list
        List of model name strings.
    """

    num_models = len(results)
    COLORS_LIST = ['b', 'g', 'r', 'k', 'm', 'c']

    if num_models < 1:
        return

    if model_names is None:
        model_names = ["model {}".format(i) for i in range(num_models)]

    pr_curves_by_model = [res['precision_recall_overall'] for res in results]
    fig, ax = _pp.subplots()

    for i in range(num_models):
        pr_curve = pr_curves_by_model[i]
        pr_curve = pr_curve.sort('recall')
        name = 'Model ' + str(i + 1)
        if model_names is not None:
            name = model_names[i]

        ax.plot(list(pr_curve['recall']), list(pr_curve['precision']),
                COLORS_LIST[i % len(COLORS_LIST)], label=name)

    ax.set_title('Precision-Recall Averaged Over Users')
    ax.set_xlabel('Recall')
    ax.set_ylabel('Precision')
    ax.legend()
    fig.show()


def _compare_results_rmse2(results, model_names=None):
    """
    Compare models that output root-mean-squared error (RMSE).

    Parameters
    ----------
    results : list
        List of dataframes describing evaluation results for each model.

    model_names : list
        List of model name strings.
    """
    assert _HAS_NUMPY, 'Cannot find numpy'
    ## Validate the results
    num_models = len(results)

    if num_models < 1 or num_models > len(_rcParams['axes.color_cycle']):
        return

    if model_names is None:
        model_names = ["model {}".format(i) for i in range(num_models)]

    ## Histograms of per-user and per-item RMSE
    user_rmse_by_model = [list(elem['rmse_by_user']['rmse']) for elem in results]
    __plot_overlap_hists(user_rmse_by_model, 'Per-User RMSE', model_names, bins=100)

    item_rmse_by_model = [list(elem['rmse_by_item']['rmse']) for elem in results]
    __plot_overlap_hists(item_rmse_by_model, 'Per-Item RMSE', model_names, bins=100)

    ## Bar chart of Overall RMSE by model
    overall_rmse_by_model = [elem['rmse_overall'] for elem in results]

    fig, ax = _pp.subplots()
    BAR_WIDTH = 0.3
    centers = _numpy.arange(num_models) + BAR_WIDTH
    ax.bar(centers, overall_rmse_by_model, BAR_WIDTH)
    ax.set_xticks(centers + BAR_WIDTH / 2)
    ax.set_xticklabels(model_names)
    ax.set_title('Overall RMSE')
    fig.show()


def compare_models(dataset, models, model_names=None, user_sample=1.0,
                   metric='auto',
                   target=None,
                   exclude_known_for_precision_recall=True,
                   make_plot=True,
                   verbose=True,
                   **kwargs):
    """
    Compare the prediction or recommendation performance of recommender models
    on a common test dataset.

    Models that are trained to predict ratings are compared separately from
    models that are trained without target ratings.  The ratings prediction
    models are compared on root-mean-squared error, and the rest are compared on
    precision-recall.

    Parameters
    ----------
    dataset : SFrame
        The dataset to use for model evaluation.

    models : list[recommender models]
        List of trained recommender models.

    model_names : list[str], optional
        List of model name strings for display.

    user_sample : float, optional
        Sampling proportion of unique users to use in estimating model
        performance. Defaults to 1.0, i.e. use all users in the dataset.

    metric : str, {'auto', 'rmse', 'precision_recall'}, optional
        Metric for the evaluation. The default automatically splits
        models into two groups with their default evaluation metric respectively:
        'rmse' for models trained with a target, and 'precision_recall'
        otherwise.

    target : str, optional
        The name of the target column for evaluating rmse. If the model is
        trained with a target column, the default is to using the same column.
        If the model is trained without a target column and `metric='rmse'`,
        then this option must be provided by user.

    exclude_known_for_precision_recall : bool, optional
        A useful option when `metric='precision_recall'`. Recommender models
        automatically exclude items seen in the training data from the
        final recommendation list. If the input evaluation `dataset` is the
        same as the data used for training the models, set this option to False.

    make_plot : bool, optional
        If true, a plot is made using matplotlib.

    verbose : bool, optional
        If true, print the progress.

    Returns
    -------
    out : list[SFrame]
        A list of results where each one is an sframe of evaluation results of
        the respective model on the given dataset

    Examples
    --------
    If you have created two ItemSimilarityRecommenders ``m1`` and ``m2`` and have
    an :class:`~graphlab.SFrame` ``test_data``, then you may compare the
    performance of the two models on test data using:

    >>> train_data = graphlab.SFrame({'user_id': ["0", "0", "0", "1", "1", "2", "2", "2"],
    ...                               'item_id': ["a", "c", "e", "b", "f", "b", "c", "d"]})
    >>> test_data = graphlab.SFrame({'user_id': ["0", "0", "1", "1", "1", "2", "2"],
    ...                              'item_id': ["b", "d", "a", "c", "e", "a", "e"]})
    >>> from graphlab.recommender import item_similarity_recommender
    >>> m1 = item_similarity_recommender.create(train_data)
    >>> m2 = item_similarity_recommender.create(train_data, only_top_k=1)
    >>> graphlab.recommender.compare_models(test_data, [m1, m2], model_names=["m1", "m2"])

    The evaluation metric is automatically set to 'precision_recall', and the
    evaluation will be based on recommendations that exclude items seen in the
    training data.

    If you want to evaluate on the original training set:

    >>> graphlab.recommender.compare_models(train_data, [m1, m2],
    ...                                     exclude_known_for_precision_recall=False)

    Suppose you have four models, two trained with a target rating column, and
    the other two trained without a target. By default, the models are put into
    two different groups with "rmse", and "precision-recall" as the evaluation
    metric respectively.

    >>> train_data2 = graphlab.SFrame({'user_id': ["0", "0", "0", "1", "1", "2", "2", "2"],
    ...                                'item_id': ["a", "c", "e", "b", "f", "b", "c", "d"],
    ...                                'rating': [1, 3, 4, 5, 3, 4, 2, 5]})
    >>> test_data2 = graphlab.SFrame({'user_id': ["0", "0", "1", "1", "1", "2", "2"],
    ...                               'item_id': ["b", "d", "a", "c", "e", "a", "e"],
    ...                               'rating': [3, 5, 4, 4, 3, 5, 2]})
    >>> from graphlab.recommender import factorization_recommender
    >>> m3 = factorization_recommender.create(train_data2, target='rating')
    >>> m4 = factorization_recommender.create(train_data2, target='rating')
    >>> graphlab.recommender.compare_models(test_data2, [m1, m2, m3, m4])

    To compare all four models using the same 'precision_recall' metric, you can
    do:

    >>> graphlab.recommender.compare_models(test_data2, [m1, m2, m3, m4],
    ...                                     metric='precision_recall')
    """

    _mt._get_metric_tracker().track('toolkit.recsys.compare_models')

    num_models = len(models)

    if model_names is None:
        model_names = ['M' + str(i) for i in range(len(models))]

    if num_models < 1:
        raise ValueError("Must pass in at least one recommender model to \
                           evaluate")

    if model_names is not None and len(model_names) != num_models:
        raise ValueError("Must pass in the same number of model names as \
                          models")

    # if we are asked to sample the users, come up with a list of unique users
    if user_sample < 1.0:
        user_id_name = models[0].get('user_id')
        if user_id_name is None:
            raise ValueError("user_id not set in model(s)")
        user_sa = dataset[user_id_name]
        unique_users = list(user_sa.unique())
        nusers = len(unique_users)
        ntake = int(round(user_sample * nusers))

        _random.shuffle(unique_users)

        users = unique_users[:ntake]
        print "compare_models: using " + str(ntake) + \
              " users to estimate model performance"
        users = frozenset(users)
        ix = [u in users for u in dataset[user_id_name]]
        dataset_subset = dataset[_SArray(ix) == True]
    else:
        dataset_subset = dataset

    results = []
    for (m, mname) in zip(models, model_names):
        if verbose:
            print 'PROGRESS: Evaluate model %s' % mname
        r = m.evaluate(dataset_subset,
                       metric,
                       exclude_known_for_precision_recall,
                       target,
                       verbose=verbose,
                       cutoffs=range(2, 50, 2),
                       **kwargs)
        results.append(r)

    if _has_pyplot is True:
        # separate into two sets of models: those that evaluate rmse by default,
        # and those that evaluate precision-recall.
        ## The following relies on the fact that evaluate_rmse return a dict
        ## while evaluate_precision_recall returns an SFrame with column names
        ## of user_id, 'cutoff', 'precision', 'recall', 'count'
        is_pr = [['precision' in k for k in rr.keys()] for rr in results]
        results_pr = [results[i] for i in range(num_models) if is_pr[i]]
        results_rmse = [results[i] for i in range(num_models) if is_pr[i] is False]
        model_names_pr = None
        model_names_rmse = None
        if model_names is not None:
            model_names_pr = [model_names[i] for i in range(num_models) if is_pr[i]]
            model_names_rmse = [model_names[i] for i in range(num_models) if not is_pr[i]]

        if make_plot:
            if len(results_pr) > 0:
                _compare_results_precision_recall(results_pr, model_names_pr)
            if len(results_rmse) > 0:
                _compare_results_rmse2(results_rmse, model_names_rmse)
            _pp.show()
    else:
        "Warning: Matplotlib could not be imported - no plot output."

    return results


def precision_recall_by_user(observed_user_items,
                             recommendations,
                             cutoffs=[10]):
    """
    Compute precision and recall at a given cutoff for each user. In information
    retrieval terms, precision represents the ratio of relevant, retrieved items
    to the number of relevant items. Recall represents the ratio of relevant,
    retrieved items to the number of relevant items.

    Let :math:`p_k` be a vector of the first :math:`k` elements in the
    recommendations for a particular user, and let :math:`a` be the set of items
    in ``observed_user_items`` for that user. The "precision at cutoff k" for
    this user is defined as

    .. math::
        P(k) = \\frac{ | a \cap p_k | }{k},

    while "recall at cutoff k" is defined as

    .. math::
        R(k) = \\frac{ | a \cap p_k | }{|a|}

    The order of the elements in the recommendations affects the returned
    precision and recall scores.

    Parameters
    ----------
    observed_user_items : SFrame
        An SFrame containing observed user item pairs, where the first
        column contains user ids and the second column contains item ids.

    recommendations : SFrame
        An SFrame containing columns pertaining to the user id, the item id,
        the score given to that pair, and the rank of that item among the
        recommendations made for user id. For example, see the output of
        recommend() produced by any graphlab.recommender model.

    cutoffs : list[int], optional
        The cutoffs to use when computing precision and recall.

    Returns
    -------
    out : SFrame
        An SFrame containing columns user id, cutoff, precision, recall, and
        count where  the precision and recall are reported for each user at
        each requested cutoff, and count is the number of observations for
        that user id.

    Notes
    -----
    The corner cases that involve empty lists were chosen to be consistent
    with the feasible set of precision-recall curves, which start at
    (precision, recall) = (1,0) and end at (0,1). However, we do not believe
    there is a well-known concensus on this choice.

    Examples
    --------
    Given SFrames ``train_data`` and ``test_data`` with columns user_id
    and item_id:

    >>> from graphlab.toolkits.recommender.util import precision_recall_by_user
    >>> m = graphlab.recommender.create(train_data)
    >>> recs = m.recommend()
    >>> precision_recall_by_user(test_data, recs, cutoffs=[5, 10])
    """
    _mt._get_metric_tracker().track('toolkit.recsys.precision_recall_by_user')

    assert type(observed_user_items) == _SFrame
    assert type(recommendations) == _SFrame
    assert type(cutoffs) == list
    assert min(cutoffs) > 0, "All cutoffs must be positive integers."
    assert recommendations.num_columns() >= 2
    user_id = recommendations.column_names()[0]
    item_id = recommendations.column_names()[1]

    assert observed_user_items.num_rows() > 0, \
           "Evaluating precision and recall requires a non-empty " + \
           "observed_user_items."
    assert user_id in observed_user_items.column_names(), \
            "User column required in observed_user_items."
    assert item_id in observed_user_items.column_names(), \
            "Item column required in observed_user_items."
    assert observed_user_items[user_id].dtype() == \
           recommendations[user_id].dtype(), \
           "The user column in the two provided SFrames must have the same type."
    assert observed_user_items[item_id].dtype() == \
           recommendations[item_id].dtype(), \
           "The user column in the two provided SFrames must have the same type."

    cutoffs = _array.array('f', cutoffs)

    opts = {'data': observed_user_items,
            'recommendations': recommendations,
            'cutoffs': cutoffs}

    response = _graphlab.toolkits._main.run('evaluation_precision_recall_by_user', opts)
    sf = _SFrame(None, _proxy=response['pr'])
    return sf.sort([user_id, 'cutoff'])


def random_split_by_user(dataset,
                         user_id='user_id',
                         item_id='item_id',
                         max_num_users=1000,
                         item_test_proportion=.2,
                         random_seed=0):
    """
    Create a recommender-friendly train-test split of the provided data set.

    The test dataset is generated by first choosing `max_num_users` out of the
    total number of users in `dataset`. Then, for each of the chosen test users,
    a portion of the user's items (determined by `item_test_proportion`) is
    randomly chosen to be included in the test set. This split allows the
    training data to retain enough information about the users in the testset,
    so that adequate recommendations can be made. The total number of users
    in the test set may be fewer than `max_num_users` if a user was chosen for
    the test set but none of their items are selected.

    Parameters
    ----------
    dataset : SFrame
        An SFrame containing (user, item) pairs.

    user_id : str, optional
        The name of the column in ``dataset`` that contains user ids.

    item_id : str, optional
        The name of the column in ``dataset`` that contains item ids.

    max_num_users : int, optional
        The maximum number of users to use to construct the test set. If
        set to 'None', then use all available users.

    item_test_proportion : float, optional
        The desired probability that a test user's item will be chosen
        for the test set.

    random_seed : int, optional
        The random seed to use for randomization.

    Returns
    -------
    train, test : SFrame
        A tuple with two datasets to be used for training and testing.

    Examples
    --------
    >>> import graphlab as gl
    >>> sf = gl.SFrame('s3://GraphLab-Datasets/audioscrobbler')
    >>> train, test = gl.recommender.util.random_split_by_user(sf, max_num_users=100)
    """

    _mt._get_metric_tracker().track('toolkit.recsys.random_split_by_user')

    assert user_id in dataset.column_names(), \
        'Provided user column "{0}" not found in data set.'.format(user_id)
    assert item_id in dataset.column_names(), \
        'Provided item column "{0}" not found in data set.'.format(item_id)

    if max_num_users == 'all':
        max_num_users = None

    opts = {'dataset': dataset,
            'user_id': user_id,
            'item_id': item_id,
            'max_num_users': max_num_users,
            'item_test_proportion': item_test_proportion,
            'random_seed': random_seed}

    response = _graphlab.toolkits._main.run('recsys_train_test_split', opts)
    train = _SFrame(None, _proxy=response['train'])
    test = _SFrame(None, _proxy=response['test'])
    return train, test


class _Recommender(_Model):
    """
    The abstract class for GraphLab recommender system models. This class
    defines methods common to all recommender system models, but leaves unique
    model details to separate model classes.
    """
    @classmethod
    def _get_queryable_methods(cls):
        '''Returns a list of method names that are queryable through Predictive
        Service'''
        return {'predict': {'dataset': 'sframe', 'new_observation_data': 'sframe',
                             'new_user_data': 'sframe', 'new_item_data': 'sframe'
                             },
                'recommend': {'items': 'sarray', 'new_observation_data': 'sframe',
                              'new_user_data': 'sframe', 'new_item_data': 'sframe',
                              'exclude': 'sframe'}
                }

    def list_fields(self):
        """
        Get the current settings of the model. The keys depend on the type of
        model.

        Returns
        -------
        out : list
            A list of fields that can be queried using the ``get`` method.
        """

        opts = {'model': self.__proxy__}
        response = _graphlab.toolkits._main.run('recsys_list_fields', opts)
        return [s for s in response['value'] if not s.startswith("_")]

    def get(self, field):
        """
        Get the value of a particular field.

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out
            The current value of the requested field.

        Examples
        --------
        >>> data = graphlab.SFrame({'user_id': ["0", "0", "0", "1", "1", "2", "2", "2"],
        ...                         'item_id': ["a", "b", "c", "a", "b", "b", "c", "d"],
        ...                         'rating': [1, 3, 2, 5, 4, 1, 4, 3]})
        >>> from graphlab.recommender
        >>> m = factorization_recommender.create(data, "user_id", "item_id", "rating")
        >>> d = m.get("coefficients")
        >>> U1 = d['user_id']
        >>> U2 = d['movie_id']
        """

        opts = {'model': self.__proxy__, 'field': field}
        response = _graphlab.toolkits._main.run('recsys_get_value', opts)

        def type_tr(v):
            if type(v) is dict:
                return dict( (k, type_tr(v)) for k, v in v.iteritems())

            elif isinstance(v, _graphlab.cython.cy_sframe.UnitySFrameProxy):
                return _SFrame(None, _proxy=v)

            else:
                return v

        return type_tr(response["value"])


    def summary(self):
        """
        Prints a summary of statistics about the model, the data
        used to train the model, and the training performance.

        """
        _mt._get_metric_tracker().track('toolkit.recsys.summary')
        print self.__repr__()

    def __str__(self):
        """
        Returns the type of model.

        Returns
        -------
        out : string
            The type of model.
        """
        return self.__class__.__name__

    def __repr__(self):
        """
        Returns a string description of the model, including (where relevant)
        the schema of the training data, description of the training data,
        training statistics, and model hyperparameters.

        Returns
        -------
        out : string
            A description of the model.
        """

        stats = self.list_fields()
        options = self.get_current_options()

        # Print model type and data schema
        ret = self.__class__.__name__ + '\n'

        # If the model is untrained, there will not be an elapsed_time
        is_trained = 'training_time' in stats

        # TODO: Refactor into functions
        if is_trained:

            ret += "\nSchema\n"
            for k in ["user_id", "item_id", "target"]:
                if k in options.keys():
                    if options[k] not in ['', 'not_specified'] and options[k] is not None:
                        ret += "  %-14s %s\n" % \
                            (k.replace('_', ' ') + ':', options[k])
                    del options[k]

            # Statistics about the training and validation set
            ret += "\nStatistics\n"
            ks = ["num_observations",
                  "num_users",
                  "num_items"]
            if all([k in stats for k in ks]):
                ret += "  %-17s %11d obs %11d users %11d items\n" % \
                    ("Training set:", self.get(ks[0]), self.get(ks[1]), self.get(ks[2]))

            if 'user_side_data_column_names' in stats:
                if len(self.get('user_side_data_column_names')) > 0:
                    ret += "  user side data:\n"
                colnames = self.get('user_side_data_column_names')
                coltypes = self.get('user_side_data_column_types')
                assert len(colnames) == len(coltypes), \
                    "Error importing user side data: mismatch between the \
                     number of column names and column types."

                for i in range(len(colnames)):
                    ret += '    {0} : {1}\n'.format(colnames[i], coltypes[i])

            if 'item_side_data_column_names' in stats:
                if len(self.get('item_side_data_column_names')) > 0:
                    ret += "  item side data:\n"
                colnames = self.get('item_side_data_column_names')
                coltypes = self.get('item_side_data_column_types')
                assert len(colnames) == len(coltypes), \
                    "Error importing item side data: mismatch between the \
                     number of column names and column types."

                for i in range(len(colnames)):
                    ret += '    {0} : {1}\n'.format(colnames[i], coltypes[i])


            # Training summary
            ks = ["training_time",
                  "random_seed"]
            if any([k in stats]):
                ret += "\nTraining summary\n"
            if 'elapsed_time' in stats:
                ret += "  %-20s %fs\n" % ("training time:", self.get("elapsed_time"))
            if 'data_load_elapsed_time' in stats:
                ret += "  %-20s %fs\n" % ("data load time:", self.get("data_load_elapsed_time"))

            if "validation_metrics_elapsed_time" in stats:
                ret += "  %-20s %fs\n" % \
                           ("metrics time:", self.get("validation_metrics_elapsed_time"))

            # If available, print performance statistics
            for k in stats:
                if 'rmse' in k or 'precision' in k or 'recall' in k:
                    ret += '  %-20s %-10s\n' % (k + ':', self.get(k))

        else:
            ret += '\nThis model has yet to be trained.\n'

        # Remove any options that should not be shown under "Settings"
        to_ignore = ['random_seed',
                     'user_id',
                     'item_id',
                     'target']
        for k in to_ignore:
            if k in options:
                del options[k]

        # Print remaining hyperparameters
        # TODO: Get max width of keys and use it for fixed width formatting.
        if len(options) > 0:
            ret += "\nSettings\n"
            for k, v in options.iteritems():
                ret += "  %-22s %-30s\n" % (k.replace('_', ' ') + ':', str(v))

        return ret

    def get_current_options(self):
        """
        A dictionary describing all the parameters of the given model
        and their current setting.
        """
        opts = {'model': self.__proxy__}
        response = _graphlab.toolkits._main.run('recsys_get_current_options', opts)
        return response

    def _set_current_options(self, options):
        """
        Set current options for a model.

        Parameters
        ----------
        options : dict
            A dictionary of the desired option settings. The key should be the name
            of the option and each value is the desired value of the option.
            The possible options are all those returne dy get_default_options().
        """
        opts = self.get_current_options()
        opts.update(options)
        opts['model'] = self.__proxy__
        response = _graphlab.toolkits._main.run('recsys_set_current_options', opts)
        return response

    def predict(self, dataset,
                new_observation_data=None, new_user_data=None, new_item_data=None):
        """
        Return a score prediction for the user ids and item ids in the provided
        data set.

        Parameters
        ----------
        dataset : SFrame
            Dataset in the same form used for training.

        new_observation_data : SFrame, optional
            ``new_observation_data`` gives additional observation data
            to the model, which may be used by the models to improve
            score accuracy.  Must be in the same format as the
            observation data passed to ``create``.  How this data is
            used varies by model.

        new_user_data : SFrame, optional
            ``new_user_data`` may give additional user data to the
            model.  If present, scoring is done with reference to this
            new information.  If there is any overlap with the side
            information present at training time, then this new side
            data is preferred.  Must be in the same format as the user
            data passed to ``create``.

        new_item_data : SFrame, optional
            ``new_item_data`` may give additional item data to the
            model.  If present, scoring is done with reference to this
            new information.  If there is any overlap with the side
            information present at training time, then this new side
            data is preferred.  Must be in the same format as the item
            data passed to ``create``.

        Returns
        -------
        out : SArray
            An SArray with predicted scores for each given observation
            predicted by the model.

        See Also
        --------
        recommend, evaluate
        """
        _mt._get_metric_tracker().track('toolkit.recsys.predict')

        if new_observation_data is None:
            new_observation_data = _SFrame()
        if new_user_data is None:
            new_user_data = _SFrame()
        if new_item_data is None:
            new_item_data = _SFrame()

        def check_type(arg, arg_name, required_type, allowed_types):
            if not isinstance(arg, required_type):
                raise TypeError("Parameter " + arg_name + " must be of type(s) "
                                + (", ".join(allowed_types))
                                + "; Type '" + str(type(arg)) + "' not recognized.")

        check_type(new_observation_data, "new_observation_data", _SFrame, ["SFrame"])
        check_type(new_user_data, "new_user_data", _SFrame, ["SFrame"])
        check_type(new_item_data, "new_item_data", _SFrame, ["SFrame"])

        # Get metadata from C++ object
        opts = {'data_to_predict': dataset,
                'model': self.__proxy__,
               'new_data': new_observation_data,
               'new_user_data': new_user_data,
               'new_item_data': new_item_data
                }

        # Call the C++ function for recommender_model
        response = _graphlab.toolkits._main.run('recsys_predict', opts)
        result = _SFrame(None, _proxy=response['data'])
        return result['prediction']

    def get_similar_items(self, items=None, k=10):
        """
        Get the k most similar items for each item in items.

        Each type of recommender has its own model for the similarity between
        items. For example, the item_similarity_recommender will return
        the most similar items according to the user-chosen similarity;
        the factorization_recommender will return the nearest items based
        on the Euclidean distance between latent item factors.

        Parameters
        ----------
        items : SArray or list; optional
            An :class:`~graphlab.SArray` or list of item ids for which to get
            similar items. If 'None', then return the `k` most similar items for
            all items in the training set.

        k : int, optional
            The number of similar items for each item.

        Returns
        -------
        out : SFrame
            A SFrame with the top ranked similar items for each item. The
            columns `item`, 'similar', 'score' and 'rank', where
            `item` matches the item column name specified at training time.
            The 'rank' is between 1 and `k` and 'score' gives the similarity
            score of that item. The value of the score depends on the method
            used for computing item similarities.

        Examples
        --------

        >>> sf = graphlab.SFrame({'user_id': ["0", "0", "0", "1", "1", "2", "2", "2"],
                                  'item_id': ["a", "b", "c", "a", "b", "b", "c", "d"]})
        >>> m = graphlab.item_similarity_recommender.create(sf)
        >>> nn = m.get_similar_items()
        """

        _mt._get_metric_tracker().track('toolkit.recsys.get_similar_items')

        if items is None:
            items = _SArray()

        assert type(k) == int

        if isinstance(items, list):
            items = _SArray(items)

        def check_type(arg, arg_name, required_type, allowed_types):
            if not isinstance(arg, required_type):
                raise TypeError("Parameter " + arg_name + " must be of type(s) "
                                + (", ".join(allowed_types) )
                                + "; Type '" + str(type(arg)) + "' not recognized.")

        check_type(items, "items", _SArray, ["SArray", "list"])

        opt = {'model': self.__proxy__,
               'items': items,
               'k': k}

        response = _graphlab.toolkits._main.run('recsys_get_similar_items', opt)

        neighbors = _SFrame(None, _proxy=response['data'])

        return neighbors


    def recommend(self, users=None, k=10, exclude=None, items=None,
                  new_observation_data=None, new_user_data=None, new_item_data=None,
                  exclude_known=True,
                  verbose=True):
        """
        Recommends the k-highest scored items for each user in users.

        Parameters
        ----------
        users :  SArray or list; optional
            An :class:`~graphlab.SArray` (or list) of users for which
            to make recommendations.  If 'None', then recommend(...)
            generates predictions for all users in the training set.
            tHE type of the SArray or list must be the same as the type
            of the user_id column in the training data.

        k : int, optional
            The number of recommendations to generate for each user.

        items : SArray, optional
            To restrict the set of items under consideration, the user
            may pass in an SArray of items from which the recommended
            items will be chosen.  This allows one to choose only
            items that are, for instance, in a particular genre or
            category.  The default is that all items are under
            consideration.

        new_observation_data : SFrame, optional
            ``new_observation_data`` gives additional observation data
            to the model, which may be used by the models to improve
            score and recommendation accuracy.  Must be in the same
            format as the observation data passed to ``create``.  How
            this data is used varies by model.

        new_user_data : SFrame, optional
            ``new_user_data`` may give additional user data to the
            model.  If present, scoring is done with reference to this
            new information.  If there is any overlap with the side
            information present at training time, then this new side
            data is preferred.  Must be in the same format as the user
            data passed to ``create``.

        new_item_data : SFrame, optional
            ``new_item_data`` may give additional item data to the
            model.  If present, scoring is done with reference to this
            new information.  If there is any overlap with the side
            information present at training time, then this new side
            data is preferred.  Must be in the same format as the item
            data passed to ``create``.

        exclude : SFrame, optional
            An :class:`~graphlab.SFrame` of user / item pairs.  The
            column names must be equal to the user and item columns of
            the main data, and it provides the model with user/item
            pairs to exclude from the recommendations.  These
            user-item-pairs are always excluded from the predictions,
            even if exclude_known is False.

        exclude_known : bool, optional
            By default, all user-item interactions previously seen in
            the training data, or in any new data provided using
            new_observation_data=..., are excluded from the
            recommendations.  Passing in ``exclude_known = False``
            overrides this behavior.

        verbose : bool, optional
            If True, print the progress of generating recommendation.

        Returns
        -------
        out : SFrame
            A SFrame with the top ranked items for each user. The
            columns are: ``user_id``, ``item_id``, *score*,
            and *rank*, where ``user_id`` and ``item_id``
            match the user and item column names specified at training
            time.  The rank column is between 1 and ``k`` and gives
            the relative score of that item.  The value of score
            depends on the method used for recommendations.

        See Also
        --------
        predict
        evaluate
        """
        _mt._get_metric_tracker().track('toolkit.recsys.recommend')
        assert type(k) == int

        if users is None:
            users = _SArray()
        if exclude is None:
            exclude = _SFrame()
        if items is None:
            items = _SArray()
        if new_observation_data is None:
            new_observation_data = _SFrame()
        if new_user_data is None:
            new_user_data = _SFrame()
        if new_item_data is None:
            new_item_data = _SFrame()

        if isinstance(users, list) or (_HAS_NUMPY and isinstance(users, _numpy.ndarray)):
            users = _SArray(users)
        if isinstance(items, list) or (_HAS_NUMPY and isinstance(items, _numpy.ndarray)):
            items = _SArray(items)

        def check_type(arg, arg_name, required_type, allowed_types):
            if not isinstance(arg, required_type):
                raise TypeError("Parameter " + arg_name + " must be of type(s) "
                                + (", ".join(allowed_types))
                                + "; Type '" + str(type(arg)) + "' not recognized.")

        check_type(users, "users", _SArray, ["SArray", "list", "numpy.ndarray"])
        check_type(exclude, "exclude", _SFrame, ["SFrame"])
        check_type(items, "items", _SArray, ["SArray", "list", "numpy.ndarray"])
        check_type(new_observation_data, "new_observation_data", _SFrame, ["SFrame"])
        check_type(new_user_data, "new_user_data", _SFrame, ["SFrame"])
        check_type(new_item_data, "new_item_data", _SFrame, ["SFrame"])

        opt = {'model': self.__proxy__,
               'users': users,
               'top_k': k,
               'exclude': exclude,
               'items': items,
               'new_data': new_observation_data,
               'new_user_data': new_user_data,
               'new_item_data': new_item_data,
               'exclude_known': exclude_known}
        response = _graphlab.toolkits._main.run('recsys_recommend', opt, verbose=verbose)
        recs = _SFrame(None, _proxy=response['data'])
        return recs

    def _training_stats(self):
        """
        Get information about model creation, e.g. time elapsed during
        model fitting, data loading, and more.

        Note: This method will be *deprecated* soon. Please use m.summary()
        instead.

        Returns
        -------
        out : dict
            Statistics about model training, e.g. runtime.

        """
        _logging.warning("This method will be deprecated soon. Please use m.summary().")
        _mt._get_metric_tracker().track('toolkit.recsys._training_stats')

        opts = {'model': self.__proxy__}
        response = _graphlab.toolkits._main.run("recsys_get_train_stats", opts)
        return response

    def evaluate_precision_recall(self, dataset, cutoffs=[5, 10, 15],
                                  skip_set=None, exclude_known=True,
                                  verbose=True, **kwargs):
        """
        Compute a model's precision and recall scores for a particular dataset.

        Parameters
        ----------
        dataset : SFrame
            An SFrame in the same format as the one used during training.
            This will be compared to the model's recommendations, which exclude
            the (user, item) pairs seen at training time.

        cutoffs : list, optional
            A list of cutoff values for which one wants to evaluate precision
            and recall, i.e. the value of k in "precision at k".

        skip_set : SFrame, optional
            Passed to :meth:`recommend` as ``exclude``.

        exclude_known : bool, optional
            Passed to :meth:`recommend` as ``exclude_known``. If True, exclude
            training item from recommendation.

        verbose : bool, optional
            Enables verbose output. Default is verbose.

        **kwargs
            Additional keyword arguments are passed to the recommend
            function, whose returned recommendations are used for evaluating
            precision and recall of the model.

        Returns
        -------
        out : SFrame
            Contains the precision and recall at each cutoff value and each
            user in ``dataset``.

        Examples
        --------

        >>> import graphlab as gl
        >>> sf = gl.SFrame('s3://GraphLab-Datasets/audioscrobbler')
        >>> train, test = gl.recommender.util.random_split_by_user(sf)
        >>> m = gl.recommender.create(train)
        >>> m.evaluate_precision_recall(test)

        See Also
        --------
        graphlab.recommender.util.precision_recall_by_user
        """

        _mt._get_metric_tracker().track('toolkit.recsys.evaluate_precision_recall')

        user_column = self.get('user_id')
        item_column = self.get('item_id')
        assert user_column in dataset.column_names() and \
               item_column in dataset.column_names(), \
            'Provided data set must have a column pertaining to user ids and \
             item ids, similar to what we had during training.'

        users = dataset[self.get('user_id')].unique()

        dataset = dataset[[self.get('user_id'), self.get('item_id')]]

        recs = self.recommend(users=users, k=max(cutoffs), exclude=skip_set,
                              exclude_known=exclude_known,
                              verbose=verbose,
                              **kwargs)

        precision_recall_by_user = _graphlab.recommender.util.precision_recall_by_user(dataset, recs, cutoffs)

        ret = {'precision_recall_by_user': precision_recall_by_user}

        pr_agg = precision_recall_by_user.groupby(
             'cutoff',
             operations={'precision' : _Aggregate.MEAN('precision'),
                         'recall'    : _Aggregate.MEAN('recall')})

        pr_agg = pr_agg[['cutoff', 'precision', 'recall']]
        ret["precision_recall_overall"] = pr_agg.sort("cutoff")

        return ret


    def evaluate_rmse(self, dataset, target):
        """
        Evaluate the prediction error for each user-item pair in the given data
        set.

        Parameters
        ----------
        dataset : SFrame
            An SFrame in the same format as the one used during training.

        target : str
            The name of the target rating column in `dataset`.

        Returns
        -------
        out : dict
            A dictionary with three items: 'rmse_by_user' and 'rmse_by_item',
            which are SFrames containing the average rmse for each user and
            item, respectively; and 'rmse_overall', which is a float.

        Examples
        --------
        >>> import graphlab as gl
        >>> sf = gl.SFrame('s3://GraphLab-Datasets/audioscrobbler')
        >>> train, test = gl.recommender.util.random_split_by_user(sf)
        >>> m = gl.recommender.create(train, target='target')
        >>> m.evaluate_rmse(test, target='target')

        See Also
        --------
        graphlab.evaluation.rmse
        """

        _mt._get_metric_tracker().track('toolkit.recsys.evaluate_rmse')

        assert target in dataset.column_names(), \
               'Provided dataset must contain a target column with the same \
                name as the target used during training.'
        y = dataset[target]
        yhat = self.predict(dataset)
        user_column = self.get('user_id')
        item_column = self.get('item_id')
        assert user_column in dataset.column_names() and \
               item_column in dataset.column_names(), \
            'Provided data set must have a column pertaining to user ids and \
             item ids, similar to what we had during training.'

        result = dataset[[user_column, item_column]]
        result['sq_error'] = (y - yhat) * (y - yhat)
        rmse_by_user = result.groupby(user_column,
                {'rmse':_graphlab.aggregate.AVG('sq_error'),
                 'count':_graphlab.aggregate.COUNT})
        rmse_by_user['rmse'] = rmse_by_user['rmse'].apply(lambda x: x**.5)
        rmse_by_item = result.groupby(item_column,
                {'rmse':_graphlab.aggregate.AVG('sq_error'),
                 'count':_graphlab.aggregate.COUNT})
        rmse_by_item['rmse'] = rmse_by_item['rmse'].apply(lambda x: x**.5)
        overall_rmse = result['sq_error'].mean() ** .5

        return {'rmse_by_user': rmse_by_user,
                'rmse_by_item': rmse_by_item,
                'rmse_overall': overall_rmse}

    def evaluate(self, dataset, metric='auto',
                 exclude_known_for_precision_recall=True,
                 target=None,
                 verbose=True, **kwargs):
        r"""
        Evaluate the model's ability to make rating predictions or
        recommendations.

        If the model is trained to predict a particular target, the
        default metric used for model comparison is root-mean-squared error
        (RMSE). Suppose :math:`y` and :math:`\widehat{y}` are vectors of length
        :math:`N`, where :math:`y` contains the actual ratings and
        :math:`\widehat{y}` the predicted ratings. Then the RMSE is defined as

        .. math::

            RMSE = \sqrt{\frac{1}{N} \sum_{i=1}^N (\widehat{y}_i - y_i)^2} .

        If the model was not trained on a target column, the default metrics for
        model comparison are precision and recall. Let
        :math:`p_k` be a vector of the :math:`k` highest ranked recommendations
        for a particular user, and let :math:`a` be the set of items for that
        user in the groundtruth `dataset`. The "precision at cutoff k" is
        defined as

        .. math:: P(k) = \frac{ | a \cap p_k | }{k}

        while "recall at cutoff k" is defined as

        .. math:: R(k) = \frac{ | a \cap p_k | }{|a|}

        Parameters
        ----------
        dataset : SFrame
            An SFrame that is in the same format as provided for training.

        metric : str, {'auto', 'rmse', 'precision_recall'}, optional
            Metric to use for evaluation. The default automatically chooses
            'rmse' for models trained with a `target`, and 'precision_recall'
            otherwise.

        exclude_known_for_precision_recall : bool, optional
            A useful option for evaluating precision-recall. Recommender models
            have the option to exclude items seen in the training data from the
            final recommendation list. Set this option to True when evaluating
            on test data, and False when evaluating precision-recall on training
            data.

        target : str, optional
            The name of the target column for evaluating rmse. If the model is
            trained with a target column, the default is to using the same
            column. If the model is trained without a target column and `metric`
            is set to 'rmse', this option must provided by user.

        verbose : bool, optional
            Enables verbose output. Default is verbose.

        **kwargs
            When `metric` is set to 'precision_recall', these parameters
            are passed on to :meth:`evaluate_precision_recall`.

        Returns
        -------
        out : SFrame or dict
            Results from the model evaluation procedure. If the model is trained
            on a target (i.e. RMSE is the evaluation criterion), a dictionary
            with three items is returned: items *rmse_by_user* and
            *rmse_by_item* are SFrames with per-user and per-item RMSE, while
            *rmse_overall* is the overall RMSE (a float). If the model is
            trained without a target (i.e. precision and recall are the
            evaluation criteria) an :py:class:`~graphlab.SFrame` is returned
            with both of these metrics for each user at several cutoff values.

        Examples
        --------
        >>> import graphlab as gl
        >>> sf = gl.SFrame('s3://GraphLab-Datasets/audioscrobbler')
        >>> train, test = gl.recommender.util.random_split_by_user(sf)
        >>> m = gl.recommender.create(train, target='target')
        >>> m.evaluate(test)

        See Also
        --------
        evaluate_precision_recall, evaluate_rmse, precision_recall_by_user
        """

        _mt._get_metric_tracker().track('toolkit.recsys.evaluate')

        ret = {}

        # If the model does not have a target column, compute prec-recall.
        if metric in ['precision_recall', 'auto']:
            results = self.evaluate_precision_recall(dataset,
                                                     exclude_known=exclude_known_for_precision_recall,
                                                     verbose=verbose,
                                                     **kwargs)
            ret.update(results)
            if verbose:
                print "\nPrecision and recall summary statistics by cutoff"
                print results['precision_recall_by_user'].groupby('cutoff', \
                        {'mean_precision': _graphlab.aggregate.AVG('precision'),
                         'mean_recall': _graphlab.aggregate.AVG('recall')}).topk('cutoff', reverse=True)
        if metric in ['rmse', 'auto']:
            if target is None:
                target = self.get('target')
            if target is None or target == "":
                _logging.warning("Model trained without a target. Skipping RMSE computation.")
            else:
                results = self.evaluate_rmse(dataset, target)
                ret.update(results)

                if verbose:
                    print "\nOverall RMSE: ", results['rmse_overall']
                    print "\nPer User RMSE (best)"
                    print results['rmse_by_user'].topk('rmse', 1, reverse=True)
                    print "\nPer User RMSE (worst)"
                    print results['rmse_by_user'].topk('rmse', 1)
                    print "\nPer Item RMSE (best)"
                    print results['rmse_by_item'].topk('rmse', 1, reverse=True)
                    print "\nPer Item RMSE (worst)"
                    print results['rmse_by_item'].topk('rmse', 1)
        if metric not in ['rmse', 'precision_recall', 'auto']:
            raise ValueError('Unknown evaluation metric %s, supported metrics are [\"rmse\", \"precision_recall\"]' % metric)

        return ret
