import graphlab.canvas
import graphlab.canvas.views as views
import graphlab.canvas.views.base
import graphlab.canvas.views.task
import graphlab.canvas.views.sframe
import graphlab.deploy
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.deploy._artifact import Pipeline as _Pipeline

import os as _os
import __builtin__

class JobView(graphlab.canvas.views.base.BaseView):

    def __init__(self, obj, params=None):
        super(JobView, self).__init__(obj, params)
        self.register_handler('get', 'logs', self.__get_logs)
        self.register_handler('get', 'get_results', self.__get_results_handler)
        self.__child_views = {}
        self.__cache_logs = None
        self.__job = None
        self.objectType = 'Job' # override when job loads

    def _set_job(self, job=None):
        if job is not None:
            self.__job = job

    def __load_job(self):
        if not self.__job:
            self.__job = graphlab.deploy.jobs[self.obj]
        return self.__job

    def get_metadata(self):
        job = self.__load_job()
        # override job type
        self.objectType = __builtin__.type(job).__name__
        tasks = []
        # get list of tasks from Tasks or Pipeline
        if job.tasks is not None and len(job.tasks) > 0:
            task = job.tasks[0]
            if isinstance(task, list):
                tasks = list([stage[0].__to_dict__() for stage in job.tasks]) 
            elif isinstance(task, _Pipeline):
                for step in task.steps:
                    tasks.extend(list([task_step.__to_dict__() for task_step in step]))
                        
        # weakly check if there is possible results
        has_results = False 
        if hasattr(job, "get_results") or 'results' in self.__child_views:
            has_results = True
            
        return {
            'jobname' : [job.name],
            'tasks' : tasks,
            'env' : job.environment.__to_dict__(),
            'status' : job.get_status(),
            'metrics' : job.get_metrics(),
            'taskstatus' : job._get_task_status(),
            'jobtype' : __builtin__.type(job).__name__,
            'starttime' : job._get_start_time(),
            'endtime' : job._get_end_time(),
            'has_results' : has_results
        }

    def __get_results_handler(self, url, handler):
        job = self.__load_job()
        has_results = False
        # double check if there are results
        if hasattr(job, "get_results"):
            if 'results' not in self.__child_views:
                # load results
                results = job.get_results()
                if results is not None and type(results) == _SFrame:
                    self.__child_views['results'] = graphlab.canvas.views.sframe.SFrameView(results)
                    has_results = True
            else:
                has_results = True # already a child view

        handler.write({"has_results":has_results})
                
    def get_staticdata(self):
        return self.get_metadata()

    # return logs for the current job
    def __get_logs(self, url, handler):
        log_data = self._get_logs();
        handler.write({'logfile': log_data})

    def _get_logs(self):
        job = self.__load_job()
        log_file_path = job._get_log_file_path()
        log_data = {'url': '', 'data': ''}
        if log_file_path is None:
            return log_data 
        
        if isinstance(job.environment, graphlab.deploy.environment.Local) or \
           isinstance(job.environment, graphlab.deploy.environment.LocalAsync):
            if self.__cache_logs is None:
                if not _os.path.isfile(log_file_path) and self.__cache_logs is None:
                    log_data['data'] = "Log file path does not exist"
                    return log_data
                
                log_data['url'] = "file://" + log_file_path
                # load max of 1mb log file to python
                if _os.path.getsize(log_file_path) > 1048576L:
                    log_data['data'] = "Log file too large to be displayed. Please use the link to download the log file directly"
                    return log_data
                
                data = self.__load_local_log_file(log_file_path)
                log_data['data'] = data
                status = job.get_status()
                if status == "Completed" or status == "Failed":
                    self.__cache_logs = data
            else:
                log_data['url'] = "file://" + log_file_path
                log_data['data'] = self.__cache_logs
        elif isinstance(job.environment, graphlab.deploy.environment.EC2):
            if self.__cache_logs is None:
                if not isinstance(log_file_path, tuple):
                    log_data['data'] = log_file_path
                    return log_data
            
                log_data['url'] = log_file_path[0]
                # load max of 1mb log file to python
                if log_file_path[1] > 1048576L:
                    log_data['data'] = "Log file too large to be displayed. Please use the link to download the log file directly"
                    return log_data
                    
                data = self.__load_s3_log_file(log_file_path[0])
                log_data['data'] = data
                status = job.get_status()
                if status == "Completed" or status == "Failed":
                    self.__cache_logs = data
            else:
                log_data['url'] = log_file_path[0]
                log_data['data'] = self.__cache_logs
        return log_data 

    def __load_local_log_file(self, log_file_path):
        data = ""
        try:
            with open(log_file_path, 'r') as f:
                data = f.read() 
            f.close()
        except IOError:
            data = "Unable to read file from %s " % log_file_path
        return data

    def __load_s3_log_file(self, log_file_path_s3):
        import urllib2
        data = ""
        try:
            f = urllib2.urlopen(log_file_path_s3)
            data = f.read()
            f.close()
        except URLError:
            data = "Unable to load s3 log file from %s " % log_file_path_s3
        return data

    def child_views(self):
        return self.__child_views

    def get_js_file(self):
        return 'job'

