import graphlab.canvas
import graphlab.canvas.views.base
import graphlab.deploy
import json
import datetime
import calendar

class PredictiveService(graphlab.canvas.views.base.BaseView):
    def __init__(self, obj, params=None):
        super(PredictiveService, self).__init__(obj, params)
        self.objectType = 'PredictiveService'
        self.register_handler('post', 'query', self.__query_handler)
        self.register_handler('get', 'get_ps_status', self.__get_ps_status)
        self.register_handler('get', 'get_ps_metrics', self.__get_ps_metrics)
        self.__cache_ps = None

    def get_metadata(self):
        ps_info = self.__get_ps_info()
        data = {
            'ps_endpoint' : self.params['endpoint'],
            'ps_name' : self.obj,
            'ps_info' : ps_info,
            'pslistName' : 'Predictive Service'
        }
        return data

    def __get_ps_info(self):
        name = self.obj 
        ps = self.__load_ps_object()
        # return to session page if loaded ps none
        if ps is None:
            return {'deleted': True} 

        ps_info = {}
        
        # get deployed and pending predictive objects
        ps_info['deployed'] = ps.deployed_predictive_objects
        ps_info['pending'] = ps.pending_changes

        # get metadata on the predictive services
        ps_info['s3_endpoint'] = self.params['endpoint']
        ps_info['name'] = name
        ps_info['description'] = ps._description if ps._description else ""
        ps_info['api_key'] = ps._api_key if ps._api_key else ""

        # get query endpoint (LB dns name)
        try:
            ps_info['lb_dns_name'] = ps._environment.load_balancer_dns_name
        except Exception as e:
            ps_info['lb_dns_name'] = "N/A"

        return ps_info

    def __get_ps_status(self, url, handler):
        ps = self.__load_ps_object()
        
        # get status list of all instances
        status_info = {}
        try:
            status_list = ps._get_status()
            for status in status_list:
                instance_id = str(status['id'])
                description = str(status['reason'])
                dns = str(status['dns_name'])
                state = str(status['state'])
                status_info[instance_id] = {'description' : description, 'state' : state, 'dns': dns}
        except Exception as e:
            status_info = {}

        # get metrics url
        metrics_url = ''
        try:
            raw_text = ps.get_metrics_url()
            parsed = json.loads(str(raw_text))
            if isinstance(parsed, dict):
                if 'url' in parsed:
                    metrics_url = parsed['url']
        except Exception as e:
            metrics_url = ''

        handler.write({'ps_status': status_info, 'ps_metrics_url': metrics_url})
    
    def __get_metrics_data(self, sf, x, y, time_frame):
        # get x, y values of the metrics
        xList = list(sf[x])
        yList = list(sf[y])
        valArray = []
        for i in range(0, len(xList)):
            valArray.append({'x': xList[i], 'y': yList[i]})

        # get extrema
        xMin = calendar.timegm(time_frame['start_time'].timetuple())
        xMax = calendar.timegm(time_frame['end_time'].timetuple())
        yMin = float(sf[y].min())
        yMax = float(sf[y].max())
        if yMin > 0:
            yMin = 0
        return [valArray, {'minX': xMin, 'maxX': xMax, 'minY': yMin, 'maxY': yMax}]

    def __get_ps_metrics(self, url, handler):
        ps = self.__load_ps_object()
        # get metrics
        (time_frame, num_requests, latency, hosts, cache) = ps.get_metrics()
        # initialize empty data array 
        num_requests_data = [[], {'minX':0, 'maxX':0, 'minY': 0, 'maxY':0}] 
        latency_data = [[], {'minX':0, 'maxX':0, 'minY': 0, 'maxY':0}] 
        hosts_data = [[], {'minX':0, 'maxX':0, 'minY': 0, 'maxY':0}] 
        cache_data = [[], {'minX':0, 'maxX':0, 'minY': 0, 'maxY':0}] 
        # add column containing int value of datetime
        if num_requests is not None:
            num_requests = num_requests.add_column(num_requests['num_requests.Timestamp'].astype(int), name='num_requests.time')
            num_requests_data = self.__get_metrics_data(num_requests, 'num_requests.time', 'num_requests.Sum', time_frame)
        if latency is not None:
            latency = latency.add_column(latency['latency.Timestamp'].astype(int), name='latency.time')
            latency_data = self.__get_metrics_data(latency, 'latency.time', 'latency.Average', time_frame)
        if hosts is not None:
            hosts = hosts.add_column(hosts['num_healthy_hosts.Timestamp'].astype(int), name='num_healthy_hosts.time')
            hosts_data = self.__get_metrics_data(hosts, 'num_healthy_hosts.time', 'num_healthy_hosts.Average', time_frame)
        if cache is not None:
            cache = cache.add_column(cache['num_keys_in_cache.Timestamp'].astype(int), name='num_keys_in_cache.time')
            cache_data = self.__get_metrics_data(cache, 'num_keys_in_cache.time', 'num_keys_in_cache.Maximum', time_frame)

        handler.write({"num_requests_data":num_requests_data,
                        "latency_data" : latency_data,
                        "hosts_data" : hosts_data,
                        "cache_data" : cache_data})

            
    def __load_ps_object(self):
        """
        Loads the predictive object by name, from the current session.
        """
        return graphlab.deploy.predictive_services[self.obj]

    def __query_handler(self, url, handler):
        # obtain the json query parameters from canvas
        model = handler.get_argument('model').encode('utf-8')
        data = handler.get_argument('data')
        data_dict = json.loads(data)

        # loads and query the ps with the query parameters
        ps = self.__load_ps_object()
        try:
            resp = ps.query(model, data_dict)
        except BaseException as e:
            resp = "Error when querying the Predictive Service: \n%s" % e.message

        # write back response
        handler.write({"results": resp})

    def child_views(self):
        return {}

    def get_js_file(self):
        return 'predservice'

    def get_temporary_name(self):
        return (str(obj),)
