from ConfigParser import ConfigParser as _ConfigParser
from copy import copy as _copy
import json as _json
from logging import getLogger as _getLogger
from tempfile import NamedTemporaryFile as _NamedTemporaryFile
import time as _time
from re import compile as _compile
from requests import ConnectionError as _ConnectionError
from uuid import uuid4 as _random_uuid
import os as _os
import datetime as _datetime
import types as _types

from boto import connect_s3 as _connect_s3
from boto.s3.key import Key as _s3_key

from _file_util import is_path as _is_path, parse_s3_path as _parse_s3_path,  \
    s3_recursive_delete as _s3_recursive_delete, s3_delete_key as _s3_delete_key
import graphlab as _gl
import graphlab.canvas as _canvas
import graphlab.connect as _mt

from . import PREDICTIVE_SERVICE_SCHEMA_VERSION

# since _predictive_service_environment imports these, need to have them defined first
MAX_CREATE_TIMEOUT_SECS = 600 # 10m

from _predictive_service_environment import Ec2PredictiveServiceEnvironment
from _predictive_service_environment import predictive_service_environment_factory
from _model_predictive_object import ModelPredictiveObject
from _predictive_service_endpoint import PredictiveServiceEndpoint as _PredictiveServiceEndpoint
from _predictive_object import PredictiveObject as _PredictiveObject
from _custom_query_predictive_object import CustomQueryPredictiveObject as _CustomQueryPredictiveObject

_logger = _getLogger(__name__)
_name_checker = _compile('^[a-zA-Z-]+$')

class PredictiveService(object):

    # S3 Config Section Names
    _DEPLOYMENT_SECTION_NAME = 'Predictive Objects Service Versions'
    _PREDICTIVE_OBJECT_DOCSTRING = 'Predictive Objects Docstrings'
    _ENVIRONMENT_SECTION_NAME = 'Environment Info'
    _SERVICE_INFO_SECTION_NAME = 'Service Info'
    _META_SECTION_NAME = 'Meta'

    # S3 Directory Names
    _PREDICTIVE_OBJECT_DIR = 'predictive_objects'
    _DEPENDENCIES_DIR ='dependencies'


    def __repr__(self):
        return self.__str__()

    def __str__(self):
        ret = ""
        ret += 'Name: %s' % self.name + '\n'
        ret += 'S3 Path: %s' % self._s3_state_path + '\n'
        ret += 'Description: %s' % self._description + '\n'
        ret += 'API Key: %s' % self._api_key + '\n'
        if self._environment is not None:
            ret += 'Load Balancer DNS Name: %s' % self._environment.load_balancer_dns_name + '\n'

        ret += "\nDeployed predictive objects:\n"
        for (po_name, po_info) in self._predictive_objects.iteritems():
            ret += '\tname: %s, version: %s\n' % (po_name, po_info['version'])

        if len(self._local_changes) == 0:
            ret += "No Pending changes.\n"
        else:
            ret += "Pending changes: \n"
            for (po_name, po_info) in self._local_changes.iteritems():
                version = po_info[1]['version'] if po_info[1] else None
                if version:
                    if version == 1:
                        ret += '\tAdding: %s\n' % po_name
                    else:
                        ret += '\tUpdating: %s to version: %s\n' % (po_name, version)
                else:
                    ret += "\tRemoving: %s\n" % po_name

        return ret

    @property
    def _all_predictive_objects(self):
        '''
        Return all predictive objects that are currently registered with a
        Predictive Service, including both deployed and pending objects.

        Returns
        --------
        out : a dictionary with predictive object names as keys and the current
        state of the predictive object as the correspond value.

        Examples
        --------

        >>> ps.predictive_objects
        {'recommender_one': {'version': 1, 'state': 'deployed', 'docstring':'this is my first recommender'},
         'recommender_two': {'version': 2, 'state': 'pending', 'docstring':'this is my second recommender'},
        '''
        ret = {}
        for (po_name, po_info) in self._predictive_objects.iteritems():
            ret[po_name] = {'state': 'deployed', 'version':po_info['version'], 'docstring': po_info['docstring']}

        # Add local changes on top of deployed ones
        for po_name in self._local_changes:
            (po_obj, po_info) = self._local_changes[po_name]
            if po_obj is not None:
                ret[po_name] = {'state': 'pending', 'version':po_info['version'], 'docstring': po_info['docstring']}
            else:  # delete
                del ret[po_name]

        return ret

    @property
    def deployed_predictive_objects(self):
        '''
        Return all deployed Predictive Objects for a Predictive Service.

        Returns
        --------
        out : dict
            One entry for each Predictive Object. The keys are the names of the
            objects and the values are the corresponding versions.

        Examples
        --------

        >>> ps.deployed_predictive_objects
        {'recommender_one': 2, 'recommender_two': 1}
        '''
        self._ensure_not_terminated()

        return _copy(self._predictive_objects)

    @property
    def pending_changes(self):
        '''
        Return all currently pending updates for a Predictive Service.

        Returns
        --------
        out: dict
            Keys are the changed Predictive Object names, values are a
            dictionary with two keys: 'action' and 'version'. 'action' is one of
            'add', 'update', or 'remove' and 'version' is the new version
            number or None if action is 'remove'

        Examples
        --------

        >>> ps.pending_changes
        Out:
            {'recommender_one': {'action': 'remove', 'version': None},
             'recommender_three': {'action': 'add', 'version': 1},
             'recommender_two': {'action': 'update', 'version': 2}}
        '''
        ret = {}
        for po_name in self._local_changes:
            (po_obj, po_info) = self._local_changes[po_name]
            po_version = po_info['version'] if po_info is not None else None
            if po_version == 1:
                ret[po_name] = {'action': 'add', 'version':po_version}
            elif po_obj == None:  # deleted case
                ret[po_name] = {'action': 'remove', 'version':None}
            else:
                ret[po_name] = {'action': 'update', 'version':po_version}
        return ret

    @staticmethod
    def _get_s3_state_config(s3_bucket_name, s3_key_name, credentials):
        conn = _connect_s3(**credentials)
        bucket = conn.get_bucket(s3_bucket_name, validate=False)
        key = bucket.get_key(s3_key_name)

        if not key:
            raise IOError("No Predictive Service at the specified location.")

        with _NamedTemporaryFile() as temp_file:
            key.get_contents_to_file(temp_file)
            temp_file.flush()
            config = _ConfigParser(allow_no_value=True)
            config.optionxform = str
            config.read(temp_file.name)
            temp_file.close()  # deletes temp file

        return config


    def __init__(self, name, s3_state_path, description, api_key, aws_credentials,
                 _new_service = True):
        '''
        Initialize a new Predictive Service object

        Notes
        -----
        Do not call this method directly.

        To create a new Predictive Service, use:
             graphlab.deploy.predictive_service.create(...)

        To load an existing Predictive Service, use
            graphlab.deploy.predictive_service.load(<ps-s3-path>)
        '''
        if type(name) != str:
            raise TypeError("Name of Predictive Service needs to be a string")

        self.name = name
        self._s3_bucket_name, self._s3_key_name = _parse_s3_path(s3_state_path)
        self._s3_state_key = self._s3_key_name + '/state.ini'
        self._description = description
        self._api_key = api_key

        self._local_changes = {}
        self._predictive_objects = {}
        self._s3_state_path = s3_state_path
        self.aws_credentials = aws_credentials
        self._session = _gl.deploy._default_session

        if _new_service:
            # Verify we're not overriding another predictive service.
            bucket = _connect_s3(**self.aws_credentials).get_bucket(self._s3_bucket_name,
                                                                    validate=False)
            key = bucket.get_key(self._s3_state_key)
            if key:
                raise IOError("There is already a Predictive Service at the specified location. Use"
                              " a different S3 path. If you want to load an existing Predictive"
                              " Service, call 'load(...)'.")

            # Init version data
            self._revision_number = 0
            self._schema_version = PREDICTIVE_SERVICE_SCHEMA_VERSION

            # No environment yet. A launched one must be attached later.
            self._environment = None

            # Write init data to S3
            self._save_state_to_s3()
        else:
            # Read version data
            self._update_from_s3()

    def get_status(self):
        '''
        Gets the status of each host in the environment.
        '''
        result = '\nSERVICE\n'
        result += self.__str__() + '\n'

        result += 'HOSTS\n'
        for host in self._get_status():
            result += "%s (id: %s)\n" % (host['dns_name'], host['id'])
            result +=  "State: %s" % host['state']
            if host['state'] != 'InService':
                result += " - %s" % host['reason']
            result += "\nServing Models:\n"
            result += _json.dumps(host['models'], sort_keys=True, indent=4, separators=(',', ':'))
            result += '\n'

        print result

    def get_predictive_objects_status(self):
        ''' Get Predictive Objects deployment status in all nodes.

        Examples
        --------

        >>> ps.get_predictive_objects_status()
        +-----------------+------------------+-------------------------+-------------------------+
        |       name      | expected version |     node.i-29397b26     |     node.i-2a397b25     |
        +-----------------+------------------+-------------------------+-------------------------+
        | Image Predictor |        1         | 1 (Loaded successfully) | 1 (Loaded successfully) |
        | Book Recommender|        2         | 2 (Loaded successfully) | 2 (Loaded successfully) |


        '''
        self._ensure_not_terminated()

        expected = self.deployed_predictive_objects
        result = self._get_status()

        # result is in the following format
        # [{'dns_name': u'ec2-54-191-36-143.us-west-2.compute.amazonaws.com',
        #   'id': u'i-29397b26',
        #   'models': [{u'description': u'None',
        #     u'name': u'Image Predictor',
        #     u'status': u'Loaded successfully',
        #     u'version': 1}],
        #   'reason': u'N/A',
        #   'state': u'InService'},
        #   {...}
        #   ]
        sf = _gl.SArray(result)

        # unpack the dictionary to different columns, result looks like:
        # +------------+------------+--------+-----------+--------------------------------+
        # |  dns_name  |     id     | reason |   state   |             models             |
        # +------------+------------+--------+-----------+--------------------------------+
        # | ec2-54 ... | i-29397b26 |  N/A   | InService | {'status': 'Loaded success ... |
        # | ec2-54 ... | i-29397b26 |  N/A   | InService | {'status': 'Loaded success ... |
        sf = sf.unpack(column_name_prefix="").stack('models', 'models')

        # in case no model in any instance, there is no way to infer type of 'models' column

        if sf['models'].dtype() == float:
            print 'No models deployed in any instance'
            return

        # expand the "models" column, results looks like:
        # +------------+------------+--------+-----------+------------+------------+------------+---------+
        # |  dns_name  |     id     | reason |   state   | descri ... |    name    |   status   | version |
        # +------------+------------+--------+-----------+------------+------------+------------+---------+
        # | ec2-54 ... | i-29397b26 |  N/A   | InService |    None    | Image  ... | Loaded ... |    1    |
        # | ec2-54 ... | i-29397b26 |  N/A   | InService | finds  ... | shorte ... | Loaded ... |    3    |

        sf = sf.unpack('models', column_name_prefix = "")

        # Add expected and actual versions
        sf['expected version'] = sf.apply(lambda x: expected[x['name']]['version'] if expected.has_key(x['name']) else None )
        sf['actual'] = sf.apply(lambda x: '%s (%s)' % (x['version'], x['status']))

        # pivot the table by model and show one row for each model
        return sf \
            .groupby('name', {'state':_gl.aggregate.CONCAT('id', 'actual'), 'expected version':_gl.aggregate.SELECT_ONE('expected version')}) \
            .unpack('state', column_name_prefix='node')

    def _get_status(self):
        '''
        Gets the status of each host in the environment.
        '''
        self._ensure_not_terminated();

        return self._environment.get_status()

    def query(self, po_name, data):
        '''
        Queries the operation of the Predictive Object.

        Same as the query operation the client uses, useful for testing.

        Parameters
        ----------
        po_name : str
            The name of the Predictive Object to query

        data : dict
            Any additional data/parameters to the query

        Examples
        ---------

        If there is a Predictive Object named 'book recommender' and we want to
        recommend some books for users, the following query may be used:

          >>> ps.query('book recommender', {
                'method':'recommend',
                'data': {
                    'users': [
                        {'user_id':12345, 'book_id':2},
                        {'user_id':12346, 'book_id':3},
                    ],
                    'k': 5
                }
            })


        '''
        self._ensure_not_terminated();

        return self._environment.query(po_name, data, self._api_key)


    def add(self, name, obj, dependencies=None, required_packages=None, description=''):
        '''
        Adds a new model or custom predictive object to this predictive service.

        The name of the predictive object must be unique to the Predictive
        Service. This operation will not take effect until `apply_changes` is
        called.

        Parameters
        ----------
        name : str
            A unique identifier for the predictive object. This name is used
            when querying the Predictive Service deployment.

        obj : Model | function | path (local or S3) to a saved Model
            The Predictive Object to add to this Predictive Service.

            The Predictive Object can either be a GraphLab Model or a custom
            user-defined function.

            If 'obj' is a Model, then it can either be an actual GraphLab Model
            object, or it can be a path to a place where the model is saved.

            For a user-defined function (to define a custom predictive object),
            the function must have the following signature:

                >>> def query(input, dependencies=None, state=None):
                >>>           # input is a dict
                >>>           # dependencies is a dict
                >>>           # state is a dict
                >>>     return dict

            The 'dependencies' argument is the dictionary specified below. The
            'state' argument is a dictionary that is maintained by the
            Predictive Service deployment between each call to the query
            object. Use this dictionary to track anything that requires state
            between calls.

        dependencies : dict
            A dictionary that specifies all dependent GraphLab objects that
            this custom query relies on. The key is the identifier for the
            dependent object, the value is either a url or an actual GraphLab
            object. This parameter is only used if the obj parameter is a
            function.

        required_packages : list(str)
            A list that specifies all python packages this custom Predictive
            Object is dependent on. The dependencies are specifed in the format
            of distutils, like: ['mysql==0.3.0', 'abc==1.2.3']. The deployment
            will install these dependencies from Pypi when apply_changes is
            called. This parameter is only used if the obj parameter is a
            function.

        description : str
            The description of the custom predictive object, optional.

        See Also
        --------
        apply_changes

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        Examples
        --------
        To add a GraphLab Model:

            >>> model = graphlab.recommender.create(...)
            >>> ps.add('recommender', model)

        To add a GraphLab Model saved locally / S3:

            >>> ps.add('recommender', '~/saved_models/recommender')
            >>> ps.add('recommender2', 's3://model-archive/recommender')

        To add a custom predictive object (define a function, no dependencies or state):

            >>> def query(input, dependencies=None, state=None):
            >>>     if input['query'] == 'Hello':
            >>>         return {'result':' world!'}
            >>>     else:
            >>>         return {'result':''}
            >>>
            >>> ps.add('helloworld', query)

        To add a custom predictive object (define a function, with dependencies, and state):

            >>> def query(input, dependencies=None, state=None):
            >>>     import time
            >>>     recommender = dependencies['recommender']
            >>>
            >>>     if 'user' in input:
            >>>         state[input['user']] = time.time() # record last time query for this user
            >>>         return {'result': recommender.recommend(input['user'])
            >>>     else:
            >>>         return {'result':'Missing user parameter'}
            >>>
            >>> # set up dependencies for Custom Predictive Object
            >>> deps = { 'recommender' : graphlab.recommender.create(...) }
            >>> ps.add('custom-recommender', query, dependencies=deps)

        '''
        self._ensure_not_terminated();

        if not isinstance(name, str):
            raise TypeError("'name' must be a string")

        if name == None or name == '':
            raise TypeError("'name' cannot be empty")

        if name in self.deployed_predictive_objects.keys():
            raise ValueError("There is already a predictive object with name '%s'." % name)

        predictive_object = None
        if isinstance(obj, _types.FunctionType):
            predictive_object = self._create_custom_po(obj, dependencies, required_packages, description)
        elif isinstance(obj, _gl.Model) or _is_path(obj):
            predictive_object = ModelPredictiveObject(model=obj)
        elif isinstance(obj, _PredictiveObject):
            predictive_object = obj
        else:
            raise TypeError("'obj' parameter has to be either an instance of GraphLab model, a path to a saved GraphLab model, or an instance of GraphLab PredictiveObject")

        # extract doc string
        if not _is_path(obj):
            docstring = predictive_object.get_doc_string().strip()
        else:
            docstring = ''
        self._local_changes[name] = (predictive_object, {'version':1, 'docstring':docstring})

        _logger.info("New predictive object '%s' added, use apply_changes() to deploy all pending changes, or continue other modification." % name)

    def update(self, name, obj, dependencies=None, required_packages=None, description = ''):
        '''
        Updates the version of the predictive object being served.

        Parameters
        ----------
        name : str
            The unique identifier for the model.

        obj : Model | function | path (local or S3) to a saved Model
            The Predictive Object to add to this Predictive Service.

            The Predictive Object can either be a GraphLab Model or a custom
            user-defined function.

            If 'obj' is a model, then it can either be an actual GraphLab Model
            object, or it can be a path to a place where the model is saved.

            For a user-defined function (to define a custom predictive object),
            the function must have the following signature:

                >>> def query(input, dependencies=None, state=None):
                >>>           # input is a dict
                >>>           # dependencies is a dict
                >>>           # state is a dict
                >>>     return dict

            The 'dependencies' argument is the dictionary specified below. The
            'state' argument is a dictionary that is maintained by the
            Predictive Service deployment between each call to the query
            object. Use this dictionary to track anything that requires state
            between calls.

        dependencies : dict
            A dictionary that specifies all dependent GraphLab objects that
            this custom query relies on. The key is the identifier for the
            dependent object, the value is either a url or an actual GraphLab
            object. This parameter is only used if the obj parameter is a
            function.

        required_packages : list(str)
            A list that specifies all python packages this custom Predictive
            Object is dependent on. The dependencies are specifed in the format
            of distutils, like: ['mysql==0.3.0', 'abc==1.2.3']. The deployment
            will install these dependencies from Pypi when apply_changes is
            called. This parameter is only used if the obj parameter is a
            function.

        See Also
        --------
        add, apply_changes

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        Examples
        --------
        To update a predictive object named 'recommender':

            >>> new_recommender = graphlab.recommender.create(...)
            >>> ps.update('recommender', new_recommender)

        '''
        self._ensure_not_terminated();

        new_version = 1
        if name not in self.deployed_predictive_objects.keys():
            if name not in self.pending_changes.keys():
                raise ValueError("Cannot find a predictive object with name '%s'." % name)
            else:
                new_version = 1
        else:
            new_version = 1 + self._predictive_objects[name]['version']

        predictive_object = None
        if isinstance(obj, _types.FunctionType):
            predictive_object = self._create_custom_po(obj, dependencies, required_packages, description)
        elif isinstance(obj, _gl.Model) or _is_path(obj):
            predictive_object = ModelPredictiveObject(model=obj)
        elif isinstance(obj, _PredictiveObject):
            predictive_object = obj
        else:
            raise TypeError("'obj' parameter has to be either an instance of GraphLab model, a path to a saved GraphLab model, or an instance of GraphLab PredictiveObject")

        if not _is_path(obj):
            docstring = predictive_object.get_doc_string().strip()
        else:
            docstring = ''
        self._local_changes[name] = (predictive_object, {'version':new_version, 'docstring':docstring})

        _logger.info("Predictive object '%s' updated to version '%s', use apply_changes() to deploy all pending changes, or continue other modification." % (name, new_version))

    def remove(self, name):
        '''
        Removes a deployed Predictive Object

        Parameters
        ----------
        name : str
            The name of the Predictive Object to be removed

        Notes
        -----
        This operation will not take effect until `apply_changes` is called.

        See Also
        --------
        apply_changes
        '''
        self._ensure_not_terminated();

        if name not in self.deployed_predictive_objects.keys():
            if name in self.pending_changes.keys():
                del self._local_changes[name]
            else:
                raise ValueError("Cannot find a model with name '%s'." % name)
        else:
            self._local_changes[name] = (None, None)

        _logger.info("Predictive object '%s' removed, use apply_changes() to deploy all pending changes, or continue other modification." % name)

    def apply_changes(self):
        '''
        Applies all pending changes to this Predictive Service deployment.

        If apply_changes returns success, then all pending changes have been
        staged to the S3 bucket associated with the Predictive Service. Each of
        the nodes in the Predictive Service will pick up the state eventually.

        Use get_status() to check the status of each node.

        See Also
        --------
        add, update, remove
        '''
        self._ensure_not_terminated()

        self._save_state_to_s3()
        try:
            self._environment.poke()
        except _ConnectionError as e:
            _logger.warn("Unable to connect to running Predictive Service: %s" %
                         (e.message))


    def clear_pending_changes(self):
        '''
        Clears all changes which have not been applied.

        Clears all actions done using `add`, `update` and `remove`, since the
        last time `apply_changes` was called.
        '''
        _logger.info('Clearing all pending changes.')
        self._local_changes = {}


    def save_client_config(self, file_path, predictive_service_cname):
        '''
        Create the config file that can be used by applications accessing the
        Predictive Service.

        The file is stored in an .ini format, with the specific information
        necessary to query this deployment.

        Parameters
        ----------
        file_path : str
            The path where the config file will be saved.

        predictive_service_cname : str
            The CNAME for the Predictive Service endpoint. It is *highly recommended*
            that all client connect through a CNAME record that you have created.
            If this value is set to None, the "A" record will be used.
            Using the "A" record is only advisable for testing, since this value
            may change over time.
        '''
        self._ensure_not_terminated();

        out = _ConfigParser(allow_no_value=True)
        out.optionxform = str
        out.add_section(PredictiveService._SERVICE_INFO_SECTION_NAME)
        out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'api key', self._api_key)

        if self._environment.certificate_name:
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'verify certificate',
                    not(self._environment.certificate_is_self_signed))
            schema = 'https://'
        else:
            schema = 'http://'

        if predictive_service_cname:
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'endpoint',
                    schema + predictive_service_cname)
        else:
            _logger.warn('Creating client config using the "A" Record name. You\'re crazy to use' \
                             ' this for anything other than testing!')
            out.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'endpoint',
                    schema + self._environment.load_balancer_dns_name)

        with open(file_path, 'w') as f:
            out.write(f)


    def _has_state_changed_on_s3(self):
        '''
        Returns whether any changes have been uploaded to S3 since this Predictive
        Service was loaded or created.
        '''
        if self._revision_number == 0:
            return False
        config = PredictiveService._get_s3_state_config(self._s3_bucket_name, self._s3_state_key,
                                                        self.aws_credentials)
        saved_version_num = config.getint(self._META_SECTION_NAME, 'Revision Number')
        return (self._revision_number != saved_version_num)


    def terminate_service(self, remove_logs = False, remove_state = True):
        '''
        Terminates the Predictive Service deployment.

        This will terminate all EC2 hosts and delete the load balancer. The
        Predictive Service object is not usable after terminate_service is
        called.

        This operation can not be undone.

        Parameters
        ----------
        remove_logs : bool
            Delete all logs associated with this Predictive Service Deployment from S3.

        remove_state : bool
            Delete all state data associated with this Predictive Service Deployment from S3.
        '''
        if not self._environment:
            _logger.error('Predictive service has already been terminated.')
            return

        # Terminate hosts and delete load balancer.
        self._environment.terminate(remove_logs)
        self._environment = None

        if remove_state:
            _logger.info('Deleting state data.')
            try:
                _s3_recursive_delete(self._get_root_path() + PredictiveService._DEPENDENCIES_DIR,
                                     self.aws_credentials)
                _s3_recursive_delete(self._get_root_path() + PredictiveService._PREDICTIVE_OBJECT_DIR,
                                     self.aws_credentials)
                _s3_delete_key(self._s3_bucket_name, self._s3_state_key, self.aws_credentials)
            except:
                _logger.error("Could not delete predictive object data from S3. Please manually delete data under: %s" %
                              self._get_root_path())

        # Delete local predictive service endpoint
        _gl.deploy.predictive_services.delete(self.name)


    def _update_from_s3(self):
        '''
        Update the predictive objects from S3.

        Returns
        -------
        out : dict
            Returns a summary of what has changed. There is one entry in the dictionary for each
            predictive object that has changed. The keys are the name of the predictive object. The
            values are tuples: (model_version_number, s3_path_to_new_version).
        '''
        # TODO: make sure we're not "ahead" of S3 due to eventual consistency

        s3_config = PredictiveService._get_s3_state_config(self._s3_bucket_name, self._s3_state_key,
                                                           self.aws_credentials)

        if s3_config.has_option(PredictiveService._META_SECTION_NAME, 'Schema Version'):
            self._schema_version = s3_config.getint(PredictiveService._META_SECTION_NAME, 'Schema Version')
        else:
            self._schema_version = PREDICTIVE_SERVICE_SCHEMA_VERSION

        if self._schema_version > PREDICTIVE_SERVICE_SCHEMA_VERSION:
            raise RuntimeError("Your GraphLab Create only supports Predictive "
                "Service with schema version up to '%s', the Predictive Service "
                "you are trying to load has schema version '%s'. Please upgrade your "
                "GraphLab Create version to most up-to-date one." % \
                (PREDICTIVE_SERVICE_SCHEMA_VERSION, self._schema_version))

        updated_deployment_versions = {}
        model_names = s3_config.options(PredictiveService._DEPLOYMENT_SECTION_NAME)
        for cur_model_name in model_names:
            cur_model_version = s3_config.getint(PredictiveService._DEPLOYMENT_SECTION_NAME,
                                              cur_model_name)
            updated_deployment_versions[cur_model_name] = {'version':cur_model_version, 'docstring': ''}

        if s3_config.has_section(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING):
            for cur_model_name in model_names:
                cur_model_docstring = s3_config.get(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING,
                                                  cur_model_name)
                updated_deployment_versions[cur_model_name]['docstring'] = cur_model_docstring.decode('string_escape')

        if s3_config.has_section(PredictiveService._ENVIRONMENT_SECTION_NAME):
            # Create and attach the environment.
            environment_info = dict(s3_config.items(PredictiveService._ENVIRONMENT_SECTION_NAME))
            environment_info['aws_credentials'] = self.aws_credentials
            self._environment = predictive_service_environment_factory(environment_info)
        else:
            self._environment = None

        diff = {}
        for (cur_po_name, cur_po_info) in updated_deployment_versions.items():
            if(cur_po_name not in self._predictive_objects or
               cur_po_info['version'] != self._predictive_objects[cur_po_name]['version']):
                # Either a new predictive object or new predictive object version
                s3_path_to_new_version = self._get_predictive_object_save_path(cur_po_name,
                                                                               cur_po_info['version'])
                diff[cur_po_name] = (cur_po_info['version'], s3_path_to_new_version)

        # add removed models too
        for (po_name, po_info) in self._predictive_objects.iteritems():
            if po_name not in updated_deployment_versions.keys():
                diff[po_name] = (None, None)

        self._revision_number = s3_config.getint(PredictiveService._META_SECTION_NAME, 'Revision Number')

        self._predictive_objects = updated_deployment_versions
        self._local_changes = {}

        return diff

    def get_metrics_url(self):
        '''
        Return a URL to Amazon CloudWatch for viewing the metrics assoicated with
        the service

        Returns
        -------
        out : str
            A URL for viewing Amazon CloudWatch metrics
        '''
        self._ensure_not_terminated();

        _logger.info("retrieving metrics from predictive service...")
        try:
            return self._environment.get_metrics_url()
        except Exception as e:
            _logger.error("Error retrieving metrics: %" % e.message)

    def _get_metrics(self, start_date=None, end_date=None):
        '''
        Get the metrics associated with the Predictive Service instance.

        Parameters
        -----------
        start_date : datetime, optional
            The begin time (in utc) to query metrics.

        end_date : datetime, optional
            The end time (in utc) to query metrics

        Returns
        -------
        out : dict
            Return a dictionary containing aggregated metric data as a time
            series from the specified date range.

        Notes
        -----
        If start_date and end_date are not given, metrics for the past 7 days
        are retrieved.

        '''
        self._ensure_not_terminated();

        return self._environment.get_metrics(start_date, end_date)

    def get_metrics(self, start_time=None, end_time=None, time_range=None, period=None):
        '''
        Get the metrics associated with the Predictive Service instance. The metrics includes:
        number of requests, latency, number of healthy hosts, and number of keys in cache.

        Parameters
        ----------
        start_time : datetime or str, optional
            The begin time to query metrics.

        end_time : datetime or str, optional
            The end time to query metrics.

        time_range : int , optional
            The range of time in hours to query metrics. Default is 12 hours.

        period : int, optional
            The period of data points in seconds for query metrics. Default is 5 minutes.

        Returns
        -------
        out : tuple (SFrames)
            Returns a tuple of SFrames containing the SFrames for each metrics:
            number of requests, latency, number of healthy hosts, number of keys in cache.
            It also contains a SFrame with the start and end time for this metrics query.

        Notes
        -----
        If none of the parameters are given, this function would return the metrics for
        the last 12 hours.

        Examples
        --------

        # last 12 hours of metrics, with 5 minute (300 seconds) period
        >>> ps._get_metrics()
        ({'end_time': datetime.datetime(2014, 11, 13, 11, 36, 3, 157682),
          'start_time': datetime.datetime(2014, 11, 12, 23, 36, 3, 157682)},
        +------------------+---------------------------+-------------------+
        | num_requests.Sum |   num_requests.Timestamp  | num_requests.Unit |
        +------------------+---------------------------+-------------------+
        |       8.0        | 2014-11-13 00:31:00+00:00 |       Count       |
        |       2.0        | 2014-11-13 00:36:00+00:00 |       Count       |
        |       7.0        | 2014-11-13 00:41:00+00:00 |       Count       |
        |     24707.0      | 2014-11-13 00:46:00+00:00 |       Count       |
        |       5.0        | 2014-11-13 00:51:00+00:00 |       Count       |
        ......

        # last 6 hours of metrics, with 1 minute (60 seconds) period
        >>> ps._get_metrics(end_time = datetime.datetime.now(), time_range = 6, period = 60)
        ({'end_time': datetime.datetime(2014, 11, 13, 11, 49, 7, 743095),
          'start_time': datetime.datetime(2014, 11, 13, 5, 49, 7, 743095)},
        +------------------+---------------------------+-------------------+
        | num_requests.Sum |   num_requests.Timestamp  | num_requests.Unit |
        +------------------+---------------------------+-------------------+
        |       1.0        | 2014-11-13 05:52:00+00:00 |       Count       |
        |       1.0        | 2014-11-13 05:54:00+00:00 |       Count       |
        |       1.0        | 2014-11-13 05:55:00+00:00 |       Count       |
        |       1.0        | 2014-11-13 06:04:00+00:00 |       Count       |
        |       1.0        | 2014-11-13 06:06:00+00:00 |       Count       |
        |       1.0        | 2014-11-13 06:19:00+00:00 |       Count       |
        ......

        '''
        # default time range of metrics to 12 hours
        if not time_range or type(time_range) != int:
            time_range = 12

        # convert timestamps to datetimes if necessary
        if start_time is not None:
            if type(start_time) == str or type(start_time) == unicode:
                start_time = _datetime.datetime.strptime(start_time.split(".")[0], '%Y-%m-%dT%H:%M:%S')
        if end_time is not None:
            if type(end_time) == str or type(end_time) == unicode:
                end_time = _datetime.datetime.strptime(end_time.split(".")[0], '%Y-%m-%dT%H:%M:%S')

        # get actual start_time and end_time
        if start_time is not None and end_time is not None:
            pass
        elif start_time is not None and end_time is None:
            end_time = start_time + _datetime.timedelta(hours=time_range)
        elif end_time is not None and start_time is None:
            start_time = end_time - _datetime.timedelta(hours=time_range)
        else:
            end_time = _datetime.datetime.utcnow()
            start_time = end_time - _datetime.timedelta(hours=time_range)

        # default period to 5 min
        if not period or type(period) != int:
            period = 300

        # get CloudWatch metrics
        metrics = self._environment._get_metrics(self.name, start_time, end_time, period)
        if not metrics:
            return {'start_time': start_time,
                      'end_time': end_time}

        # get requests, latency, num_healthy_hosts
        num_requests_dict = map(dict, sorted(metrics['num_requests']))
        req_sf = None
        if len(num_requests_dict) > 0:
            req_sf = _gl.SArray(num_requests_dict).unpack('num_requests').sort('num_requests.Timestamp')

        latency_dict = map(dict, sorted(metrics['latency']))
        latency_sf = None
        if len(latency_dict) > 0:
            latency_sf = _gl.SArray(latency_dict).unpack('latency').sort('latency.Timestamp')

        num_healthy_hosts_dict = map(dict, sorted(metrics['num_healthy_hosts']))
        hosts_sf = None
        if len(num_healthy_hosts_dict) > 0:
            hosts_sf = _gl.SArray(num_healthy_hosts_dict).unpack('num_healthy_hosts').sort('num_healthy_hosts.Timestamp')

        num_keys_in_cache_dict = map(dict, sorted(metrics['num_keys_in_cache']))
        cache_sf = None
        if len(num_keys_in_cache_dict) > 0:
            cache_sf = _gl.SArray(num_keys_in_cache_dict).unpack('num_keys_in_cache').sort('num_keys_in_cache.Timestamp')

        # time frame sframe
        time_frame = {'start_time': start_time,
                      'end_time': end_time}

        # return a tuple of metrics sframes
        return (time_frame, req_sf, latency_sf, hosts_sf, cache_sf)

    @_canvas.inspect.find_vars
    def show(self):
        """
        Visualize the Predictive Service with GraphLab Canvas. This function
        starts Canvas if it is not already running.

        Returns
        -------
        view: graphlab.canvas.view.View
            An object representing the GraphLab Canvas view

        See Also
        --------
        canvas
        """
        self._ensure_not_terminated();

        _canvas.get_target().state.set_selected_variable(('Predictive Services', self.name))
        return _canvas.show()


    def save(self):
        """
        Saves the Predictive Service information to disk. Information saved
        contains: Name, S3 path, and AWS credentials. Note that only metadata
        of the Predictive Service is stored, it does not impact the actual
        deployed Predictive Service.

        This information can be useful later to access the Predictive Service,
        for example:

            >>> import graphLab
            >>> my_ps = graphlab.deploy.predictive_services[<name>]

        """
        self._ensure_not_terminated();

        self._session.save(self, typename="PredictiveService")


    def _save_state_to_s3(self):
        # Dump immutable state data to a config
        state = _ConfigParser(allow_no_value=True)
        state.optionxform = str
        state.add_section(PredictiveService._SERVICE_INFO_SECTION_NAME)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'Name', self.name)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'Description', self._description)
        state.set(PredictiveService._SERVICE_INFO_SECTION_NAME, 'API Key', self._api_key)

        # Save environment, if we have one
        if self._environment:
            state.add_section(PredictiveService._ENVIRONMENT_SECTION_NAME)
            for (key, value) in self._environment._get_state().iteritems():
                state.set(PredictiveService._ENVIRONMENT_SECTION_NAME, key, value)

        # Save deployment version data to config
        state.add_section(PredictiveService._DEPLOYMENT_SECTION_NAME)
        current_predictive_objects = _copy(self._all_predictive_objects)
        for (model_name, info) in current_predictive_objects.iteritems():
            state.set(PredictiveService._DEPLOYMENT_SECTION_NAME, model_name, info['version'])

        state.add_section(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING)
        for (model_name, info) in current_predictive_objects.iteritems():
            state.set(PredictiveService._PREDICTIVE_OBJECT_DOCSTRING, model_name, info['docstring'].encode('string_escape'))

        if self._has_state_changed_on_s3():
            raise IOError("Can not save changes. The Predictive Service has changed on S3. Please "
                          "reload from S3.")

        # Save any new predictive objects to S3.
        for predictive_object_name in self._local_changes:
            (predictive_object, po_info) = self._local_changes[predictive_object_name]
            if predictive_object:         # if this is not a model deletion:
                save_path = self._get_predictive_object_save_path(predictive_object_name, po_info['version'])
                dependency_path = self._get_dependency_save_path(predictive_object_name, po_info['version'])
                predictive_object.save(save_path, dependency_path, self.aws_credentials)

        # Update the revision number after we have successfully written all predictive objects
        self._revision_number += 1
        state.add_section(self._META_SECTION_NAME)
        state.set(self._META_SECTION_NAME, 'Revision Number', self._revision_number)
        state.set(self._META_SECTION_NAME, 'Schema Version', self._schema_version)

        # Write state file to S3
        with _NamedTemporaryFile() as temp_file:
            state.write(temp_file)
            temp_file.flush()
            conn = _connect_s3(**self.aws_credentials)
            bucket = conn.get_bucket(self._s3_bucket_name, validate=False)
            key = _s3_key(bucket)
            key.key = self._s3_state_key
            key.set_contents_from_filename(temp_file.name)
            temp_file.close()  # deletes temp file

        # Update our state
        self._local_changes = {}
        self._predictive_objects = dict(zip(current_predictive_objects.keys(),
            [{'version':info['version'], 'docstring': info['docstring']} for info in current_predictive_objects.values()]))


    def _get_predictive_object_save_path(self, predictive_object_name, version_number):
        return self._get_root_path() + '/'.join([PredictiveService._PREDICTIVE_OBJECT_DIR,
                                                 predictive_object_name, str(version_number)])

    def _get_dependency_save_path(self, predictive_object_name, version_number):
        return self._get_root_path() + '/'.join([PredictiveService._DEPENDENCIES_DIR,
                                                 predictive_object_name, str(version_number)])

    def _get_root_path(self):
        if self._s3_key_name:
            return "s3://%s/%s/" % (self._s3_bucket_name, self._s3_key_name)
        else:
            return "s3://%s/" % self._s3_bucket_name

    def _ensure_not_terminated(self):
        if not self._environment:
            raise RuntimeError("This operation is not supported because Predictive Service is already terminated")

    def _create_custom_po(self, query, dependencies=None, required_packages=None, description = ''):
        return _CustomQueryPredictiveObject(
            query = query,
            dependencies = dependencies,
            required_packages = required_packages,
            description = description)
