from graphlab.connect import _get_metric_tracker
from inspect import getmodule as _get_module
from itertools import product as _product
from os.path import dirname as _dirname, join as _path_join
from random import sample as _random_sample
from time import time as _time

import graphlab as _gl
from graphlab import SFrame as _SFrame
from graphlab import SArray as _SArray
from graphlab.deploy import _Pipeline as _Pipeline, Task as _Task, \
    parallel_for_each as _parallel_for_each


def _get_all_parameters_combinations(parameters):
    '''
    Takes a dictionary where the keys are parameter names. The value of a key is a list of all
    possible values parameter.

    Returns a list of all possible parameter combinations. Each parameter set is a dictionary.

    For example an input of {'foo':[1,2], 'bar': {'x': ['a','b']}} will produce
    [{'foo':1, 'bar':{'x': 'a'}}, {'foo':1, 'bar':{'x': 'b'}},
     {'foo':2, 'bar':{'x': 'a'}}, {'foo':2, 'bar': {'x': 'b'}}]
    '''

    # Get all possible combinations
    parameter_names = parameters.keys()
    arg_list = []
    for i in parameter_names:
        val = parameters[i]
        if type(val) is dict:
            arg_list.append(_get_all_parameters_combinations(val))
        elif type(val) is not list:
            arg_list.append([val])
        else:
            arg_list.append(val)
    param_iter = _product(*arg_list)

    # Construct the output
    result = []
    for param_tuple in param_iter:
        param_dict = {}
        for i in range(len(param_tuple)):
            cur_arg_name = parameter_names[i]
            cur_arg_value = param_tuple[i]
            param_dict[cur_arg_name] = cur_arg_value
        result.append(param_dict)

    return result


def _get_all_model_fields(model):
    '''
    Saves all fields that a model exposes.
    '''
    result = {}
    for field_name in model.list_fields():
        result[field_name] = model[field_name]
    return result


def _get_default_parameter_range(model_factory, factory_params):
    '''
    Lookup the default hyper parameters for this model. First, try the module of the factory
    method. If that doesn't work, create a model and look in that module.
    '''
    if model_factory is _gl.recommender.create:
        if(('method' not in factory_params or factory_params['method'] == 'matrix_factorization')
           and 'target' in factory_params):
            model_factory = _gl.recommender.matrix_factorization.create
    module = _get_module(model_factory)
    if(not hasattr(module, 'DEFAULT_HYPER_PARAMETER_RANGE')):
        raise TypeError("%s does not support hyper parameter tuning" % model_factory)
    return module.DEFAULT_HYPER_PARAMETER_RANGE


def _flatten_to_single_sframe_saveable_dict(dic):
    '''
    Takes a dict or an SFrame. Does it's best to convert that to a dictionary that can be saved as
    an entry in an SFrame.
    '''
    if not dic:
        return {}

    test_sframe = _SFrame()
    def test_save_to_sframe(v):
        # An exception will be thrown if assignment fails.
        test_sframe['test'] = [v]

    data = {}
    for key, value in dic.items():
        try:
            if isinstance(value, _SFrame):
                data[key] = {}
                for inner_key in value.column_names():
                    data[key][str(inner_key)] = list(value[inner_key])
                test_save_to_sframe(data[key])
            elif isinstance(value, _SArray):
                value = list(value)
                test_save_to_sframe(value)
                data[key] = value
            elif isinstance(value, dict):
                value = _flatten_to_single_sframe_saveable_dict(value)
                test_save_to_sframe(value)
                data[key] = value
            else:
                test_save_to_sframe(value)
                data[key] = value
        except:
            data[key] = "Unable to store field"

    return data


def _recursive_dict_update(d1, d2):
    for k, v in d2.iteritems():
        if k in d1 and type(d1[k]) is dict and type(v) is dict:
            d1[k].update(v)
        else:
            d1[k] = v

def _train_test_model(task):
    '''
    This is the actual top level function that will be run (possibly remotely) to do the actual work
    of creating and evaluating models with different parameters.
    '''

    train_set = _SFrame(task.params['train_set'])
    del task.params['train_set']
    test_set = None
    if task.params['test_set'] is not None:
        test_set = _SFrame(task.params['test_set'])
    del task.params['test_set']

    model_factory = task.params['model_factory']
    del task.params['model_factory']

    # Create the model
    model = model_factory(train_set, **task.params)
    results = _SFrame({'model_name': [model.name()]})

    # Save 'model_details'
    model_info = _get_all_model_fields(model)
    results['model_details'] = [_flatten_to_single_sframe_saveable_dict(model_info)]

    # Save test info
    if test_set:
        test_evaluation_info = model.evaluate(test_set)
        results['test_metrics'] = [_flatten_to_single_sframe_saveable_dict(test_evaluation_info)]

    task.outputs['results'] = results


def model_parameter_search(model_factory, train_set_path, save_path, test_set_path=None,
                           standard_model_params={}, hyper_params=None, max_num_models='all',
                           name=None, environment = None):
    '''
    Searches for optimal model parameters. Automatically creates models using different parameters.
    Optionally, evaluates these models using a test set.

    You can specifiy the values of parameters you want to search over, using the `hyper_params`
    parameter. By default if you specify this, all possible combinations of all parameter values
    will be tried. If you do not specify `hyper_params`, a set of default parameter combinations,
    based on the model type, will be tried.

    This function can also be used for random search, rather than grid search. Using the
    `max_num_models` you can specify max number of models to try; that many combinations of
    `hyper_params`, will be randomly picked from all possible combinations.

    model_parameter_search is supported for:
    :py:class:`~graphlab.linear_regression.LinearRegression`,
    :py:class:`~graphlab.boosted_trees_regression.BoostedTreesRegression`,
    :py:class:`~graphlab.logistic_classifier.LogisticClassifier`,
    :py:class:`~graphlab.svm_classifier.SVMClassifier`,
    :py:class:`~graphlab.boosted_trees_classifier.BoostedTreesClassifier`,
    :py:class:`~graphlab.neuralnet_classifier.NeuralNetClassifier`,
    :py:class:`~graphlab.recommender.factorization_recommender.FactorizationRecommender`, and
    :py:class:`~graphlab.kmeans.KmeansModel`

    Parameters
    ----------
    model_factory : function
        This is the function (always ending in "create") that you would normally use to create the
        model. For example, if you wanted to use `model_parameter_search` for
        :py:class:`~graphlab.linear_regression.LinearRegression`, you would set this parameter
        to `graphlab.linear_regression.create`.

    train_set_path : str
        Path to a saved SFrame containing the train set. If running locally, this must be an absolute
        path. If running in EC2, this must be an S3 path. If running in Hadoop, this must be an HDFS
        path.

    save_path : str
        Path where the result of the model parameter search will be saved. Results will be saved as
        an SFrame with one row for each model that was created. If running locally, this path must be
        an absolute path. If running in EC2, this must be an S3 path. If running in Hadoop, this must
        be an HDFS path.

    test_set_path : str, optional
        Path to a saved SFrame containing the test set. This SFrame must be in the same format as the
        train set. If running locally, this must be an absolute path. If running in EC2, this must
        be an S3 path. If running in Hadoop, this must be an HDFS path.

    standard_model_params : dict, optional
        A set of arguments passed to every call to `model_factory`: these parameters will be passed
        to every model that gets created as part of the search.

    hyper_params : dict, optional
        The keys in the dictionary must be the names of the `model_factory` parameters that you want
        to search over. The value for a key will depend on the type that `model_factory` expects for
        that parameter. If the type of the parameter is not a dictionary, use a list of all the
        values you want to search over. For example if we we're using
        :class:`~graphlab.recommender.FactorizationRecommender` and want to search over all
        `num_factors` from 10 to 15 and the `regularization` values of 1 and 5, you would use:
        `{ 'num_factors': [10, 11, 12, 13, 14, 15], 'regularization': [1, 5] }`. If `model_factory`
        expects a dictionary for a parameter, the same rules apply recursively (see boosted trees
        example below).

    max_num_models : int, optional
        The max number of models to create. If `max_num_models` is less than the number of possible
        combinations of `hyper_params`, `max_num_models` of the possible `hyper_params` combinations
        will be randomly picked.

    name : str, optional
        The name for the :class:`Job <graphlab.deploy>` that will be created. If not specified then
        the name will be 'Model-Parameter-Search-(timestamp)'.

    environment : Environment
        This must be an environment created from the :py:class:`~graphlab.deploy` module. This is
        the environment where the parameter search will actually be done. The environment could be a
        set of EC2 hosts or a Hadoop cluster. It could be an environment that executes locally on
        your computer either as a blocking call or as an asynchronous background job.

    Returns
    -------
    out : :class:`job <graphlab.deploy>` object
        The job for the parameter search, which was run using the `environment` parameter. This
        object can be used to track the progress of the parameter search.

    Examples
    --------

    The easiest option to use for an environment is just to have it execute synchronously (i.e. as a
    blocking call) on your local computer. For that you'll need to create a
    :class:`Local Environment <graphlab.deploy>`. Since you're running it locally and synchronously,
    you don't need to worry about the return value of `model_parameter_search`; the function will not
    return until the parameter search is completed.

    .. sourcecode:: python

        # SFrames need to be saved. Make sure all file paths are absolute.
        >>> train_file_path = '/data/train_set.gl'
        >>> test_file_path = '/data/test_set.gl'
        >>> test_set.save(test_file_path)
        >>> train_set.save(train_file_path)

        # Setup a python environment
        >>> env = graphlab.deploy.environment.Local('local env')

        # For Kmeans using the default range of hyper-parameters for that model.
        >>> model_parameter_search(env, graphlab.kmeans.create, train_file_path, '/data/results.gl')

        # For Matrix Factorization with a target of 'rating', using the default range of hyper-params.
        >>> model_parameter_search(env, graphlab.factorization_recommender.create, train_file_path,
                                '/data/results.gl', test_set_path = test_file_path,
                                standard_model_params = {'target': 'rating'}
                              )


        # For boosted trees with a target of 'results', trying only a max_depth 5 and 10.
        # This time the parameter search will be done asynchronously on an EC2 host.
        >>> train_file_path = 's3://my-bucket-name/train_file_path'
        >>> test_file_path = 's3://my-bucket-name/test_file_path'
        >>> test_set.save(test_file_path)
        >>> train_set.save(train_file_path)

        >>> graphlab.aws.set_credentials('RBZH792CTQPP7T435BGQ', '7x2hMqplWsLpU/qQCN6xAPKcmWo46TlPJXYTvKcv')
        >>> ec2_env = graphlab.deploy.environment.EC2('ec2 env', 's3://my-bucket-name/log-dir')
        >>> job = model_parameter_search(ec2_env, graphlab.boosted_trees_classifier.create, train_file_path,
                                          's3://my-bucket-name/save_file_path',
                                          test_set_path = test_file_path,
                                          standard_model_params = {'target': 'result'},
                                          hyper_params = {'max_depth': [5, 10]}
                                        )

        # Get the status of the jobs
        >>> job.get_status()

    Upon completion, an SFrame will be saved at the path specified by the
    save_path parameter. This SFrame will contain the following columns:

        model_name: the type of model
        model_details: additional model details (all the data available from
          a `Model`'s `list_fields` method)
        parameters: the parameters used to generate the model associated with
          that row

    And if the test_set_path parameter was specified, there will be one
    additional column:

        test_metrics: additional model statistics generated on the test set
    '''

    _get_metric_tracker().track('jobs.model_parameter_search')

    if name is None:
        name =  "Model-Parameter-Search-%s" % _time()

    # Determine search space
    if(hyper_params is None):
        hyper_params = _get_default_parameter_range(model_factory, standard_model_params)
    search_space = _get_all_parameters_combinations(hyper_params)
    if(max_num_models != 'all' and max_num_models < len(search_space)):
        search_space = _random_sample(search_space, max_num_models)
    for d in search_space:
        _recursive_dict_update(d, standard_model_params)

    train_test_task = _Task("Model Train Test %s" % _time())
    train_test_task.set_code(_train_test_model)
    train_test_task.set_outputs({'results': save_path})
    train_test_task.set_params({'train_set':train_set_path, 'test_set':test_set_path,
                                'model_factory': model_factory
                                })
    _gl.deploy.tasks.delete(train_test_task)

    return _parallel_for_each(train_test_task, search_space, name = name, environment = environment)
