import graphlab
import graphlab.canvas
import graphlab.canvas.views as views
import graphlab.canvas.views.bar
import graphlab.canvas.views.base
import graphlab.canvas.views.heatmap
import graphlab.canvas.views.sarray
import graphlab.canvas.views.scatter
import graphlab.connect as _mt

import copy
import json
import re
import textwrap
import threading

class SFrameView(graphlab.canvas.views.base.BaseView):

    def __init__(self, obj, params=None):
        super(SFrameView, self).__init__(obj, params)

        # metrics
        _mt._get_metric_tracker().track('canvas.sframe.row.size', value=len(obj))
        _mt._get_metric_tracker().track('canvas.sframe.column.size', value=len(obj.column_names()))

        # initialize members
        self.__child_views = {}
        self.__child_identifiers = {}
        self.__columns = {}

        # initialize parameters
        if not('columns' in self.params) or self.params['columns'] is None:
            # if user does not specify columns, pick all by default
            self.params['columns'] = self.obj.column_names()

            # if they specify x/y columns, make sure those are added
            if 'x' in self.params and \
               self.params['x'] is not None and \
               not(self.params['x'] in self.params['columns']):
                self.params['columns'].append(self.params['x'])
            if 'y' in self.params and \
               self.params['y'] is not None and \
               not(self.params['y'] in self.params['columns']):
                self.params['columns'].append(self.params['y'])
        else:
            column_names = set(self.obj.column_names())
            #validate that requested column identifiers are available
            for col in self.params['columns']:
                if not(col in column_names):
                    raise ValueError('Column "%s" not found in this SFrame.' % col)

        # initialize plot views
        self.scatter = graphlab.canvas.views.scatter.ScatterPlot(self.obj, self.params)
        self.heatmap = graphlab.canvas.views.heatmap.HeatmapView(self.obj, self.params)
        self.bar = graphlab.canvas.views.bar.BarGraphView(self.obj, self.params)

        # register SFrame-wide (Table, Summary) handlers
        self.register_handler('post', 'plot_params', self.__post_plot_params)
        self.register_handler('post', 'columns', self.__post_columns)
        self.register_handler('get', 'rows', self.__get_rows)
        self.register_handler('get', 'sketch/.*', self.__get_sketch)
        self.register_handler('get', 'cancel_sketch/.*', self.__cancel_sketch)
        self.register_handler('get', 'cached_sketches', self.__get_cached_sketches)

        # handlers for Scatter view
        self.register_handler('get', 'scatter_values', self.scatter._get_values)
        self.register_handler('get', 'sketchvalues', self.scatter._get_sketchvalues)
        self.register_handler('get', 'coefficients', self.scatter._get_coefficients)

        # handlers for Heat Map
        self.register_handler('get', 'bins', self.heatmap._get_bins)

        # handlers for BarGraph view
        self.register_handler('get', 'bar_values', self.bar._get_values)
        self.register_handler('get', 'extrema', self.bar._get_extrema)

        # if plot picked columns but were not specified in params, put into params
        if 'view' in self.params and self.params['view'] is not None:
            view = self.__get_view_instance()
            if view is not None:
                if not('x' in self.params) or self.params['x'] is None:
                    self.params['x'] = view.col1
                if not('y' in self.params) or self.params['y'] is None:
                    self.params['y'] = view.col2

        # materialize this sframe so __get_content_identifier__ returns valid results
        self.obj.__materialize__()

    def __check_identifier(self, col):
        # need to check identifier against real column (not cached)
        # in case the column itself has changed
        sa = self.obj[col]

        # do not check identifiers on special SGraph columns
        # they are different each time.
        import graphlab.data_structures.gframe # import here instead of at file-level to avoid circular dependency
        if isinstance(sa, graphlab.data_structures.gframe.GFrame):
            if not(col in self.__child_identifiers):
                self.__child_identifiers[col] = 0
                self.__child_views[col] = graphlab.canvas.views.sarray.SArrayView(sa)
            return

        ci = sa.__get_content_identifier__()
        if not(col in self.__child_identifiers):
            self.__child_identifiers[col] = ci
            self.__child_views[col] = graphlab.canvas.views.sarray.SArrayView(sa)
        elif self.__child_identifiers[col] != ci:
            # mismatch, delete cached objects
            self.__columns[col] = sa
            self.__child_identifiers[col] = ci
            self.__child_views[col] = graphlab.canvas.views.sarray.SArrayView(sa)

    def __get_column(self, col):
        # memoize access to columns on SFrame
        if not(col in self.__columns):
            self.__columns[col] = self.obj[col]
        return self.__columns[col]

    def __post_columns(self, url, handler):
        columns = handler.get_argument('columns')
        columns = map(str, json.loads(columns))
        self.params['columns'] = columns

    def __get_view_instance(self):
        views = {
            'Scatter Plot': self.scatter,
            'Heat Map': self.heatmap,
            'Bar Chart': self.bar,
            'Line Chart': self.scatter
        }
        if self.params['view'] in views:
            return views[self.params['view']]
        return None

    def __post_plot_params(self, url, handler):
        view = handler.get_argument('view')
        col1 = handler.get_argument('col1').encode('utf-8')
        col2 = handler.get_argument('col2').encode('utf-8')
        self.params['view'] = view
        plot = self.__get_view_instance()
        plot.col1 = self.params['x'] = col1
        plot.col2 = self.params['y'] = col2

    def __expand_columns(self):
        # given an SFrame, expands columns into a structure like:
        # {'str': ['col1', 'col3'], 'int': ['col2']}
        columns = []
        for (name,dtype) in zip(self.obj.column_names(), self.obj.column_types()):
            columns.append({
                'name': name,
                'dtype': dtype.__name__
            })
        return columns

    def __get_row_values(self, start, end, columns):
        # build a table (2d-array) of values,
        # like orient='values' in Pandas to_json
        # except we will substitute placeholder string values for
        # vector types (array, list, dict)
        sf = self.obj[columns][start:end]
        return [[graphlab.canvas.views.base._encode_value(row[col]) \
                for col in columns] \
                for row in sf]

    def __get_rows(self, url, handler):
        m = re.match('rows/(.*)/(.*)', url)
        columns = self.params['columns']
        start = int(m.group(1))
        end = int(m.group(2))
        handler.write({
            'values': self.__get_row_values(start, end, columns)
        })

    def __get_sketch(self, url, handler):
        m = re.match('sketch/(.*)', url)
        col = m.group(1).encode('utf-8')
        childView = self.child_views()[col]
        handler.write(childView.get_sketch())

    def __cancel_sketch(self, url, handler):
        m = re.match('cancel_sketch/(.*)', url)
        col = m.group(1).encode('utf-8')
        childView = self.child_views()[col]
        childView.cancel_sketch()
        handler.write({})

    def __get_cached_sketches(self, url, handler):
        child_views = self.child_views()
        handler.write({col: child_views[col].get_sketch(cached_only=True) for col in self.params['columns']})

    def __update_child_views(self):
        column_names = set(self.params['columns'])
        previous_column_names = set(self.__child_identifiers.keys())
        # check existing or generate new cached identifiers and child views
        map(self.__check_identifier, column_names)

    def child_views(self):
        # lazily initialize SArrayView objects for children and store them here
        # so that we can share cached sketches with SArrayView
        self.__update_child_views()
        return self.__child_views

    def get_metadata(self):
        self.__update_child_views()
        return {
            'descriptives': {
                'rows': len(self.obj),
                'columns': len(self.obj.column_names())
            },
            'columns': self.__expand_columns(),
            'column_identifiers': list(self.__child_identifiers),
            'view_params': self.params
        }

    def get_staticdata(self):
        self.__update_child_views()
        columns = self.params['columns']
        data = {
            'columns': self.__expand_columns()
        }
        if self.get_js_component() == 'Summary':
            data.update({
                'sketch': dict(map(lambda x: (x, self.__child_views[x].get_sketch(background=False)), columns))
            })
        elif self.get_js_component() == 'Plots':
            if self.params['view'] == 'Heat Map':
                data.update(self.heatmap.get_staticdata())
            elif self.params['view'] == 'Scatter Plot' or \
                 self.params['view'] == 'Line Chart':
                data.update(self.scatter.get_staticdata())
            elif self.params['view'] == 'Bar Chart':
                data.update(self.bar.get_staticdata())
        return data

    def get_js_file(self):
        return 'sframe'

    def __in_ipython(self):
        import graphlab.canvas.target
        return isinstance(graphlab.canvas.get_target(), graphlab.canvas.target.IPythonTarget)

    def get_js_component(self):
        if 'view' in self.params:
            name = self.params['view']
            if name == 'Summary' or \
               name == 'Table':
                return name # go to top-level tabs
            elif name is not None:
                self.validate_js_component_name(name)
                return 'Plots' # go straight to plot within plots tab
        return 'Summary'

    def get_js_components(self):
        return [
            'Summary',
            'Table',
            'Bar Chart',
            'Line Chart',
            'Scatter Plot',
            'Heat Map',
            'Plots'
        ]

    def validate_js_component_name(self, name):
        super(SFrameView, self).validate_js_component_name(name)
        if self.__in_ipython():
            if name == 'Table':
                raise ValueError('%s view of SFrame is not supported in ipynb target.' % name)
