"""
This package contains methods for evaluating the quality of predictive machine
learning models.
"""
import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.data_structures.sarray import SArray as _SArray
from graphlab.toolkits._internal_utils import _raise_error_if_not_sarray

def _supervised_evaluation_error_checking(targets, predictions):
    """ 
        Perform basic error checking for the evaluation metrics. Check
        types and sizes of the inputs.

    """
    _raise_error_if_not_sarray(targets, "targets")
    _raise_error_if_not_sarray(predictions, "predictions")
    assert targets.size() == predictions.size(), \
         "Input SArrays 'targets' and 'predictions' must be of the same length."

def max_error(targets, predictions):
    r"""
    Compute the maximum absolute deviation between two SArrays.

    Parameters
    ----------
    targets : SArray[float or int]
        An Sarray of ground truth target values.

    predictions : SArray[float or int]
        The prediction that corresponds to each target value.
        This vector must have the same length as ``targets``.

    Returns
    -------
    out : float
        The maximum absolute deviation error between the two SArrays.

    See Also
    --------
    rmse

    Notes
    -----
    The maximum absolute deviation between two vectors, x and y, is defined as:

    .. math::

        \textrm{max error} = \max_{i \in 1,\ldots,N} \|x_i - y_i\|

    Examples
    --------
    >>> targets = graphlab.SArray([3.14, 0.1, 50, -2.5])
    >>> predictions = graphlab.SArray([3.1, 0.5, 50.3, -5])
    >>> graphlab.evaluation.max_error(targets, predictions)
    2.5
    """

    _mt._get_metric_tracker().track('evaluation.max_error')
    _supervised_evaluation_error_checking(targets, predictions)
    return _graphlab.extensions._supervised_streaming_evaluator(targets, 
                                                    predictions, "max_error")

def rmse(targets, predictions):
    r"""
    Compute the root mean squared error between two SArrays.

    Parameters
    ----------
    targets : SArray[float or int]
        An Sarray of ground truth target values.

    predictions : SArray[float or int]
        The prediction that corresponds to each target value.
        This vector must have the same length as ``targets``.

    Returns
    -------
    out : float
        The RMSE between the two SArrays.

    See Also
    --------
    max_error

    Notes
    -----
    The root mean squared error between two vectors, x and y, is defined as:

    .. math::

        RMSE = \sqrt{\frac{1}{N} \sum_{i=1}^N (x_i - y_i)^2}

    References
    ----------
    - `Wikipedia - root-mean-square deviation
      <http://en.wikipedia.org/wiki/Root-mean-square_deviation>`_

    Examples
    --------
    >>> targets = graphlab.SArray([3.14, 0.1, 50, -2.5])
    >>> predictions = graphlab.SArray([3.1, 0.5, 50.3, -5])
    >>> graphlab.evaluation.rmse(targets, predictions)
    1.2749117616525465
    """

    _mt._get_metric_tracker().track('evaluation.rmse')
    _supervised_evaluation_error_checking(targets, predictions)
    return _graphlab.extensions._supervised_streaming_evaluator(targets, 
                                                          predictions, "rmse")

def confusion_matrix(targets, predictions):
    r"""
    Compute the confusion matrix for classifier predictions.

    Parameters
    ----------
    targets : SArray
        Ground truth class labels. 

    predictions : SArray
        The prediction that corresponds to each target value.
        This vector must have the same length as ``targets``.

    Returns
    -------
    out : SFrame 
        An SFrame containing counts for 'target_label', 'predicted_label' and 
        'count' corresponding to each pair of true and predicted labels.

    See Also
    --------
    accuracy

    Examples
    --------
    >>> targets = graphlab.SArray([0, 1, 1, 0])
    >>> predictions = graphlab.SArray([0.1, 0.35, 0.7, 0.99])
    >>> graphlab.evaluation.confusion_matrix(targets, predictions)
    """

    _mt._get_metric_tracker().track('evaluation.confusion_matrix')
    _supervised_evaluation_error_checking(targets, predictions)
    return _graphlab.extensions._supervised_streaming_evaluator(targets, 
                                              predictions, "confusion_matrix")

def accuracy(targets, predictions):
    r"""
    Compute the proportion of correct predictions.

    Parameters
    ----------
    targets : SArray
        Ground truth class labels. 

    predictions : SArray
        The prediction that corresponds to each target value.
        This vector must have the same length as ``targets``.

    Returns
    -------
    out : float
        The ratio of the number of correct classifications and the total number
        of data points.

    See Also
    --------
    confusion_matrix

    Examples
    --------
    >>> targets = graphlab.SArray([0, 1, 1, 0])
    >>> predictions = graphlab.SArray([0.1, 0.35, 0.7, 0.99])
    >>> graphlab.evaluation.accuracy(targets, predictions)
    """
    _mt._get_metric_tracker().track('evaluation.accuracy')
    _supervised_evaluation_error_checking(targets, predictions)
    return _graphlab.extensions._supervised_streaming_evaluator(targets, 
                                                    predictions, "accuracy")
