import atexit
from copy import copy
import functools
import logging as _logging
import os
from requests import get, post, ConnectionError
from subprocess import Popen
from time import sleep
from datetime import datetime, timedelta
import urllib
from subprocess import Popen, PIPE, STDOUT
import boto.ec2.elb
import boto.ec2.cloudwatch
from boto.exception import BotoServerError
import boto.iam.connection
from urllib import quote_plus
import json

try:
    from subprocess import DEVNULL # py3k
except ImportError:
    DEVNULL = open(os.devnull, 'wb')

from _file_util import parse_s3_path, s3_recursive_delete
from graphlab.connect.aws._ec2 import _ec2_factory, get_credentials as _get_credentials, \
    _get_ec2_instances, _stop_instances, GRAPHLAB_NAME

from _predictive_client import PredictiveServiceClient
from _predictive_service import MAX_CREATE_TIMEOUT_SECS

_logger = _logging.getLogger(__name__)

ENV_TYPE = 'environment type'
# The maximum number of datapoints CloudWatch will return for a given query
CW_MAX_DATA_POINTS = 1440
# The minimum allowable period (seconds) for CloudWatch
CW_MIN_PERIOD = 60

# disable spurious output from requests library
_logging.getLogger("requests").setLevel(_logging.WARNING)

def predictive_service_environment_factory(env_info):
    env_type = env_info[ENV_TYPE]
    del env_info[ENV_TYPE]

    assert(env_type in ['TestStubEnv', 'LocalPredictiveServiceEnvironment', 'Ec2PredictiveServiceEnvironment'])
    if(env_type == 'LocalPredictiveServiceEnvironment'):
        return LocalPredictiveServiceEnvironment(**env_info)
    elif(env_type == 'Ec2PredictiveServiceEnvironment'):
        return Ec2PredictiveServiceEnvironment(**env_info)
    else:
        return TestStubEnv(**env_info)

class PredictiveServiceEnvironment:
    _PORT_NUM = 9005

    def _poke_host(self, host_name):
        _logger.info("Notifying: %s" % host_name)
        data = json.dumps({'admin key': self.admin_key})
        url = 'http://%s:%d/control/poke' % (host_name, PredictiveServiceEnvironment._PORT_NUM)
        self._post(url, data=data)

    def __init__(self, admin_key):
        self.admin_key = admin_key
        self.certificate_name = None

        self._request_schema = None
        self._should_verify_certificate = None

    def get_status(self):
        pass

    def launch(self):
        pass

    def poke(self):
        pass

    def query(self, po_name, method_params, api_key):
        if not hasattr(self, 'client_connection'):
            if self.certificate_name:
                schema = 'https://'
            else:
                schema = 'http://'

            endpoint = schema + self.load_balancer_dns_name
            verify_cert = not self.certificate_is_self_signed
            self.client_connection = PredictiveServiceClient(endpoint= endpoint, api_key=api_key,
                                                             should_verify_certificate = verify_cert)

        return self.client_connection.query(po_name, method_params)

    def get_metrics_url(self, ps_name):
        pass

    def _get_metrics(self, name, start_time, end_time, period):
        pass

    def terminate(self, del_flag):
        pass

    def _post(self, url, data):
        if not self._should_verify_certificate:
            self._should_verify_certificate = self.__should_verify_certificate()

        headers = {'content-type': 'application/json'}
        return post(url=url, data=data, headers=headers,
                    verify=self._should_verify_certificate, timeout=10)

    @classmethod
    def __make_url(cls, host_dns_name, endpoint):
        return "http://%s:%d/%s" % (host_dns_name,
                                    PredictiveServiceEnvironment._PORT_NUM,
                                    endpoint)

    def __cache_op_params(self, name, restart):
        data = {"admin key": self.admin_key}

        if name:
            data.update({"name": name})

        if restart:
            data.update({"restart": restart})

        return data

    def __host_addrs(self):
        if isinstance(self, Ec2PredictiveServiceEnvironment):
            return [inst.public_dns_name for inst in self._get_all_hosts()]
        else:
            return ["localhost"]

    def cache_enable(self, name, restart=True):
        for host in self.__host_addrs():
            try:
                url = self.__make_url(host, "control/cache_enable")

                params = self.__cache_op_params(name, restart)

                self._post(url, json.dumps(params))
            except Exception as e:
                _logger.error('Could not enable cache on %s: %s' %
                              (host, e.message))

    def cache_disable(self, name):
        for host in self.__host_addrs():
            try:
                url = self.__make_url(host, "control/cache_disable")

                params = self.__cache_op_params(name, False)

                self._post(url, json.dumps(params))
            except Exception as e:
                _logger.error('Could not disable cache on %s: %s' %
                              (host, e.message))

    def cache_clear(self, name):
        for host in self.__host_addrs():
            try:
                url = self.__make_url(host, "control/cache_clear")

                params = self.__cache_op_params(name, False)

                self._post(url, json.dumps(params))
            except Exception as e:
                _logger.error('Could not clear cache on %s: %s' %
                              (host, e.message))

    def flush_logs(self):
        for host in self.__host_addrs():
            try:
                url = self.__make_url(host, "control/flush_logs")
                self._post(url, json.dumps({"admin key": self.admin_key}))
            except Exception as e:
                pass

    def get_node_status(self, host_addr):
        url = self.__make_url(host_addr, "control/status")
        response = self._post(url, json.dumps(
            {"admin key": self.admin_key}))
        try:
            _logger.info(response.text)
            return json.loads(response.text)
        except Exception as e:
            return {"error": "Cannot get status for host %s, error: %s" %
                    (host_addr, e.message)}

    def get_service_status(self):
        status = {}
        for host in self.__host_addrs():
            status.update({host: self.get_node_status(host)})
        return status

    def _get_model_versions_for_host(self, host_dns_name):
        data = json.dumps({'admin key': self.admin_key})
        url = "http://%s:%d/control/list_objects" % (host_dns_name,
                                                     PredictiveServiceEnvironment._PORT_NUM)
        resp = self._post(url, data=data)
        try:
            response = resp.text
            return json.loads(response)
        except Exception as e:
            return {'error': 'Cannot get model info from host, error: %s' % resp.text}

    def __is_using_certificates(self):
        return hasattr(self, 'certificate_name') and self.certificate_name

    def __get_schema(self):
        if not self._request_schema:
            self._request_schema = 'https' if self.__is_using_certificates() else 'http'
        return self._request_schema

    def __should_verify_certificate(self):
        return self.__is_using_certificates() and not (hasattr(self, 'certificate_is_self_signed') and self.certificate_is_self_signed)

class LocalPredictiveServiceEnvironment(PredictiveServiceEnvironment):
    def __init__(self, log_path, admin_key, num_hosts = 1, redis_manager = None,
                 web_server = None, graphlab_service = None, **kwargs):
        PredictiveServiceEnvironment.__init__(self, admin_key)
        self.log_path = log_path
        self.num_hosts = int(num_hosts)

        self.redis_manager = redis_manager
        self.web_server = web_server
        self.graphlab_service = graphlab_service

        self.load_balancer_dns_name = 'localhost:%d' % PredictiveServiceEnvironment._PORT_NUM

        self.certificate_is_self_signed = True

    def _get_state(self):
        result = {}
        result[ENV_TYPE] = 'LocalPredictiveServiceEnvironment'
        result['num_hosts'] = self.num_hosts
        result['log_path'] = self.log_path
        result['admin_key'] = self.admin_key
        return result

    def get_status(self):
        return [ { 'id': 'localhost', 'dns_name': 'localhost', 'state':'InService',
                   'models': self._get_model_versions_for_host('localhost') } ]

    def terminate(self, delete_logs):
        _logger.info('Terminating service.')
        try:
            if self.web_server:
                _logger.info('Terminating web server.')
                self.web_server.terminate()
        except:
            pass

        try:
            if self.graphlab_service:
                _logger.info('Terminating graphlab_service.')
                self.graphlab_service.terminate()
        except:
            pass

        try:
            if self.redis_manager:
                _logger.info('Terminating redis manager.')
                self.redis_manager.terminate()
        except:
            pass

        if delete_logs:
            _logger.info('Deleting log files.')
            try:
                s3_recursive_delete(self.log_path)
            except:
                _logger.info("Could not delete log file. Please manually delete files under: %s"
                             % self.log_path)

    @staticmethod
    def launch(predictive_service_path, log_path, num_hosts, admin_key):
        node_manager_dir = os.environ['NODE_MANAGER_ROOT']

        # Start Redis for caching
        env_vars = copy(os.environ)
        env_vars['PYTHONPATH'] = os.pathsep.join([env_vars['PYTHONPATH'], '.'])
        env_vars['S3_PREDICTIVE_SERVICE'] = predictive_service_path
        env_vars['LOG_PATH'] = log_path

        env_vars['REDIS_TRIB_PATH'] = os.environ['REDIS_TRIB_PATH']
        env_vars['REDIS_SERVER_PATH'] = os.environ['REDIS_SERVER_PATH']

        redis_manager = Popen(
            ['python', 'redismanager/redis_manager.py', str(max(3, num_hosts))],
            cwd = node_manager_dir, env = env_vars,  stdout=DEVNULL, stderr=STDOUT)

        _logger.info(
            "Running Redis manager with PID: %d" % (redis_manager.pid))

        # Start the Predictive Service
        web_server = Popen(
            ['python', 'psws/ps_server.py'],
              cwd = node_manager_dir, env = env_vars,  stdout=DEVNULL, stderr=STDOUT)

        _logger.info(
            "Running web server with PID: %d" % (web_server.pid))

        # Start GraphLab Service
        graphlab_service = Popen(
            ['python', 'glservice/graphlab_service.py'],
            cwd = node_manager_dir, env = env_vars,  stdout=DEVNULL, stderr=STDOUT)

        _logger.info(
            "Running GraphLab service with PID: %d" % (graphlab_service.pid))

        instance = LocalPredictiveServiceEnvironment(
            log_path, admin_key, num_hosts, redis_manager = redis_manager,
            web_server = web_server, graphlab_service = graphlab_service)

        atexit.register(functools.partial(instance.terminate, False))

        return instance

    def poke(self):
        self._poke_host('localhost')


class Ec2PredictiveServiceEnvironment(PredictiveServiceEnvironment):
    def __init__(self, load_balancer_dns_name, region, log_path, admin_key,
                 certificate_name, certificate_is_self_signed, aws_credentials):
        PredictiveServiceEnvironment.__init__(self, admin_key)
        self.load_balancer_dns_name = load_balancer_dns_name
        self.region = region
        self.log_path = log_path
        self.certificate_name = certificate_name
        self.certificate_is_self_signed = certificate_is_self_signed
        self.aws_credentials = aws_credentials

    def __repr__(self):
        ret = ""
        ret += 'Ec2PredictiveServiceEnvironment:\n'
        ret += '\tload_balancer_dns_name: %s\n' % str(self.load_balancer_dns_name)
        ret += '\tregion: %s\n' % str(self.region)
        ret += '\tlog_path: %s\n' % str(self.log_path)
        ret += '\tcertificate_name: %s\n' % str(self.certificate_name)
        if self.certificate_name:
            ret += '\tcertificate_is_self_signed: %s\n' % str(self.certificate_is_self_signed)
        return ret

    def __str__(self):
        return self.__repr__()

    @staticmethod
    def launch(name, ec2_config, s3_predictive_object_path, admin_key, ssl_credentials,
               aws_credentials, started):

        s3_log_path = "s3://%s/%s%s_logs" % (ec2_config.s3_bucket, ec2_config.s3_log_folder_path,
                                              name) # s3_log_folder_path always ends in '/'
        user_data = {
            'log_path': s3_log_path,
            'aws_access_key': aws_credentials['aws_access_key_id'],
            'aws_secret_key': aws_credentials['aws_secret_access_key'],
            's3_predictive_service': s3_predictive_object_path
            }

        _logger.info("[Step 1/5]: Launching EC2 instances.")

        # Start the hosts up.
        ec2_hosts = _ec2_factory(ec2_config.instance_type, region = ec2_config.region,
                                 CIDR_rule = ec2_config.CIDR_rule,
                                 security_group_name = ec2_config.security_group,
                                 tags = ec2_config.tags, user_data = user_data,
                                 credentials = aws_credentials, num_hosts = ec2_config.num_hosts,
                                 ami_service_parameters = {'service': 'predictive'})

        lb = None
        try:
            # Determine host ids and availability zones used.
            zones, host_ids, is_vpc = set(), [], False
            if ec2_config.num_hosts == 1:
                host_ids = [ec2_hosts.instance_id]
                zones.add(ec2_hosts.instance.placement)
                is_vpc = bool(ec2_hosts.instance.vpc_id)
            else:
                for i in ec2_hosts:
                    zones.add(i.instance.placement)
                    host_ids.append(i.instance_id)

                is_vpc = bool(ec2_hosts[0].instance.vpc_id)

            certificate_name, certificate_is_self_signed = None, None
            if ssl_credentials:
                # Using HTTPS
                (private_key_path, public_certificate_path, certificate_is_self_signed) = ssl_credentials
                certificate_name = name
                certificate_id = Ec2PredictiveServiceEnvironment._upload_ssl_info(certificate_name,
                                                                                  private_key_path,
                                                                                  public_certificate_path,
                                                                                  aws_credentials)
                listener_tuple = (443, Ec2PredictiveServiceEnvironment._PORT_NUM, 'https', certificate_id)
            else:
                # Using HTTP
                listener_tuple = (80, Ec2PredictiveServiceEnvironment._PORT_NUM, 'http')

            conn = boto.ec2.elb.connect_to_region(ec2_config.region, **aws_credentials)

            # Create the load balancer.
            _logger.info("[Step 2/5]: Launching Load Balancer.")

            while ((datetime.now() - started).total_seconds() < MAX_CREATE_TIMEOUT_SECS):
                try:
                    lb = conn.create_load_balancer(name, zones, [listener_tuple])
                    break
                except BotoServerError as e:
                    # We just uploaded the certificate, so there's a good chance it will not be found, yet.
                    if "<Code>CertificateNotFound</Code>" not in str(e):
                        raise e
                    sleep(1)
            else:
                raise RuntimeError("Unable to successfully create load balancer. Please confirm in AWS Management Console")


            # If we're running in a VPC (i.e. not "EC2-Classic") we need to attach a security group.
            if is_vpc:
                # Look up security group id
                security_group = None
                while ((datetime.now() - started).total_seconds() < MAX_CREATE_TIMEOUT_SECS):
                    ec2_conn = boto.ec2.connect_to_region(ec2_config.region, **aws_credentials)
                    for sg in ec2_conn.get_all_security_groups():
                        if(GRAPHLAB_NAME == sg.name):
                            security_group = sg
                            break   # found it
                    if security_group:
                        break # no more eventual consistency
                    else:
                        sleep(1)
                else:
                    raise RuntimeError("Unable to find expected security group needed to configure load balancer.")
                lb.apply_security_groups([sg.id])

            _logger.info("[Step 3/5]: Configuring Load Balancer.")

            # Configure healthCheck
            health_target = "HTTP:%d/control/healthcheck" % PredictiveServiceEnvironment._PORT_NUM
            health_check = boto.ec2.elb.HealthCheck(interval=20, healthy_threshold=3,
                                                    unhealthy_threshold=5,
                                                    target= health_target)
            lb.configure_health_check(health_check)

            # Add EC2 instances to the load balancer.
            lb.register_instances(host_ids)

        except:
            _logger.error("Could not create or configure the load balancer, terminating EC2 instances.")
            _stop_instances([h.instance_id for h in ec2_hosts], ec2_config.region,
                            credentials = aws_credentials)

            if lb:
                _logger.info("Deleting the load balancer.")
                lb.delete()

            raise

        return Ec2PredictiveServiceEnvironment(lb.dns_name, ec2_config.region, s3_log_path, admin_key,
                                               certificate_name, certificate_is_self_signed,
                                               aws_credentials)

    def get_status(self, _show_errors = True):
        result = []
        load_balancer = self._get_load_balancer_handle()

        for host in load_balancer.get_instance_health():
            host_id = host.instance_id
            host_instance = _get_ec2_instances([host_id], self.region,
                                               aws_credentials = self.aws_credentials)

            dns_name = host_instance[0].public_dns_name

            try:
                model_data = self._get_model_versions_for_host(dns_name)
            except:
                if _show_errors:
                    _logger.error("Can not connect to %s" % dns_name)
                model_data = None

            result.append( { 'id': host_id, 'dns_name': dns_name, 'models': model_data,
                             'state': host.state, 'reason': host.reason_code } )
        return result

    def _get_metrics(self, name, start_time, end_time, period=300):
        # Calculate time range for period determination/sanity-check
        delta = end_time - start_time
        delta_seconds = ( delta.days * 24 * 60 * 60 ) + delta.seconds + 1 #round microseconds up

        # Determine min period as the smallest multiple of 60 that won't result in too many data points
        min_period = 60 * int(delta_seconds / CW_MAX_DATA_POINTS / 60)
        if ((delta_seconds / CW_MAX_DATA_POINTS) % 60) > 0:
            min_period += 60

        # set the min period if necessary
        if period < min_period:
            period = min_period
        if period < CW_MIN_PERIOD:
            period = CW_MIN_PERIOD

        # get cloud watch handle
        cw = self._get_cloudwatch_handle()

        # get AWS/ELB requests, latency, num of healthy hosts, and num of cache keys
        num_requests = cw.get_metric_statistics(period, start_time, end_time,
                        'RequestCount', 'AWS/ELB', ['Sum'], {"LoadBalancerName": name}, 'Count')

        latency = cw.get_metric_statistics(period, start_time, end_time,
                        'Latency', 'AWS/ELB', ['Average'], {"LoadBalancerName": name}, 'Seconds')

        num_healthy_hosts = cw.get_metric_statistics(period, start_time, end_time,
                        'HealthyHostCount', 'AWS/ELB', ['Average'], {"LoadBalancerName": name}, 'Count')

        num_keys_in_cache = cw.get_metric_statistics(period, start_time, end_time,
                        'num_keys_in_cache', name, ['Maximum'], {}, 'Count')

        return {'num_requests' : num_requests, 'latency' : latency, 'num_healthy_hosts' : num_healthy_hosts,
                'num_keys_in_cache' : num_keys_in_cache}


    def get_metrics_url(self, ps_name):
        return 'https://console.aws.amazon.com/cloudwatch/home?region=' \
               '%s#metrics:metricFilter=Pattern%%253D%s' % (self.region, quote_plus(ps_name))

    def _get_cloudwatch_handle(self):
        try:
            conn = boto.ec2.cloudwatch.connect_to_region(self.region, **self.aws_credentials)
            return conn
        except Exception as e:
            _logger.error("Unable to connect to to CloudWatch in region '%s'" % self.region)
        raise Exception("Cannot connect to CloudWatch in region '%s'." % self.region)

    def _get_load_balancer_handle(self):
        try:
            conn = boto.ec2.elb.connect_to_region(self.region, **self.aws_credentials)
            for i in conn.get_all_load_balancers():
                if i.dns_name == self.load_balancer_dns_name:
                    return i
        except Exception as e:
            _logger.error("Unable to connect to ELB with name '%s' in region '%s'"
                          % (self.load_balancer_dns_name, self.region))

        raise Exception("Cannot find load balancer with name '%s' in region '%s'."
                        % (self.load_balancer_dns_name, self.region))


    def _get_state(self):
        result = {}
        result[ENV_TYPE] = 'Ec2PredictiveServiceEnvironment'
        result['load_balancer_dns_name'] = self.load_balancer_dns_name
        result['region'] = self.region
        result['certificate_name'] = self.certificate_name
        result['log_path'] = self.log_path
        result['admin_key'] = self.admin_key
        result['certificate_is_self_signed'] = self.certificate_is_self_signed
        return result


    def poke(self):
        for instance in self._get_all_hosts():
            try:
                self._poke_host(instance.public_dns_name)
            except Exception as e:
                _logger.error('Could not notify %s: %s' % (instance.public_dns_name, e.message))


    def terminate(self, delete_logs):
        try:
            load_balancer_handle = self._get_load_balancer_handle()
            conn = boto.ec2.elb.connect_to_region(self.region, **self.aws_credentials)
            _logger.info("Deleting load balancer: %s" % load_balancer_handle.name)
            conn.delete_load_balancer(load_balancer_handle.name)
        except:
            _logger.error("Could not delete load balancer. Please manually delete the following load "
                      "balancer: %s" % self.load_balancer_dns_name)

        # Terminate all hosts
        try:
            host_ids = [i.id for i in load_balancer_handle.instances]
            _stop_instances(host_ids, self.region, credentials = self.aws_credentials)
        except:
            _logger.error("Could not terminate hosts. Please manually terminate from the AWS console.")

        if delete_logs:
            _logger.info('Deleting log files.')
            try:
                s3_recursive_delete(self.log_path, self.aws_credentials)
            except:
                _logger.info("Could not delete log file. Please manually delete files under: %s"
                             % self.log_path)

        # Delete the server certificate
        if self.certificate_name:
            conn = boto.iam.connection.IAMConnection(**self.aws_credentials)
            conn.delete_server_cert(self.certificate_name)


    def _get_all_hosts(self):
        load_balancer_handle = self._get_load_balancer_handle()

        host_ids = [i.id for i in load_balancer_handle.instances]
        return _get_ec2_instances(host_ids, self.region, self.aws_credentials)

    @staticmethod
    def _upload_ssl_info(certificate_name, private_key_path, public_certificate_path,
                         credentials):
        # Read in private key and public certificate
        with open(private_key_path, 'r') as file:
            private_key = file.read()
        with open(public_certificate_path, 'r') as file:
            cert = file.read()

        # Upload private key and public certificate
        conn = boto.iam.connection.IAMConnection(**credentials)
        resp = conn.upload_server_cert(certificate_name, cert, private_key)

        # Get the certificate id
        certificate_id =  resp['upload_server_certificate_response'] \
            ['upload_server_certificate_result']['server_certificate_metadata']['arn']

        return certificate_id


class TestStubEnv(PredictiveServiceEnvironment):
    def __init__(self, aws_credentials):
        PredictiveServiceEnvironment.__init__(self, 'test api key')
        self.load_balancer_dns_name = 'load_balancer_dns_name'

    def _get_state(self):
        return {ENV_TYPE: 'TestStubEnv'}

    def _get_metrics(self, name, start_time, end_time, period):
        num_requests = []
        latency = []
        num_healthy_hosts = []
        num_keys_in_cache = []
        time_delta = timedelta(seconds=period)
        start = start_time
        while start < end_time:
            num_requests.append({'Sum': 1.0, 'Timestamp': start, 'Unit':'Count'})
            latency.append({'Average': 1.0, 'Timestamp': start, 'Unit':'Seconds'})
            num_healthy_hosts.append({'Average': 1.0, 'Timestamp': start, 'Unit':'Count'})
            num_keys_in_cache.append({'Sum': 1.0, 'Timestamp': start, 'Unit':'Count'})
            start += time_delta
        return {'num_requests': num_requests, 'latency': latency, 'num_healthy_hosts': num_healthy_hosts,
                        'num_keys_in_cache': num_keys_in_cache}

    def get_metrics_url(self, ps_name):
        return 'test'

    def poke(self):
        pass
