import __builtin__
import collections as _collections
import tempfile as _tempfile
import shutil as _shutil
import imp as _imp
import copy as _copy
import os as _os
import fnmatch as _fnmatch
import logging

import inspect as _inspect
from inspect import isfunction as _isfunction

import _timer as _timer
import graphlab as _gl

__LOGGER__ = logging.getLogger(__name__)


class Task(object):
    """
    A Task encapsulates a block of code.

    A Task is a modular, composable, reusable component that encapsulates a
    unit-of-work. It is the robust building block for building GraphLab Data
    Pipelines.

    Tasks allow for code to logically reference inputs, outputs, and parameters
    by name, so the code is reusable in many places.

    Tasks make it easy to build a Data Pipeline of composable parts, each
    having a single responsibility in the pipeline, with the tasks loosely
    coupled. All inputs, outputs, and parameters can be bound at runtime.

    Parameters
    ----------
    name : str
        Name of the Task, this is how the Task will be retrieved in subsequent
        retrievals.

    description : str, optional
        Description for this Task, optional.

    Returns
    -------
    out : :class:`~graphlab.deploy.Task`
        A new Task.

    Notes
    -----
    - Task names must be unique.

    Examples
    --------
    An example Task definition, for generating random names and building an
    :class:`~graphlab.SFrame` from them:

    >>> def name_task(task):
    >>>    import names
    >>>    people = [names.get_full_name() for i in range(task.params['number'])]
    >>>    sf = graphlab.SFrame({'names':people})
    >>>    task.outputs['names'] = sf
    >>>    print sf
    ... 
    >>> names = graphlab.deploy.Task('names')
    >>> names.set_required_packages(['names == 0.3.0'])
    >>> names.set_code(name_task)
    >>> names.set_params({'number':10})
    >>> names.set_outputs(['names'])
    """

    def __init__(self, name, description=None):
        """
        Create a new Task specifying its name and optionally a description.
        """
        if name is None or name == "":
            raise TypeError("Name is required when creating a Task.")

        if not isinstance(name, str):
            raise TypeError("Name is required to be a string.")

        self._session = _gl.deploy._default_session

        self.name = name
        self._data = dict()
        self._data['code'] = None
        self._data['codestr'] = None
        self._data['inputs'] = dict()
        self._data['outputs'] = dict()
        self._data['params'] = dict()
        self._data['packages'] = set()
        self._data['description'] = ''
        self._modified_since_last_saved = None
        self.metrics = {}

        if description is not None:
            self.set_description(description)

        self._session.register(self)

    def set_name(self, name):
        """
        Set the name of the Task, which must be unique.

        Parameters
        ----------
        name : str
            Name of the Task.

        Returns
        -------
        self : Task
        """
        if not isinstance(name, str):
            raise TypeError('Name must be a string')

        self.name = str(name)
        self._set_dirty_bit()
        return self

    def set_description(self, description):
        """
        Set the description for this Task.

        Parameters
        ----------
        description : str
            A description for the Task.

        Returns
        -------
        self : Task
        """
        if not isinstance(description, str):
            raise TypeError('Description must be a string')

        self._data['description'] = str(description)
        self._set_dirty_bit()
        return self

    def clone(self, name):
        """
        Make a copy of the current Task with a new name. The new task will have
        a different name, but will share all other task metadata like inputs,
        outputs, description, and code.

        Parameters
        ----------
        name : str
            Name of the cloned Task.

        Returns
        -------
        out
            A newly cloned Task instance.
        """
        return self._clone(name, session_aware=True)

    def _clone(self, name, session_aware=True):
        new = Task.__new__(Task)
        new.name = name
        new._data = _copy.deepcopy(self._data)
        new._session = self._session
        new._modified_since_last_saved = None
       
        if session_aware is True:
            self._session.register(new)

        return new

    def set_inputs(self, names, delete=False):
        """
        Set input(s) for this Task.

        Inputs are required to be GraphLab Data structures
        (:class:`~graphlab.SFrame`, :class:`~graphlab.SGraph`,
        :class:`~graphlab.SArray`, Model), and must come from other Tasks. That
        means each input must be defined as a dependency to another Task's
        output.

        Parameters
        ----------
        names : list [str] | dict [str, (str|Task, str) | (str|Task)]
            Input Tasks can be referenced either by name or by reference and
            the name for the slot in which that data can be found in the input
            Task can be made explicit or left implicit ('output'). If a dict is
            provided, then each key is considered a name for an input in this
            Task, and each value is considered the definition of the input.
            When a list is provided,  then each entry is considered a name for
            an input in this Task, and the value for that slot is set to None.

        delete : bool, optional
            If delete is set to True then the named parameters are removed.

        Returns
        -------
        self : Task

        See Also
        --------
        set_outputs

        Examples
        --------
        To define only input names for a task, use a list of strings:

        >>> t1 = graphlab.deploy.Task('set_inputs_ex1')
        >>> t1.set_inputs(['one', 'two', 'three'])

        Values for the dict can be either of the following:

        - a pair representing a Task, and its corresponding output that we wish
          to pass along as input

        - or simply a Task, in which case the output name is assumed to be
          'output'
        
        Tasks can be referred to either by name (a string) or directly by
        reference. Example of different valid dict input definitions:

        >>> t2 = graphlab.deploy.Task('set_inputs_ex2')
        >>> t2.set_outputs(['out', 'output', 'dw'])
        
        >>> t3 = graphlab.deploy.Task('set_inputs_ex3')
        >>> t3.set_inputs({
        ...     # Task named "set_inputs_ex2", output named "out"
        ...     'a' : ('set_inputs_ex2', 'out'),
        ...     # Task named "set_inputs_ex2", implied output named "output"
        ...     'b' : 'set_inputs_ex2', 
        ...     # Task t2, implied output named "output"
        ...     'c' : t2, 
        ...     # Task t2, output named "dw"
        ...     'd' : (t2, 'dw')}) 
        """
        if names is None:
            raise TypeError('names parameter is required for set_inputs')

        if isinstance(names, list):
            for name in names:
                self._set_one_input(name=name, delete=delete)
        elif isinstance(names, dict):
            for key, value in names.iteritems():
                self._set_one_input(name=key, from_task=value, delete=delete)
        else:
            raise TypeError('Unknown type for names parameter. names parameter can be a list or a dict')

        return self

    def _set_one_input(self, name='input', from_task=None, delete=False):
        """
        Set/Update an input for this Task.

        Parameters
        ----------

        name : str
            Name for this input. This will be how the code refers to this input at runtime.
            Default is 'input'.

        from_task : tuple of (Task|string, string)
            Dependent Task to set as input, specifying the tuple with: (Task,
            output_name). Tasks can be referred to either by name or by
            reference. The output_name needs to be a string. If no output_name
            is specified then 'output' is used.

            For example, if the following is specified:

            >>> task.set_input(name='in', from=('dep', 'out'))
            
            then an input named 'in' will be defined on this Task, which 
            has a dependency on the output named 'out' from Task named 'dep'.

            If input is specified using an object:
            
            >>> t1 = graphlab.deploy.Task('t1').set_output('out')
            >>> t2.set_input(name='in', from=(t1, 'out'))
            
            then an input for t2 would be defined as being the 'out' output 
            of t1.

        delete : bool, optional
            If delete is set to True then the name input is removed.
        """
        if delete is True and name in self._data['inputs']:
            del self._data['inputs'][name]
            return self

        if not isinstance(name, str):
            raise TypeError('name must be a string')
        
        self.__validate_slot_name_uniqueness(name, self._data['inputs'])

        if from_task is None:
            self._data['inputs'][name] = None
            self._set_dirty_bit()
            return self

        elif isinstance(from_task, tuple):
            task = from_task[0]
            output_name = from_task[1]

            if output_name is None or output_name == '':
                output_name = 'output'

            if isinstance(task, Task):
                return self._set_input(name, task=task, output_name=output_name)
            elif isinstance(task, str):
                loaded_task = _gl.deploy.tasks[task]
                if loaded_task is None:
                    raise TypeError("Unable to load the from_task Task object given the name '%s', please confirm that this Task exists." % from_task)
                return self._set_input(name, task=loaded_task, output_name=output_name)
            else:
                raise TypeError("Unable to parse from_task parameter, please specify as a tuple or a single object or string")
        else:
            task = from_task
            output_name = 'output'
            if isinstance(task, Task):
                return self._set_input(name, task=task, output_name=output_name)
            elif isinstance(from_task, str):
                loaded_task = _gl.deploy.tasks[task]
                return self._set_input(name, task=loaded_task, output_name=output_name)
            else:
                raise TypeError("Unable to parse from_task parameter, please specify as a tuple or a single object or string")

        raise Exception("Cannot set input given parameters. Please see documentation")

    def _set_input(self, name, task=None, output_name=None):
        """
        Set/Update an input for this Task, by name.

        To remove an input, simply provide its name with no other parameters.
    
        Parameters
        ----------
        name : str
            Name of the Task. This must be unique for this session.

        task : str
            Name of the Task to use as an input.

        output_name : str
            Name of output from input Task. If Task or pipeline is set, this is required.

        Returns
        -------
        Returns itself, so can be used in chaining operations (Builder pattern)
        """
        if name is None:
            raise AttributeError('Missing name')

        # TODO: Add Validation to confirm that file/task/pipeline exist with given names

        if task is None:
            raise AttributeError("task needs to be defined")

        self._data['inputs'][name] = (task.name, output_name)

        self._set_dirty_bit()
        return self

    def set_code(self, code, function_name='run', directory=None, pattern="*.py"):
        """
        Set the code block to run when Task is executed.

        The code to be run needs to be a function that takes one argument. When
        this function is called, the argument will be an object with three
        dictionary attributes (inputs, outputs, params). The inputs dictionary will
        have instantiated data sources by name. The outputs dictionary needs to
        be assigned by name to the results to save. The params dictionary
        allows capturing any additional keys/values with the Task.
        
        Parameters
        ----------
        code : function | string 
            Function to be called when this Task is executed. This can
            either be a defined function or it can be a full file path. As a
            file path this method will inspect the file and confirm it has a
            function named according to the parameter function_name, which must
            take only one argument. This function will then also take, by
            default, the working directory of that file and run find_packages()
            on it, collecting all the source files indicated. If code is set to
            None, then this Task's code is removed.

        function_name : str, optional
            Name of the function to call. Default value is 'run'.

        directory : os.path, optional
            Optional parameter for the directory from which to grab the pattern
            of files.
        
        pattern : str, optional
            Optional parameter for a "glob" pattern (ex. *.py) to specify the
            subset of files from which code should be captured. By default, all
            .py files from the same directory as the file containing the
            specified function are captured. This pattern is matched using
            ``fnmatch``, so rules from fnmatch documentation apply.

        Returns
        -------
        self : Task

        Examples
        --------
        Using a defined function:

        >>> def func(task):
        >>>     input = task.inputs['input']
        >>>     task.outputs['output'] = input.apply(lambda x : x * 2)
        
        >>> t1 = graphlab.deploy.Task("set_code_ex1")
        >>> t1.set_code(func)
        
        Another example, with different named tasks:

        >>> t2 = graphlab.deploy.Task('set_code_ex2')
        >>> t2.set_inputs(['one', 'two', 'three'])
        >>> t2.set_outputs(['result_one', 'result_two'])

        >>> def func(task):
        >>>     one = task.inputs['one']
        >>>     two = task.inputs['two']
        >>>     three = task.inputs['three']
        >>>     ...
        >>>     task.outputs['result_one'] = first_result
        >>>     task.outputs['result_two'] = second_result
        
        The example below illustrates how to specify code in a file:

        >>> t3 = graphlab.deploy.Task('set_code_ex3')
        >>> t3.set_code('/full/path/to/munge.py')
        
        Using a file with a custom fuction name, and custom directory and
        pattern to include:

        >>> t4 = graphlab.deploy.Task('set_code_ex4')
        >>> t4.set_code('/full/path/to/transform.py', function_name='to_csv', 
        >>>             directory='src', pattern='*.py')

        """
        if code is None:
            keys = ['code', 'codestr', 'funcname', 'filename']
            self._data = Task.__remove_unwanted_keys(self._data, keys)
            self._data['code'] = None
            return self

        if _isfunction(code):
            # code is callable, so store it as is
            self._data['code'] = code
            try:
                self._data['codestr'] = _inspect.getsource(code)
            except IOError:
                pass

        elif _inspect.ismethod(code):
            raise TypeError("Code cannot be an instance method, please use a function or a file")

        else:

            # specifying code using file path, and not function
            code = _os.path.expanduser(code)
            
            if not _os.path.exists(code) and directory is not None: 
                if not _os.path.exists(_os.path.join(directory, code)):
                    raise TypeError("Unable to find the full path for the code file specified, please verify the full path to the file to use. Location searched: %s" % _os.path.join(directory, code))
                else:
                    code = _os.path.join(directory, code)
                
            if _os.path.exists(_os.path.abspath(code)):

                # code is a file path to a python file, capture the source etc
                if directory is None:
                    directory = _os.path.join(_os.path.dirname(_os.path.abspath(code)))

                # confirm that function exists in file provided use getattr and try to load the function in the module
                module = _imp.load_source('', code)
                if not hasattr(module, function_name):
                    raise TypeError("Unable to find function name \"%s\" from the code file spcified, please verify the function name and the code file. Code file searched: %s" % (function_name, code))
                self._data['codestr'] = _inspect.getsource(getattr(module, function_name))

                files = {}
                for f in Task.__find_files(directory, pattern):
                    with open(f, 'r') as src:
                        rel_path = _os.path.relpath(f, start=directory)
                        files[rel_path] = src.read()

                self._data['funcname'] = function_name
                self._data['filename'] = _os.path.basename(code)

                # now all source files are referred to dictionary of files with filename
                self._data['code'] = files

            else:
                raise TypeError("Unable to find the full path for the code file specified, please verify the path is correct. Path searched: %s" % code)

        self._set_dirty_bit()
        return self

    @staticmethod
    def __find_files(directory, pattern):
        for root, dirs, files in _os.walk(directory):
            for basename in files:
                if _fnmatch.fnmatch(basename, pattern):
                    filename = _os.path.join(root, basename)
                    yield filename

    def set_required_packages(self, packages=None, delete=False):
        """
        Set the required packages for running this Task. When running this
        Task in a remote setting/environment, these packages are installed
        prior to execution.

        Parameters
        ----------
        packages : list | set [str]
            List of package requirements (same as disutils.requires) format for
            packages required for running this Task. If packages is set to None,
            then this Task's current required packages are removed.

        Returns
        -------
        self : Task

        Notes
        -----
        - No validation is performed on the specified packages list. So for
          example, if a required package is specified twice with two different
          versions, this API will not complain.

        Examples
        --------
        >>> t = graphlab.deploy.Task('set_required_packages_ex1')
        >>> t.set_required_packages(['numpy==1.8.0', 'pandas==0.13.1'])
        """
        if delete is True:
            self._data['packages'].clear()
        else:
            self._data['packages'].update(packages)

        self._set_dirty_bit()
        return self

    def set_outputs(self, names, delete=False):
        """
        Set/update output(s) for this Task. Outputs are GraphLab Data
        Structures (:class:`~graphlab.SFrame`, :class:`~graphlab.SGraph`,
        :class:`~graphlab.SArray`, Model).
    
        Parameters
        ----------
        names : list(str) | dict(str : str)
            If a list of strings is provided then each entry is considered a
            name for an output slot, and the value for that slot is None.

        delete : bool, optional
            If delete is set to True then the names parameters are removed.

        Returns
        -------
        self : Task

        Examples
        --------
        A few examples of valid list output definitions:

        >>> t = graphlab.deploy.Task("set_outputs_ex_1")
        >>> t.set_outputs(['one', 'two', 'three'])

        If a dict is provided, then each key is considered a name for an
        output, and each value is the path for that output.

        Examples of setting dict output definitions:

        >>> t = graphlab.deploy.Task("set_outputs_ex_2")
        >>> t.set_outputs({'one'   : '/tmp/path',
        >>>                'two'   : 's3://file/path.csv',
        >>>                'three' : None,
        >>>                'four'  : 'hdfs://host/path'}) 

        See Also
        --------
        set_inputs
        """
        if names is None:
            raise TypeError('names is a required parameter for set_outputs and cannot be None')

        if isinstance(names, list):
            for name in names:
                self._set_one_output(name=name, delete=delete)
        elif isinstance(names, dict):
            for key, value in names.iteritems():
                self._set_one_output(name=key, path=value, delete=delete)
        else:
            raise TypeError('Unknown type for names parameter. names parameter can be a list or a dict.')

        return self

    def _set_one_output(self, name='output', path=None, delete=False):
        """
        Set/update a GraphLab data structure output (:class:`~graphlab.SFrame`,
        :class:`~graphlab.SGraph`, :class:`~graphlab.SArray`, Model). Outputs
        are for GraphLab data structures.

        Parameters
        ----------
        name : str
            Named output parameter, default to 'output'
        path : str, optional
            Path to save output. If None specified then named output is
            removed. Paths can include local, S3, or HDFS.

            For example, to save locally:

            >>> t1.set_output('out', path='/tmp/savehere')
            
            For example, to save on S3:
            
            >>> t1.set_output('out', 
            >>>               path='s3:///bucket/object-path-inside-bucket')
            
            For example, to save on HDFS:
            
            >>> t1.set_output('out', path='hdfs:///tmp/save-on-hdfs-here')

        delete : bool, optional
            Delete the named output. If set to True then the output by that name is deleted.

        Returns
        -------
        self : Task
        """
        if delete is True and name in self._data['outputs']:
            del self._data['outputs'][name]
            return self
        
        self.__validate_slot_name_uniqueness(name, self._data['outputs'])

        if path is not None:
            self._data['outputs'][name] = _os.path.expanduser(path)
        else:
            self._data['outputs'][name] = None

        self._set_dirty_bit()
        return self

    def save(self):
        """
        Save changes made to this Task to disk.

        When a Task is instantiated, it lives in memory and will not be
        persisted to disk unless save is called explicitly. Persisted Tasks can
        be reloaded after server restarts, whereas unsaved Tasks are lost.

        See Also
        --------
        graphlab.deploy.tasks
        """
        self._session.save(self, typename=type(self).__name__)
        __LOGGER__.info("Task saved successfully.")

    def set_params(self, names, delete=False):
        """
        Add/set simple parameter(s) for this Task.

        Parameters can only be strings or numbers, not Python objects.

        Parameters
        ----------
        names : dict
            Dictionary with key/values (strings), merged into current params.

        delete : bool, optional
            To delete a key, set to True and pass params with dictionary. To
            delete all keys, pass None for params and delete=True.

        Returns
        -------
        self : Task

        Examples
        --------
        To set params using a list of names:

            >>> task.set_params(['one', 'two', 'three'])

        To set params using a dict:
            
            >>> task.set_params({'x':1, 'y':2})

        To delete/clear all params:

            >>> task.set_params(['one', 'two', 'three'])
            >>> task.set_params(None, delete=True)

        To delete a specific param:
            
            >>> task.set_params(['one', 'two', 'three'])
            >>> task.set_params(['one'], delete=True) # param named 'one' deleted

        """
        if names is None and delete is False:
            raise TypeError('names is a required parameter for set_params and cannot be None.')

        if isinstance(names, list):
            dictify_params = { name : None for name in names }
            self._set_params(params=dictify_params, delete=delete)
        elif isinstance(names, dict):
            self._set_params(params=names, delete=delete)
        elif names is None and delete is True:
            # delete all params
            self._set_params(None, True)
        else:
            raise TypeError('Unknown type for names parameter. names must be list or dict.')
        return self

    def _set_params(self, params=None, delete=False):
        """
        Add/set params with the Task.

        Parameters
        ----------
        params : dict
            Dictionary with key/values (strings), merged into current params.

        delete : bool
            To delete a key, set to True and pass params with dictionary. To
            delete all keys, pass None for params and delete=True.

        Returns
        -------
        Returns itself.
        """
        if params is None and delete is True:
            self._data['params'].clear()
            self._set_dirty_bit()
            return self

        if type(params) != dict:
            raise AttributeError('Invalid parameter, requires dict or None')

        if delete is True:
            for key in params.keys():
                if key in self._data['params']:
                    del self._data['params'][key]
        else:
            for key in params.keys():
                if key not in self._data['params']:
                    # validate uniqueness of names only if not already in params
                    self.__validate_slot_name_uniqueness(key, self._data['params'])
                self._data['params'][key] = params[key]

        self._set_dirty_bit()
        return self

    def get_name(self):
        """
        Return the name of this Task.
        """
        return str(self.name)

    def get_description(self):
        """
        Return the description of this Task.
        """
        return str(self._data['description'])

    def get_inputs(self):
        """
        Return a dictionary of inputs, by name.
        """
        return dict.copy(self._data['inputs'])

    def get_code(self):
        """
        Return a human-readable string indicating the code that is set for
        this Task.
        """
        return self.__code_str__()

    def _get_runnable_code(self):
        """
        Return machine-callable version of the code stored in Task. Not
        intended for interactive usage.
        """
        return self._data['code']

    def get_required_packages(self):
        """
        Return the set of packages (in distutils.requires format) required for
        this Task.

        References
        ----------
        - `Python distutils <https://docs.python.org/2/library/distutils.html>`_
        """
        return set.copy(self._data['packages'])

    def get_outputs(self):
        """
        Return output dictionary for the named outputs for this Task.
        """
        return dict.copy(self._data['outputs'])

    def get_params(self):
        """
        Return params set on Task.
        """
        return dict.copy(self._data['params'])

    def run(self, binding={}):
        """
        Run this Task standalone for development/debugging purposes.

        Parameters
        ----------
        binding : dict
            Dictionary of bindings to provide wiring for inputs/outputs/params.
            This allows binding input/output/params for this Task. This
            parameter supports everything job.create's bindings parameter
            supports, but also allows specifying a binding with a specific
            object.

        Returns
        -------
        task : RuntimeTask
            Instance of the RuntimeTask, but after it has been run. Useful for
            verifying that code ran as expected. This instance has been cloned
            and used for execution.

        See Also
        --------
        job.create

        Notes
        -----
        - This method is the same as calling :func:`graphlab.deploy.job.create`.
          It provides a convenient way to debug a Task in isolation.

        Examples
        --------
        You have a Task t1 that produces an output named 'results' and a Task t2
        that has an input named 'in' that is defined from the 'results' output
        from t1. This could be defined as follows:

        >>> t2.set_input('in', from=(t1, 'output'))

        Now, if you want to run t2 in isolation using this run method,
        pass in an instance of an :class:`~graphlab.SFrame` to override the
        definition of 'in':

        >>> sf = graphlab.SFrame() # SFrame to use that represents 'output' from
        >>> t2.run(bindings={'in': sf})

        Using this binding technique enables iteratively building a Task with
        dependencies without requiring to create Jobs while developing.
        """
        metrics = {}
        clone = self._clone(self.name, session_aware=False)
        _gl.deploy.job._apply_binding_to_task(clone, binding, allow_object=True)
        execenv = _gl.deploy._executionenvironment.LocalExecutionEnvironment()
        runtime_task = execenv.run_task(clone, metrics)
        return runtime_task

    def _realize_input(self):
        # collect inputs of GL data structures as Python objects, pass to code
        inputs_obj = dict()
        inputs = self.get_inputs()
        for name in inputs.keys():
       
            # default to None
            inputs_obj[name] = None

            if hasattr(self, '_local_binding') and name in self._local_binding:
                # special-casing task.run() with binding to object, 
                inputs_obj[name] = self._local_binding[name]
        
            elif inputs[name] is not None:
                __LOGGER__.debug("Realizing input name: %s, from %s" % (name, inputs[name]))
                if isinstance(inputs[name], tuple):
                    # dealing with task/pipeline dependency
                    (dependency, output_name) = inputs[name]
                    task = None                                        

                    # When task is being run in a Job, the Job adds a .env attribute
                    # which is a reference to the execution environment from which this job is running.
                    # Query that environment to find the dependent task.
                    if hasattr(self, 'env'):
                        dep = self.env.get_dependent_task(dependency)
                        if dep is not None:
                            task = dep
                        else:
                            task = self._session._load(dependency, typename='Task')

                    else:
                        task = self._session._load(dependency, typename='Task')

                    if task is not None:
                        # realize just this one output
                        output = task.outputs[output_name]
                        inputs_obj[name] = output
                    else:
                        __LOGGER__.warning("Unable to load Task required for realizing this input, named: %s" % name)

                else:
                    # dealing with file input
                    filepath = inputs[name]
                    if str(filepath) == "":
                        continue
                    else:
                        inputs_obj[name] = self.__load_data_structure__(filepath)

        return inputs_obj

    def _realize_output(self):
        """
        Collect outputs (if exists) and return in dictionary, return None for non-realized outputs.
        """
        outputs_obj = _OutputTypeDict(self)
        outputs = self.get_outputs()
        for name in outputs.keys():
            if outputs[name] is not None:
                filepath = outputs[name]
                __LOGGER__.debug('Realizing output %s from %s' % (name, filepath))
                outputs_obj[name] = self.__load_data_structure__(filepath)
        return outputs_obj

    @staticmethod
    def _is_valid_data_structure(obj):
        return (isinstance(obj, _gl.SFrame) or 
                isinstance(obj, _gl.SGraph) or 
                isinstance(obj, _gl.Model) or 
                isinstance(obj, _gl.SArray))

    def __load_data_structure__(self, filepath):
        """Return data structure if can be loaded, otherwise returns None and logs warning"""
        # try to load different supported types, since don't know what type just try all and swallow exceptions
        obj = None
        try:
            obj = _gl.load_sframe(filepath)
            return obj
        except:
            pass
        try:
            obj = _gl.load_sgraph(filepath)
            return obj
        except:
            pass
        
        try:
            obj = _gl.load_model(filepath)
            return obj
        except:
            pass

        try:
            obj = _gl.SArray(data=filepath)
            return obj
        except:
            pass

        __LOGGER__.debug("Unable to load dependency, unsupported type at path: %s" % filepath)
        return None

    def __getattr__(self, item):
        """
        The preferred way to get .inputs, .outputs, and .params objects realized
        """
        if item == 'inputs':
            self.inputs = self._realize_input()
            return self.inputs
        elif item == 'outputs':
            self.outputs = self._realize_output()
            return self.outputs
        elif item == 'params':
            self.params = self._data['params']
            return self._data['params']

        raise AttributeError(item)

    def __validate_slot_name_uniqueness(self, name, slot_list):
        names = self._data['inputs'].keys() + self._data['params'].keys() + self._data['outputs'].keys()
        if name in names and name not in slot_list.keys():
            raise TypeError('Name already in use (as input, param, or output). Names must be unique.')
    
    @staticmethod
    def __remove_unwanted_keys(dict, keys=[]):
        dict = {k: v for k, v in dict.iteritems() if k not in keys}
        return dict

    def __getstate__(self):
        odict = dict.copy(self.__dict__)
        keys = ['_session', 'input', 'output', 'inputs', 'outputs']
        odict = Task.__remove_unwanted_keys(odict, keys=keys)
        return odict

    def __repr__(self):
        return self.__str__()
    def __code_str__(self):
        code = self._data['code']
        if code is None or code == "":
            return ""

        if isinstance(code, dict):
            out = "Calling '%s' function in filename '%s'. Other packaged files:\n%s" % \
                  (self._data['funcname'], self._data['filename'], code.keys())
            return out
        else:
            if 'codestr' in self._data:
                return "%s" % self._data['codestr']
            return "%s" % code

    def __inputs_str__(self):
        inputs = self.get_inputs()
        ret = {}
        for input in inputs:
            try:
                (val, output_name) = inputs[input]
            except (ValueError, TypeError):
                val = inputs[input]
                output_name = None
            if output_name is None:
                ret[input] = val
            else:
                ret[input] = "(task: %s, output_name: %s)" % (val, output_name)
        return str(ret)

    def __str__(self):
        header = "Task: %s" % (self.get_name())
        desc = "Description:\n\t%s" % (self.get_description())
        inputs = "Input(s):\n\t%s" % self.__inputs_str__()
        code = "Code:\n\n%s" % self.__code_str__()
        outputs = "Output(s):\n\t%s" % self.get_outputs()
        params = "Parameter(s):\n\t%s" % self.get_params()
        packages = "Package(s):\n\t%s\n" % self.get_required_packages()
        out = "%s\n\n%s\n\n%s\n\n%s\n\n%s\n\n%s\n\n%s\n" % (header, desc, inputs, code, outputs, params, packages)

        if hasattr(self, 'status'):
            out += "Status: %s\n" % self.status

        return out

    def __code_snapshot__(self):
        if 'codestr' in self._data and self._data['codestr'] is not None:
            return self._data['codestr']
        return self.__code_str__()
            
    def __to_dict__(self):
        params = self.get_params()
        param_dict = {}
        isFunction = False
        for k,v in params.iteritems():
            param_dict[k] = v
            if _isfunction(v):
                param_dict[k] = v.func_name
                isFunction = True
            
        task_json = {}
        task_json["name"] = self.get_name()
        task_json["description"] = self.get_description()
        task_json["inputs"] = self.get_inputs()
        task_json["code"] = self.__code_snapshot__()
        task_json["outputs"] = self.get_outputs()
        task_json["params"] = param_dict
        task_json["isFunction"] = isFunction
        task_json["required_packages"] = list(self.get_required_packages())
        return task_json

    def _set_dirty_bit(self):
        if self._modified_since_last_saved is not None and not self._modified_since_last_saved:
            self._modified_since_last_saved = True


class Pipeline(object):
    """
    A Pipeline references a sequence of steps being executed, and captures the
    order of operations between steps to manage a data product's execution.
    """

    def __init__(self, name, *args, **kwargs):
        self.steps = []
        self._session = _gl.deploy._default_session
        self.name = name
        self._modified_since_last_saved = None

    def clone(self, name):
        """
        Clone current object to new pipeline, with the specified name.

        Parameters
        ----------
        name : str
            Name of the cloned pipeline

        Returns
        -------
        Newly cloned pipeline instance.
        """
        return self._clone(name, session_aware=True)

    def _clone(self, name, session_aware=True):
        new = Pipeline.__new__(Pipeline)
        new.name = name
        # new._data = _copy.deepcopy(self._data)
        new._session = self._session
        new._modified_since_last_saved = None

        new_steps = []
        for step_idx in range(len(self.steps)):
            new_steps.append([])
            for cur_task in self.steps[step_idx]:
                cloned_cur_task = cur_task._clone(cur_task.name, session_aware = False)
                new_steps[step_idx].append(cloned_cur_task)

        new.set_tasks(new_steps)
       
        return new

    def get_required_packages(self):
        """
        Retrieve the set of required packages for this Pipeline to execute.
        """
        result = set()
        for i in self.steps:
            for j in i:
                result.update(j.get_required_packages())
        return result

    def set_tasks(self, steps):
        """
        Set A list of steps to run for current pipeline.

        Each step is a list of Tasks that can be executed in parallel.
        """
        for task in steps:
            # TODO: Verify that steps are Tasks/Pipelines, and can function
            pass
        self.steps = steps
        self._set_dirty_bit()

    def set_params(self, params):
        for cur_step in self.steps:
            for cur_task in cur_step:
                if cur_task.name in params:
                    cur_params = params[cur_task.name]
                    cur_task.set_params(cur_params)
        self._set_dirty_bit()

    def save(self):
        self._session.save(self, typename=type(self).__name__)
        __LOGGER__.info("Pipeline saved successfully.")

    def __getstate__(self):
        # session attribute gets added when loaded from disk
        # make sure to remove it when pickling
        odict = dict.copy(self.__dict__)
        if hasattr(odict, '_session'):
            del odict._session
        return odict

    def _set_dirty_bit(self):
        if self._modified_since_last_saved is not None and not self._modified_since_last_saved:
            self._modified_since_last_saved = True

    def __str__(self):
        return "Pipeline: %s\nSteps: %s" % (self.name, self.steps)

    def __repr__(self):
        return self.__str__()

class _OutputTypeDict(_collections.MutableMapping, dict):
    """
    Helper dict type that verifies that value set must be a GraphLab Data Structure type.
    """
    def __init__(self, task):
        super(_OutputTypeDict, self).__init__()
        self._task = task

    def __getitem__(self, key):
        return dict.__getitem__(self, key)

    def __setitem__(self, key, value):
        if key is not None:
            self._verify_output_slot(key)
        if value is not None:
            self._verify_output_type(value)
        dict.__setitem__(self, key, value)

    def __delitem__(self, key):
        dict.__delitem__(self, key)

    def __iter__(self):
        return dict.__iter__(self)

    def __len__(self):
        return dict.__len__(self)

    def __contains__(self, x):
        return dict.__contains__(self, x)

    def _verify_output_slot(self, key):
        if key not in self._task.get_outputs():
            raise NameError("Invalid output name used in output assignment. Name '%s' has not been specified in this Task." % key)

    def _verify_output_type(self, value):
        if not Task._is_valid_data_structure(value):
            raise TypeError('Unsupported type for output. Outputs can only be SFrame, SGraph, SArray, or Model types. Type: %s' % type(value))

class RuntimeTask(object):
    """
    A runtime representation for a Task. When a Task is run, the code that is
    executed receives a RuntimeTask instance for that Task, with all of its
    parameters bound.
    """
    def __init__(self, task):
        if task is None:
            raise TypeError('Misconfigured Task, please contact support@graphlab.com')

        self.__task = task
        self.inputs = self.__task.inputs
        self.outputs = self.__task.outputs
        self.params = self.__task.params
        self.metrics = {}

    def get_outputs(self):
        """
        Return the output names defined in the Task being run.

        Returns
        -------
        List of output names defined for this Task.
        """
        result = sorted(list(self.__task.get_outputs().keys()))
        return result

    def get_inputs(self):
        """
        Return the input names defined in the Task being run.

        Returns
        -------
        List of input names defined for this Task.
        """
        result = sorted(list(self.__task.get_inputs().keys()))
        return result

    def get_params(self):
        """
        Return the parameter names defined in the Task being run.

        Returns
        -------
        List of parameter names defined for this Task.
        """
        result = sorted(list(self.__task.get_params().keys()))
        return result

    def __str__(self):
        header = "Runtime Task for Task: %s" % (self.__task.get_name())
        inputs = "Input Name(s):\n\t%s" % self.get_inputs()
        outputs = "Output Name(s):\n\t%s" % self.get_outputs()
        params = "Parameter Name(s):\n\t%s" % self.get_params()
        metrics = "Metrics:\n\t%s" % self.metrics
        out = "%s\n\n%s\n\n%s\n\n%s\n\n%s\n" % (header, inputs, outputs, params, metrics)
        return out

    def __repr__(self):
        return self.__str__()

