from copy import copy as _copy
from logging import getLogger as _getLogger
from time import time as _time
from os.path import dirname as _dirname, join as _path_join

import graphlab as _gl
from graphlab.connect import _get_metric_tracker
from graphlab.data_structures.sarray import SArray as _SArray
from graphlab.data_structures.sframe import SFrame as _SFrame
from graphlab.deploy import job as _job, _Pipeline, Task as _Task


_COMBINE_PARAMETER_NAME = 'parameter_values_to_intermediate_results'
_log = _getLogger(__name__)


def _generate_step_assignments(task_prototype, param_set, num_task_per_step, temp_path):
    '''
    Creates len(param_set) number of tasks. Each of which is a clone of task_prototype with a
    different element of param_set set as its parameters. The output of each task will be set
    to a temporary locations.

    Returns a list of tasks organized into step (i.e. inner lists). Also returns the mapping
    of param values to output location.
    '''
    param_set = _copy(param_set)
    param_set.reverse() # Preserve order; we'll be 'popping' in reverse order.

    tasks, params_to_outputs, step_index = [], [], 0

    while param_set:
        tasks.append([])
        num_in_cur_step = 0
        while num_in_cur_step < num_task_per_step and param_set:

            cur_params = param_set.pop()
            cur_name = '-'.join([task_prototype.name, str(step_index), str(num_in_cur_step)])

            cur_task = task_prototype._clone(cur_name, session_aware = False)
            cur_task.set_params(cur_params)

            output_base_path = _path_join(temp_path, str(step_index), str(num_in_cur_step))
            params_to_outputs.append((cur_params, output_base_path))

            # Set each output to a temporary location under temp_path.
            for name in cur_task.get_outputs():
                output_path = _path_join(output_base_path, name)
                cur_task.set_outputs({name: output_path})

            tasks[-1].append(cur_task)
            num_in_cur_step += 1
        step_index += 1


    return tasks, params_to_outputs


def _combine(task):
    '''
    The actual code that will be ran inside of a task to combine all results and add the parameter
    column to the final SFrame(s).
    '''
    # Initialize empty SFrames for each output.
    for out_name in task.get_outputs():
        task.outputs[out_name] = _SFrame()

    params_to_outputs = task.params[_COMBINE_PARAMETER_NAME]
    for params, path in params_to_outputs:
        for out_name in task.get_outputs():

            try: 
                cur_result = _SFrame(_path_join(path, out_name))
            except IOError:
                _log.info("No output for %s with parameters: %s " % (out_name, str(params)))
                continue

            # Add the 'Parameters' column and append to previous results.
            cur_result['parameters'] = _SArray.from_const(params, len(cur_result))
            cur_result.__materialize__()
            task.outputs[out_name] = task.outputs[out_name].append(cur_result)
    

def parallel_for_each(task, parameter_set, name = None, environment = None):
    '''
    Runs a task once for each entry in the parameter_set. The output will be the concatenation of
    the outputs from each iteration. Each row will also contain the parameter values used to generate
    that row.

    Parameters
    ----------
    task : The task to run for each parallel-for-each iteration. The final outputs (of all
        iterations) will be saved at the outputs specified for this task.

    parameter_set : list of dicts | SFrame
        If parameter_set is a list of dicts, each element of the list corresponds to an iteration
        of the given task. The keys of each dictionary should be the parameter names, and the value
        should be the value of that parameter.
        If parameter_set is an SFrame, each row corresponds to an iteration of the given task. The
        column names of the SFrame should correspond to the parameter names of the task.

    name : str
        The name to use for the job that will be returned. If not specified, one will be generated.

    environment : This must be an environment created from the :py:class:`~graphlab.deploy` module.
        This is the environment where the parallel-for-each will be executed. The environment could
        be a set of EC2 hosts or a Hadoop cluster. It could be an environment that executes locally
        in the background. Or it could be run as a blocking call on your local computer.

    Returns
    -------
    job : :py:class:`~graphlab.deploy._job.Job`
        The job for the parallel-for-each, which was run using the `environment` parameter. This
        object can be used to track the progress of parallel-for-each work.

    Examples
    --------
    .. sourcecode:: python

    >>> my_task = graphlab.deploy.Task('My Task Name')

    >>> def sum(task):
            sum = task.params['x'] + task.params['y']
            task.outputs['sum'] = graphlab.SFrame({'sum': [sum]})
    >>> my_task.set_code(sum)
    >>> my_task.set_params(['x', 'y'])

    >>> my_task.set_outputs({'sum': '/tmp/my_sums'})
    >>> params = [{'x': 1, 'y': 2}, {'x': 0, 'y': 9}, {'x': 23, 'y': 3}]

    >>> my_job = graphlab.deploy.parallel_for_each(my_task, params)
    >>> # The above call will return immediately.
    >>> # Once my_job.get_status() returns 'Completed' your results are ready.

    >>> results = graphlab.SFrame('/tmp/my_sums')
    >>> results
    +-----+-------------------+
    | sum |     parameters    |
    +-----+-------------------+
    |  3  |  {'y': 2, 'x': 1} |
    |  9  |  {'y': 9, 'x': 0} |
    |  26 | {'y': 3, 'x': 23} |
    +-----+-------------------+
    [3 rows x 2 columns]
    '''

    _get_metric_tracker().track('jobs.parallel_for_each')

    if not name:
        name = "parallel_for_each-%s-%s" % (task.name, _time())

    if isinstance(parameter_set, _SFrame):
        # Convert SFrame to a dict
        parameter_set = [ i for i in parameter_set ]

    if not parameter_set:
        _log.warn('An empty parameter_set was given. No iterations will be run.')

    temp_dir = _job._get_results_path(environment, name, 'temp')

    task_per_step = 1 
    if environment:
        task_per_step = environment.get_max_degree_of_parallelism()

    for_each_iterations, params_to_outputs = _generate_step_assignments(task, parameter_set,
                                                                        task_per_step, temp_dir)

    # Create the combiner task
    combiner = _Task('-'.join([name, 'combine']))
    combiner.set_outputs(task.get_outputs())
    combiner.set_params({_COMBINE_PARAMETER_NAME: params_to_outputs})
    combiner.set_code(_combine)
    _gl.deploy.tasks.delete(combiner)

    pipeline = _Pipeline(name + "-Pipeline")
    pipeline.set_tasks(for_each_iterations + [[combiner]])
    return _job.create([pipeline], name=name, environment=environment)
