"""
This module defines the CustomQueryPredictiveObject class that allows wrapping of
one or many GraphLab data structures and additional functionality within a
predictive object, which can be deployed into a Predictive Service.
"""

import types as _types
import inspect as _inspect
from _predictive_object import PredictiveObject as _PredictiveObject

class CustomQueryPredictiveObject(_PredictiveObject):
    '''
    CustomQueryPredictiveObjects are intended to facilitate deployment of
    GraphLab data structures with additional custom query logic.

    To create a CustomQueryPredictiveObject, do the following:

    1. Define a query function that has the following signature:

        >>> def query(input, dependencies=None, state=None):
        >>>           # input is a dict
        >>>           # dependencies is a dict
        >>>           # state is a dict
        >>>     return dict

        The 'dependencies' argument is the dictionary specified below. The
        'state' argument is a dictionary that is maintained by the
        Predictive Service deployment between each call to the query
        object. Use this dictionary to track anything that requires state
        between calls.

    2. (Optional) Define a list of GraphLab objects that the query uses, organized into a
        dictionary:

        my_dependencies = {'classifier':model_classifier, 'user_graph': some_graph}

    3. (Optional) Define a list of python packages the query function depends on:

        my_required_packages = ["mysql==1.0.0", ...]

    4. Create a CustomQueryPredictiveObject using above information:

        >>> my_po = CustomQueryPredictiveObject(my_query, my_dependencies, my_required_packages, my_description)

    5. Add to an existing Predictive Service:

        >>> my_ps.add('my predictor', my_po)


    Another way to add a CustomQueryPredictiveObject to a Predictive Service is a
    shortcut:

        >>> my_ps.add('my_predictor', my_query, my_dependencies, my_required_packages, my_description)

    '''
    def __init__(self, query, dependencies = None, required_packages = None, description = ''):
        '''Create a new CustomQueryPredictiveObject.

        query : function
            Function that defines a custom query method. Method should have the
            following signature:

                >>> def query(input, dependencies=None, state=None):
                >>>           # input is a dict
                >>>           # dependencies is a dict
                >>>           # state is a dict
                >>>     return dict

            The 'dependencies' argument is the dictionary specified below. The
            'state' argument is a dictionary that is maintained by the
            Predictive Service deployment between each call to the query
            object. Use this dictionary to track anything that requires state
            between calls.

        dependencies : dict
            A dictionary that specifies all dependent GraphLab objects that
            this custom query relies on. The key is the identifier for the
            dependent object, the value is either a url or an actual GraphLab
            object. This parameter is only used if the obj parameter is a
            function.

        required_packages : list(str)
            A list that specifies all python packages this custom Predictive
            Object is dependent on. The dependencies are specifed in the format
            of distutils, like: ['mysql==0.3.0', 'abc==1.2.3']. The deployment
            will install these dependencies from Pypi when apply_changes is
            called. This parameter is only used if the obj parameter is a
            function.

        description : str
            The description of the custom predictive object

        '''
        self._validate_parameters(query, dependencies, required_packages, description)

        super(CustomQueryPredictiveObject, self).__init__(description)
        self.dependencies = dependencies or {}
        self.required_packages = required_packages or []
        self.custom_query = query
        self.state = {}

    def query(self, input):
        '''Query the model according to input

        Query the custom defined query method using the given input
        input.

        Parameters
        ----------
        input : dict
            a dictionary that contains input for user query, it will be passed along
            to custom query method.

        Returns
        -------
        out: object.
            The results depends on the implementation of the query method.
            Typically the return value will be whatever that function returns.
            However if it returns an SFrame, the SFrame will be automatically
            converted to a list of dicts. If it returns an SArray, the SArray
            will be converted to a list.

        See Also
        --------
        PredictiveObject, ModelPredictiveObject
        '''
        args = _inspect.getargspec(self.custom_query).args

        if  args == ['input']:
            ret = self.custom_query(input)
        elif args == ['input', 'dependencies']:
            ret = self.custom_query(input, self.dependencies.copy())
        else:
            ret = self.custom_query(input, self.dependencies.copy(), self.state)

        return self._make_serializable(ret)

    @classmethod
    def _validate_parameters(cls, query, dependencies, required_packages, description):
        if not isinstance(query, _types.FunctionType):
            raise TypeError('Query parameter has to be a function')

        # validate expected query signature
        args = _inspect.getargspec(query).args
        if  args != ['input'] and \
            args != ['input', 'dependencies'] and \
            args != ['input', 'dependencies', 'state']:
            raise TypeError("User defined query signature not recognized. Supported arguments are ['input', 'dependencies', 'state'], current args: %s" % str(args))

        if dependencies:
            if not isinstance(dependencies, dict):
                raise TypeError("'dependencies' has to be a dictionary")

            for (key, value) in dependencies.iteritems():
                if not isinstance(key, str):
                    raise TypeError("'dependencies' should be a dictionary with keys being string type")

                if not isinstance(value, str) and \
                   not isinstance(value, unicode) and \
                   not cls._is_graphlab_object(value):
                    raise TypeError("Object %s is not a valid GraphLab object")
                # todo validate against a url

            # parameter and query signature should match
            if (len(dependencies) != 0) and ((len(args) < 2) or (args[1] != 'dependencies')):
                raise TypeError("User defined query needs to have 'dependencies' parameter because dependencies is not empty.")

        # validate python package dependencies, should be a list of strings
        if required_packages:
            if  not isinstance(required_packages, list) or \
                any([not isinstance(dependency, str) for dependency in required_packages]):
                raise TypeError("python package dependency has to be a list of strings like: ['mysql==1.0.0', 'another==0.2.3']")

        if not isinstance(description, str):
            raise TypeError("'description' has to be a string")
