"""
This package contains the Gradient Boosted Trees model class and the create function.
"""
import graphlab as _graphlab
import graphlab.connect as _mt
from graphlab.toolkits._model import _get_default_options_wrapper
from graphlab.toolkits._supervised_learning import Classifier as _Classifier
import graphlab.toolkits._supervised_learning as _sl
from graphlab.toolkits._internal_utils import _toolkit_repr_print
from graphlab.toolkits._internal_utils import _SGraphFromJsonTree
from graphlab.toolkits._internal_utils import _raise_error_evaluation_metric_is_valid
from graphlab.toolkits._internal_utils import _raise_error_if_not_sframe
from graphlab.toolkits._internal_utils import _raise_error_if_column_exists
from graphlab.toolkits._internal_utils import _check_categorical_option_type
from graphlab.toolkits._internal_utils import _map_unity_proxy_to_object


DEFAULT_HYPER_PARAMETER_RANGE = {
    'max_depth': [6, 8, 10],
    'step_size': 0.3,
    'min_loss_reduction': [0, 1, 10],
    'min_child_weight': 0.1,
    'row_subsample': 1.0,
    'column_subsample': 1.0,
    'max_iterations': [10, 50, 100]
}

get_default_options = _get_default_options_wrapper(
                          'boosted_trees_classifier',
                          'boosted_trees_classifier',
                          'BoostedTreesClassifier')

__doc_string_context = '''
      >>> url = 'http://s3.amazonaws.com/gl-testdata/xgboost/mushroom.csv'
      >>> data = graphlab.SFrame.read_csv(url)
      >>> data['target'] = (data['target'] == 'e')
      
      >>> train, test = data.random_split(0.8)
      >>> model = graphlab.boosted_trees.create(train, target='label')
'''

class BoostedTreesClassifier(_Classifier):
    """
    The gradient boosted trees model can be used as a classifier for predictive
    tasks.

    The prediction is based on a collection of base learners i.e 
    `decision tree classifiers <http://en.wikipedia.org/wiki/Decision_tree_learning>`_
    and combines them through a technique called `gradient boosting
    <http://en.wikipedia.org/wiki/Gradient_boosting>`_.

    Different from linear models like logistic regression or SVM, gradient
    boost trees can model non-linear interactions between the features and the
    target. This model is suitable for handling numerical features and
    categorical features with tens of categories but is less suitable for highly
    sparse features (text data), or with categorical variables that encode a 
    large number of categories.

    This model cannot be constructed directly.  Instead, use 
    :func:`graphlab.boosted_trees_classifier.create` to create an instance of 
    this model. 
    Additional details on parameter options and code samples are available in 
    the documentation for the create function.

    See Also
    --------
    create

    """
    def __init__(self, proxy):
        """__init__(self)"""
        self.__proxy__ = proxy
        self.__name__ = "boosted_trees_classifier"

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return BoostedTreesClassifier(model_proxy)
        return model_wrapper


    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the model.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model, when the model name is entered
        in the terminal.
        """

        data_fields = [
            ('Number of examples', 'num_examples'),
            ('Number of classes', 'num_classes'),
            ('Number of feature columns', 'num_features'),
            ('Number of unpacked features', 'num_unpacked_features')]

        training_fields = [
            ("Number of trees", 'num_trees'),
            ("Max tree depth", 'max_depth'),
            ("Train accuracy", 'training_accuracy'),
            ("Validation accuracy", 'validation_accuracy'),
            ("Training time (sec)", 'training_time')]

        return _toolkit_repr_print(self, [data_fields, training_fields], width=30)

    def summary(self):
        """
        Prints a summary of :class:`~graphlab.boosted_trees_classifier.BoostedTreesClassifier`.
        
        See Also
        --------
        show

        Examples
        --------
        >>> print model.summary()
        """
        _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.summary')
        print ""
        print "                    Model summary                       "
        print "--------------------------------------------------------"
        print self.__repr__()

    def get(self, field):
        """
        Get the value of a given field. The following table describes each 
        of the fields below.

        +-------------------------+--------------------------------------------------------------------------------+
        | Field                   | Description                                                                    |
        +=========================+================================================================================+
        | column_subsample        | Percentage of the columns for training each individual tree                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | features                | Names of the feature columns                                                   |
        +-------------------------+--------------------------------------------------------------------------------+
        | max_depth               | The maximum depth of the individual decision trees                             |
        +-------------------------+--------------------------------------------------------------------------------+
        | max_iterations          | Maximum number of iterations for training (one tree is trained per iteration)  |
        +-------------------------+--------------------------------------------------------------------------------+
        | min_child_weight        | Minimum weight assigned to leaf nodes                                          |
        +-------------------------+--------------------------------------------------------------------------------+
        | min_loss_reduction      | Minimum loss reduction required for splitting a node                           |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_features            | Number of feature columns in the model                                         |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_unpacked_features   | Number of features in the model (including unpacked dict/list type columns)    |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_examples            | Number of training examples                                                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | num_validation_examples | Number of validation examples                                                  |
        +-------------------------+--------------------------------------------------------------------------------+
        | row_subsample           | Percentage of the rows sampled for training each individual tree               |
        +-------------------------+--------------------------------------------------------------------------------+
        | step_size               | Step_size used for combining the weight of individual trees                    |
        +-------------------------+--------------------------------------------------------------------------------+
        | target                  | Name of the target column                                                      |
        +-------------------------+--------------------------------------------------------------------------------+
        | training_accuracy       | Classification accuracy measured on the training data                          |
        +-------------------------+--------------------------------------------------------------------------------+
        | training_time           | Time spent on training the model in seconds                                    | 
        +-------------------------+--------------------------------------------------------------------------------+
        | trees_json              | Tree encoded using JSON                                                        |
        +-------------------------+--------------------------------------------------------------------------------+
        | validation_accuracy     | Classification accuracy measured on the validation set                         |
        +-------------------------+--------------------------------------------------------------------------------+
        | unpacked_features       | Feature names (including expanded list/dict features)                          |
        +-------------------------+--------------------------------------------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out : [various]
            The current value of the requested field.

        See Also
        --------
        list_fields

        Examples
        --------

        >>> print model['num_features']
        3
        >>> print model.get('num_features')       # equivalent to previous line
        3

        """
        _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.get')
        return super(_Classifier, self).get(field)

    def show(self, tree_id=0, vlabel_hover=False):
        """
        Plot the tree as an SGraph in canvas.

        Parameters
        ----------
        tree_id : int
            The id of the tree to show. Starting from 0 to max_iterations-1.

        vlabel_hover : bool, optional
            If True, hide the label on vertex, and only show the label
            with mouse hovering.

        See Also
        --------
        summary

        Examples
        --------
        >>> model.show(0)
        """
        tree_json = self.get('trees_json')[tree_id]
        g = _SGraphFromJsonTree(tree_json)

        # Make the label for each vertex based on its type
        def get_vlabel(record):
            if record['type'] == 'leaf':
                return str(record['value'])
            elif record['type'] == 'indicator':
                return str(record['name'])
            else:
                return str(record['name']) + '<' + str(record['value'])
        g.vertices['__repr__'] = g.vertices.apply(get_vlabel, str)

        fuchsia = [0.69, 0., 0.48]
        blue = [0.039, 0.55, 0.77]
        orange = [1.0, 0.33, 0.]
        green = [0.52, 0.74, 0.]

        # Make the color for leaf nodes
        def get_leaf_vcolor(record):
            '''assign color to leaf vertex, negative_leaf->blue, positive_leaf->orange'''
            return blue if record['value'] < 0 else orange

        leaf_vertices = g.vertices[g.vertices['type'] == 'leaf']
        root_vertex_color = {0: fuchsia}
        leaf_vertex_color = dict([(x['__id'], get_leaf_vcolor(x)) for x in leaf_vertices])
        highlight = {}
        highlight.update(root_vertex_color)
        highlight.update(leaf_vertex_color)
        elabel = 'value' if len(g.edges) > 0 else None

        # Hack: we want the canvas to show tree_{i} instead of g, so here is how we do it.
        graph_name = 'tree_' + str(tree_id)
        locals().update({graph_name: g})
        del g
        locals()[graph_name].show(vlabel='__repr__', elabel=elabel,
                                  vlabel_hover=vlabel_hover,
                                  highlight=highlight, arrows=True)

    def evaluate(self, dataset, metric='auto'):
        """
        Evaluate the model on the given dataset.

        Parameters
        ----------
        dataset : SFrame
            Dataset in the same format used for training. The columns names and
            types of the dataset must be the same as that used in training.

        metric : str, optional
            Name of the evaluation metric.  Possible values are:
            'auto'             : Returns all available metrics.
            'accuracy '        : Classification accuracy.
            'confusion_matrix' : An SFrame with counts of possible prediction/true 
                                 label combinations.

        Returns
        -------
        out : dict
            A dictionary containing the evaluation result.

        See Also
        ----------
        create, predict, classify

        Examples
        --------
        >>> results = model.evaluate(test_data)
        >>> results = model.evaluate(test_data, metric='accuracy')
        >>> results = model.evaluate(test_data, metric='confusion_matrix')

        Notes
        -----
        When evaluating for classification metrics (e.g. auc,
        confusion_matrix), the classification threshold is set to 0.5. For more
        flexible classification accuracy, please use functions in the
        :py:mod:`~graphlab.toolkits.evaluation` module.
        """
        _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.evaluate')
        _raise_error_evaluation_metric_is_valid(metric,
                                      ['auto', 'accuracy', 'confusion_matrix'])

        results = {}
        if metric in ['auto', 'accuracy']:
          results = super(_Classifier, self).evaluate(dataset, metric = metric)

        if metric in ['confusion_matrix', 'auto']:
          predictions = self.predict(dataset, output_type = 'class')
          target = self.get('target')
          _raise_error_if_column_exists(dataset, target, 'dataset', target)
          results['confusion_matrix'] = _graphlab.evaluation.confusion_matrix(\
                                         dataset[target], predictions)
        return results

    def predict(self, dataset, output_type='class'):
        """
        A flexible and advanced prediction API.

        The target column is provided during
        :func:`~graphlab.boosted_trees.create`. If the target column is in the
        `dataset` it will be ignored.

        Parameters
        ----------
        dataset : SFrame
          A dataset that has the same columns that were used during training.
          If the target column exists in ``dataset`` it will be ignored
          while making predictions.

        output_type : {'probability', 'margin', 'class'}, optional.
          If output_type is 'probability', then predict will output the class
          probability between [0, 1]. Otherwise, it will output the margin
          score before transforming to probability using the logistic function.

        Returns
        -------
        out : SArray
           Predicted target value for each example (i.e. row) in the dataset.

        See Also
        ----------
        create, evaluate, classify

        Examples
        --------
        >>> m.predict(testdata)
        >>> m.predict(testdata, output_type='probability')
        >>> m.predict(testdata, output_type='margin')
        """
        _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.predict')

        _check_categorical_option_type('output_type', output_type, ['class', 'margin', 'probability'])
        return super(_Classifier, self).predict(dataset,
                                                    output_type = output_type)

    def predict_topk(self, dataset, output_type="probability", k=3):
        """
        Return top-k predictions for the ``dataset``, using the trained model.
        Predictions are returned as an SFrame with three columns: `row_id`,
        `class`, and `probability`, `margin`,  or `rank`, depending on the ``output_type``
        parameter. Input dataset size must be the same as for training of the model.

        Parameters
        ----------
        dataset : SFrame
            A dataset that has the same columns that were used during training.
            If the target column exists in ``dataset`` it will be ignored
            while making predictions.

        output_type : {'probability', 'rank', 'margin'}, optional
            Choose the return type of the prediction:

            - `probability`: Probability associated with each label in the prediction.
            - `rank`       : Rank associated with each label in the prediction.
            - `margin`     : Margin associated with each label in the prediction.

        k : int, optional
            Number of classes to return for each input example.

        Returns
        -------
        out : SFrame
            An SFrame with model predictions.

        See Also
        --------
        predict, classify, evaluate

        Examples
        --------
        >>> pred = m.predict_topk(validation_data, k=3)
        >>> pred
        +--------+-------+-------------------+
        | row_id | class |   probability     |
        +--------+-------+-------------------+
        |   0    |   4   |   0.995623886585  |
        |   0    |   9   |  0.0038311756216  |
        |   0    |   7   | 0.000301006948575 |
        |   1    |   1   |   0.928708016872  |
        |   1    |   3   |  0.0440889261663  |
        |   1    |   2   |  0.0176190119237  |
        |   2    |   3   |   0.996967732906  |
        |   2    |   2   |  0.00151345680933 |
        |   2    |   7   | 0.000637513934635 |
        |   3    |   1   |   0.998070061207  |
        |  ...   |  ...  |        ...        |
        +--------+-------+-------------------+
        [35688 rows x 3 columns]
        """
        _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.predict_topk')
        _raise_error_if_not_sframe(dataset, "dataset")
        _check_categorical_option_type('output_type', output_type, ['rank', 'margin', 'probability'])
        options = dict()
        options.update({'model': self.__proxy__,
                        'model_name': self.__name__,
                        'dataset': dataset,
                        'output_type': output_type,
                        'topk': k,
                        'missing_value_action': 'error'})
        target = _graphlab.toolkits._main.run('supervised_learning_predict_topk', options)
        return _map_unity_proxy_to_object(target['predicted'])

    def classify(self, dataset, missing_value_action='impute'):
        """
        Return a classification, for each example in the ``dataset``, using the
        trained boosted trees model. The output SFrame contains predictions
        as class labels (0 or 1) and probabilities associated with the the example.

        Parameters
        ----------
        dataset : SFrame
            Dataset of new observations. Must include columns with the same
            names as the features used for model training, but does not require
            a target column. Additional columns are ignored.

        missing_value_action: str, optional
            Action to perform when missing values are encountered. This can be
            one of:

            - 'impute': Proceed with evaluation by filling in the missing
                        values with the mean of the training data. Missing
                        values are also imputed if an entire column of data is
                        missing during evaluation.
            - 'error' : Do not proceed with prediction and terminate with
                        an error message.

        Returns
        -------
        out : SFrame
            An SFrame with model predictions i.e class labels and probabilities
            associated with each of the class labels.

        See Also
        ----------
        create, evaluate, predict

        Examples
        ----------
        >>> data =  graphlab.SFrame('http://s3.amazonaws.com/GraphLab-Datasets/regression/houses.csv')

        >>> data['is_expensive'] = data['price'] > 30000
        >>> model = graphlab.boosted_trees_classifier.create(data,
        >>>                           target='is_expensive',
                                      features=['bath', 'bedroom', 'size'])

        >>> classes = model.classify(data)

        """

        _mt._get_metric_tracker().track(
            'toolkit.classifier.boosted_trees_classifier.classify')
        return super(BoostedTreesClassifier, self).classify(dataset,
                      missing_value_action = missing_value_action)

    @classmethod
    def _get_queryable_methods(cls):
        '''Returns a list of method names that are queryable through Predictive
        Service'''
        return {'predict':{'dataset':'sframe'},
                'predict_topk':{'dataset':'sframe'},
                'classify':{'dataset':'sframe'}}


def create(dataset, target,
           features=None, max_iterations=10,
           validation_set=None,
           verbose=True, class_weights = None,
           **kwargs):
    """
    Create a (binary or multi-class) classifier model of type
    :class:`~graphlab.boosted_trees_classifier.BoostedTreesClassifier` using
    gradient boosted trees (sometimes known as GBMs).

    Parameters
    ----------
    dataset : SFrame
        A training dataset containing feature columns and a target column.
        The target column must be of numerical type (int, float).

    target : str
        Name of the column containing the target variable. The values in this
        column must be of string or integer type.  String target variables are
        automatically mapped to integers in alphabetical order of the variable values.
        For example, a target variable with 'cat', 'dog', and 'foosa' as possible
        values is mapped to 0, 1, and, 2 respectively.

    features : list[str], optional
        A list of columns names of features used for training the model.
        Defaults to None, which uses all columns in the SFrame ``dataset`` 
        excepting the target column..

    max_iterations : int, optional
        The maximum number of iterations for boosting. Each iteration results
        in the creation of an extra tree. 

    max_depth : float, optional
        Maximum depth of a tree.

    class_weights : {dict, `auto`}, optional

        Weights the examples in the training data according to the given class
        weights. If set to `None`, all classes are supposed to have weight one. The
        `auto` mode set the class weight to be inversely proportional to number of
        examples in the training data with the given class.


    step_size : float, [0,1], optional
        Step size (shrinkage) used in update to prevents overfitting.  It
        shrinks the prediction of each weak learner to make the boosting
        process more conservative.  The smaller the step size, the more conservative 
        the algorithm will be. Smaller step_size work well when 
        `max_iterations` is large.

    min_loss_reduction : float, optional
        Minimum loss reduction required to make a further partition on a
        leaf node of the tree. The larger it is, the more conservative the
        algorithm will be.

    min_child_weight : float, optional
        This controls the minimum number of instances needed for each leaf.
        The larger it is, the more conservative the algorithm will be.  Set
        it larger when you want to prevent overfitting.  Formally, this is
        minimum sum of instance weight (hessian) in each leaf.  If the tree
        partition step results in a leaf node with the sum of instance
        weight less than min_child_weight, then the building process will
        give up further partitioning. For a regression task, this simply
        corresponds to minimum number of instances needed to be in each
        node.

    row_subsample : float, optional
        Subsample the ratio of the training set in each iteration of tree
        construction.  This is called the bagging trick and can usually help
        prevent overfitting.  Setting this to a value of 0.5 results in the
        model randomly sampling half of the examples (rows) to grow each tree.

    column_subsample : float, optional
        Subsample ratio of the columns in each iteration of tree
        construction.  Like row_subsample, this can also help prevent
        model overfitting.  Setting this to a value of 0.5 results in the
        model randomly sampling half of the columns to grow each tree.

    validation_set : SFrame, optional
        The validation set used during the model creation process. Large 
        differences in model accuracy between the training data and validation
        data is indicative of overfitting.

    verbose : boolean, optional
        Print progress information during training (if set to true).

    Returns
    -------
      out : BoostedTreesClassifier
          A trained gradient boosted trees model for classifications tasks.

    References
    ----------
     - `Wikipedia - Gradient tree boosting
      <http://en.wikipedia.org/wiki/Gradient_boosting#Gradient_tree_boosting>`_
     - `Trevor Hastie's slides on Boosted Trees and Random Forest
      <http://jessica2.msri.org/attachments/10778/10778-boost.pdf>`_

    See Also
    --------
    BoostedTreesClassifier, graphlab.logistic_classifier.LogisticClassifier, graphlab.svm_classifier.SVMClassifier, graphlab.neuralnet_classifier.NeuralNetClassifier


    Examples
    --------

    .. sourcecode:: python

      >>> url = 'http://s3.amazonaws.com/gl-testdata/xgboost/mushroom.csv'
      >>> data = graphlab.SFrame.read_csv(url)
      >>> data['label'] = (data['label'] == 'p')
      
      >>> train, test = data.random_split(0.8)
      >>> model = graphlab.boosted_trees_classifier.create(train, target='label')
      
      >>> predicitons = model.classify(test)
      >>> results = model.evaluate(test)
    """
    _mt._get_metric_tracker().track('toolkit.classifier.boosted_trees_classifier.create')
    model = _sl.create(dataset = dataset,
                        target = target,
                        features = features,
                        model_name = 'boosted_trees_classifier',
                        max_iterations = max_iterations,
                        validation_set = validation_set,
                        class_weights = class_weights,
                        verbose = verbose, **kwargs)
    return BoostedTreesClassifier(model.__proxy__)
