"""
Methods for creating and querying a nearest neighbors model.
"""

import graphlab.connect as _mt
import graphlab as _graphlab
from graphlab.toolkits._model import Model as _Model
from graphlab.data_structures.sframe import SFrame as _SFrame
import graphlab.toolkits._internal_utils as _tkutl
from graphlab.toolkits._model import _get_default_options_wrapper as \
                                                  __get_default_options_wrapper
import array
import copy


get_default_options = __get_default_options_wrapper('nearest_neighbors_ball_tree',
                                                    'nearest_neighbors',
                                                    'NearestNeighborsModel')


def choose_auto_distance(feature_names, column_names, column_types):
    """
    Construct composite distance parameters based on selected features and their
    types.
    """

    ## Make a dictionary from the column_names and column_types
    col_type_dict = {k: v for k, v in zip(column_names, column_types)}

    ## Loop through feature names, appending a distance component if the
    #  feature's type is *not* numeric. If the type *is* numeric, append it to
    #  the numeric_cols list, then at the end make a numeric columns distance
    #  component.
    composite_distance_params = []
    numeric_cols = []

    for c in feature_names:
        if col_type_dict[c] == str:
            composite_distance_params.append([[c], _graphlab.distances.levenshtein, 1])
        elif col_type_dict[c] == dict:
            composite_distance_params.append([[c], _graphlab.distances.jaccard, 1])
        elif col_type_dict[c] == array.array:
            composite_distance_params.append([[c], _graphlab.distances.euclidean, 1])
        elif col_type_dict[c] == list:
            composite_distance_params.append([[c], _graphlab.distances.euclidean, 1])
        elif col_type_dict[c] in [int, float, array.array, list]:
            numeric_cols.append(c)
        else:
            raise TypeError("Unable to automatically determine a distance "+\
                "for column {}".format(c))

    # Make the standalone numeric column distance component
    if len(numeric_cols) > 0:
        composite_distance_params.append([numeric_cols, _graphlab.distances.euclidean, 1])

    return composite_distance_params


def create(dataset, label=None, features=None, distance='auto', method='auto',
    composite_params=None, verbose=True, **kwargs):
    """
    Create a nearest neighbor model, which can be searched efficiently and
    quickly for the nearest neighbors of a query observation. If the `method`
    argument is specified as `auto`, the type of model is chosen automatically
    based on the type of data in `dataset`.

    Parameters
    ----------
    dataset : SFrame
        Reference data. If the features for each observation are numeric, they
        may be in separate columns of 'dataset' or a single column with lists of
        values. The features may also be in the form of a column of sparse
        vectors (i.e. dictionaries), with string keys and numeric values.

    label : string, optional
        Name of the SFrame column with row labels. If 'label' is not specified,
        row numbers are used to identify reference dataset rows when the model
        is queried.

    features : list[string], optional
        Name of the columns with features to use in computing distances between
        observations and the query points. 'None' (the default) indicates that
        all columns except the label should be used as features. Each column can
        be one of the following types:

        - *Numeric*: values of numeric type integer or float.

        - *Array*: list of numeric (integer or float) values. Each list element
          is treated as a separate variable in the model.

        - *Dictionary*: key-value pairs with numeric (integer or float) values.
          Each key indicates a separate variable in the model.

        - *String*: string values.

        Columns of type *list* are not supported. Convert them to array columns
        if all entries in the list are of numeric types. Please note: if
        `composite_params` is also specified, this parameter is ignored.

    distance : string or function, optional
        Name of the function that measures the distances between two
        observations. Please see the notes and references for detailed
        descriptions of the distances. Note that for sparse vectors, missing
        keys are assumed to have value 0.0. Please note: if `composite_params`
        is also specified, this parameter is ignored.

        - *auto* (default): the model chooses a reasonable distance based on the
          data types in 'dataset'. Columns of type str will be compared using
          levenshtein distance, columns of type dict use jaccard distance, and
          columns of type float, int, or list will be combined and use
          euclidean distance. The set of column-specific distances are
          aggregated into a single composite distance.

        - *squared_euclidean*: works only with the `brute_force` method because
          it is not a metric.

        - *euclidean*

        - *manhattan*

        - *jaccard*: works only with variables in a dictionary feature, where
          the keys are treated as a set and the values are ignored.

        - *weighted_jaccard*: like jaccard distance, works only with variables
          in a dictionary feature. For the weighted version of jaccard, however,
          the values of the dictionary are used to weight the contribution of
          each key. This is done by taking the minimum of the two values for
          each key in the numerator and the maximum of the two values in the
          denominator.

        - *cosine*: works only with the 'brute_force' method because it is not a
          true metric. Please see `Wikipedia
          <http://en.wikipedia.org/wiki/Cosine_similarity>`_ for more detail.

        - *dot_product*: works only with the 'brute_force' method because it is
          not a true metric.

        - *levenshtein*: for a single column of string inputs.

    method : {'auto', 'ball_tree', 'brute_force'}, optional
        Method for computing nearest neighbors. The options are:

        - *auto* (default): the method is chosen automatically, based on the
          type of data and the distance. If the distance is 'manhattan' or
          'euclidean' and the features are numeric or vectors of numeric values,
          then the 'ball_tree' method is used. Otherwise, the 'brute_force'
          method is used.

        - *ball_tree*: use a tree structure to find the k-closest neighbors to
          each query point. The ball tree model is slower to construct than the
          brute force model, but queries are faster than linear time. This
          method is not applicable for the cosine and dot product distances. See
          `Liu, et al (2004)
          <http://papers.nips.cc/paper/2666-an-investigation-of-p
          ractical-approximat e-nearest-neighbor-algorithms>`_ for
          implementation details.

        - *brute_force*: compute the distance from a query point to all
          reference observations. There is no computation time for model
          creation with the brute force method (although the reference data is
          held in the model, but each query takes linear time.

    composite_params : list [list [list [string], string or function, float]]
        Multiple sets of features and corresponding distance functions can be
        used as inputs to a composite distance function. Each element of this
        composite is specified by a list in this argument. Each inner list must
        include a list of feature names, the name of a distance function, and a
        relative weight. See the examples and notes sections below. If
        `composite_params` is specified, any standalone `features`, `distance`,
        and `method` arguments are ignored. Keyword arguments are applied to
        each member of the composite distance computation.

    verbose: bool, optional
        If True, print progress updates and model details.

    **kwargs : optional
        Options for the distance function and query method.

        - *leaf_size*: for the ball tree method, the number of points in each
          leaf of the tree. The default is to use the max of 1,000 and n/(2^11),
          which ensures a maximum tree depth of 12. The default leaf size is
          indicated by a "0" in the
          :func:`~graphlab.nearest_neighbors.NearestNeighborsModel.get_default_options`
          method.

    Returns
    -------
    out : NearestNeighborsModel
        A structure for efficiently computing the nearest neighbors in 'dataset'
        of new query points.

    See Also
    --------
    NearestNeighborsModel.query

    Notes
    -----
    - If the features should be weighted equally in the distance calculations
      but are measured on different scales, it is important to standardize the
      features. One way to do this is to subtract the mean of each column and
      divide by the standard deviation.

    - Distance definitions. Suppose :math:`u` and :math:`v` are observations
      with :math:`d` variables each.

        - `squared_euclidean`
            .. math:: D(u, v) = \sum_i^d (u_i - v_i)^2

        - `euclidean`
            .. math:: D(u, v) = \\sqrt{\sum_i^d (u_i - v_i)^2}

        - `manhattan`
            .. math:: D(u, v) = \\sum_i^d |u_i - v_i|

        - `cosine`
            .. math::

                D(u, v) = 1 - \\frac{\sum_i^d u_i v_i}
                {\sqrt{\sum_i^d u_i^2}\sqrt{\sum_i^d v_i^2}}

        - `dot_product`
            .. math::

                D(u, v) = \\frac{1}{\sum_i^d u_i v_i}

    - For the jaccard distances, suppose :math:`S` and :math:`T` are the sets of
      keys from two observations' dictionaries. For the weighted version of
      jaccard distance, suppose :math:`S_k` and :math:`T_k` are the values
      associated with key :math:`k` in the respective dictionaries. Typically
      these values are counts, i.e. of words or n-grams.

        - `jaccard`
            .. math:: D(S, T) = 1 - \\frac{|S \cap T|}{|S \cup T|}

        - `weighted_jaccard`
            .. math::

                D(S, T) = 1 - \\frac{\sum_{k \in S \cup T} \min\{S_k, T_k\}}
                {\sum_{k \in S \cup T} \max\{S_k, T_k\}}

    - Levenshtein distance is a type of edit distance for string types. The
      distance is the number of insertion, deletion, and substituion edits
      needed to transform string :math:`A` into string :math:`B`.

        .. math::

            D(A, B) = d(|A|, |B|)

        .. math ::

            d(i, j) = \max(i, j), \quad \mathrm{if } \min(i, j) = 0

        .. math ::

            d(i, j) = \min \Big \{d(i-1, j) + 1, \ d(i, j-1) + 1, \ d(i-1, j-1) + I(A_i \\neq B_i) \Big \}, \quad \mathrm{else}

    - Composite distances are simply weighted sums of the above distances. The
      set of features input to each component distance may vary, and the weight
      on each component acts as a multiplier before each of the component
      distances is summed. For example, if ``composite_params`` is set to

      ``[[['X1', 'X2'], 'euclidean', 2], [['X2', 'X3'], 'manhattan', 3]]``,

      then the overall distance computation for rows :math:`a` and :math:`b` is:

        .. math::

            D(a, b) = 2 * D_{euclidean}(a[[X1, X2]], b[[X1, X2]]) +
            3 * D_{manhattan}(a[[X2, X3]], b[[X2, X3]])

    References
    ----------
    - `Wikipedia - nearest neighbor
      search <http://en.wikipedia.org/wiki/Nearest_neighbor_search>`_

    - `Wikipedia - ball tree <http://en.wikipedia.org/wiki/Ball_tree>`_

    - Ball tree implementation: Liu, T., et al. (2004) `An Investigation of
      Practical Approximate Nearest Neighbor Algorithms
      <http://papers.nips.cc/paper/2666-an-investigation-of-p
      ractical-approximat e-nearest-neighbor-algorithms>`_. Advances in Neural
      Information Processing Systems pp. 825-832.

    - `Wikipedia - Jaccard distance
      <http://en.wikipedia.org/wiki/Jaccard_index>`_

    - Weighted Jaccard distance: Chierichetti, F., et al. (2010) `Finding the
      Jaccard Median
      <http://theory.stanford.edu/~sergei/papers/soda10-jaccard.pdf>`_.
      Proceedings of the Twenty-First Annual ACM-SIAM Symposium on Discrete
      Algorithms. Society for Industrial and Applied Mathematics.

    - `Wikipedia - Cosine distance
      <http://en.wikipedia.org/wiki/Cosine_similarity>`_

    - `Wikipedia - Levenshtein distance
      <http://en.wikipedia.org/wiki/Levenshtein_distance>`_

    Examples
    --------
    Construct a nearest neighbors model with automatically determined method and
    distance:

    >>> sf = graphlab.SFrame({'X1': [0.98, 0.62, 0.11],
    ...                       'X2': [0.69, 0.58, 0.36],
                              'str_feature': ['cat', 'dog', 'fossa']})
    >>> model = graphlab.nearest_neighbors.create(sf, features=['X1', 'X2'])

    For datasets with a large number of rows and up to about 100 variables, the
    ball tree method often leads to much faster queries.

    >>> model = graphlab.nearest_neighbors.create(sf, features=['X1', 'X2'],
    ...                                           method='ball_tree')

    Often the final determination of a neighbor is based on several distance
    computations over different sets of features. Each part of this composite
    distance may have a different relative weight.

    >>> model = graphlab.nearest_neighbors.create(sf, composite_params=[
    ...                                          [['X1', 'X2'], 'euclidean', 2.],
    ...                                          [['str_feature'], 'levenshtein', 3.]])
    """

    ## Validate the 'dataset' input
    _tkutl._raise_error_if_not_sframe(dataset, "dataset")
    _tkutl._raise_error_if_sframe_empty(dataset, "dataset")


    ## Exclude inappropriate combinations of method an distance
    if method == 'ball_tree' and (distance == 'cosine'
                                  or distance == _graphlab.distances.cosine
                                  or distance == 'dot_product'
                                  or distance == _graphlab.distances.dot_product):
        raise TypeError("The ball tree method does not work with 'cosine' " +\
                        "or 'dot_product' distance. Please use the 'brute_force' " +\
                        "method for these distances.")


    ## Initial validation and processing of the label
    if label is None:
        _label = '__id'

        try:
            _dataset = dataset.add_row_number(column_name=_label)
        except:
            print "Tried to add a default label column '{}' ".format(_label) +\
                  "but a column by this name already exists. Using the " + \
                  "existing column as the label column."
            _dataset = dataset

    else:
        if not label in dataset.column_names():
            raise ValueError(
                "Input 'label' must be a string matching the name of a " +\
                "column in the reference SFrame 'dataset'.")

        if not dataset[label].dtype() == str and not dataset[label].dtype() == int:
            raise TypeError("The label column must contain integers or strings.")

        _label = label
        _dataset = dataset

    sf_label = _tkutl._toolkits_select_columns(_dataset, [_label])


    ## Clean the method options and create the options dictionary
    if len(kwargs) > 0:
        _method_options = {k.lower(): v for k, v in kwargs.items()}
    else:
        _method_options = {}



    ## If composite inputs aren't specifed, formulate the standalone inputs as a
    #  composite input for code simplicity. If the standalone input doesn't
    #  specify features, choose all the features and make a set of distance
    #  components based on feature type.
    if composite_params is None:

        ## If not features specified, use them all
        if features is None:
            _features = [x for x in _dataset.column_names() if x != _label]
        else:
            _features = features[:]

        ## If the distance argument is 'auto', turn it into a list of distance
        #  components by choosing an automatic distance for each feature based
        #  on its type.
        if distance == 'auto':
            _composites = choose_auto_distance(_features,
                                               dataset.column_names(),
                                               dataset.column_types())

        else:
            _composites = [[_features, distance, 1]]


    # Ignore automatically generated components if components have been provided
    else:
        if distance != 'auto':
            raise ValueError(
                "Either the 'distance' parameter or the 'composite_params' " +\
                "parameter may be specified, but not both.")

        if features is not None:
            raise ValueError(
                "Either the 'features' parameter or the 'composite_params' " +\
                "parameter may be specified, but not both.")

        if len(composite_params) == 0:
            raise ValueError(
                "'composite_params' was specified as an empty list. If " +\
                "specified, this parameter must contain at least one distance " +\
                "component, which is a list containing three elements: a list " +\
                "of feature names, a distance name or function, and a relative " +\
                "weight.")

        _composites = copy.deepcopy(composite_params)



    ## Clean the list of features in each component of the composite inputs, and
    #  compile the union of the lists of features.
    all_features = []

    for i in range(len(_composites)):

        if len(_composites[i]) != 3:
            raise ValueError("Each element of 'composite_params' must be a " +\
                             "list with three members.")

        feature_names = _composites[i][0]
        if len(feature_names) == 0:
            raise ValueError("An empty list of features cannot be passed " +\
                             "as part of a composite distance function.")

        # set of features must be iterable
        _tkutl._raise_error_if_not_iterable(feature_names, "features")

        # feature names must be strings
        if not all([isinstance(x, str) for x in feature_names]):
            raise TypeError("Input 'features' must contain only strings.")

        # remove the label name from all of the features lists
        feature_names = [x for x in feature_names if x != _label]

        # ensure that string features are in single columns
        if len(feature_names) > 1 and any([_dataset[x].dtype() is str for x in feature_names]):
            raise ValueError(                             
                "Multiple features have been entered, one of which is of string " +\
                "type. If the input features for any distance component contain a " +\
                "string column, that must be the only column for that component.")

        # ensure that relative weights are integers or floats
        if not isinstance(_composites[i][2], (int, float)):
            raise ValueError(
                "The weight of each distance component must be a single " +\
                "integer or a float value.")

        # combine all features into a big list
        _composites[i][0] = feature_names
        all_features += feature_names

        # convert distance strings to distance functions
        temp_dist = _composites[i][1]
        if isinstance(temp_dist, str):
            _composites[i][1] = _graphlab.util._get_distance(temp_dist)


    # Pull out the relevant features from the input dataset (the union of
    # features over all distance components)
    all_features = list(set(all_features))
    sf_features = _tkutl._toolkits_select_columns(_dataset, all_features)


    ## Decide which method to use
    ## - If more than one distance component (specified either directly or
    #  generated automatically because distance set to 'auto'), then do brute
    #  force.
    if len(_composites) > 1:
        _method = 'brute_force'

        if method == 'ball_tree':
            print "Defaulting to brute force instead of ball tree because " +\
                "there are multiple distance components."

    else:

        if method == 'auto':

            # get the total number of variables. Assume the number of elements in
            # array type columns does not change
            num_variables = sum([len(x) if hasattr(x, '__iter__') else 1
                for x in sf_features[0].itervalues()])

            # flag if all the features in the single composite are of numeric
            # type.
            numeric_type_flag = all([x in [int, float, list, array.array]
                for x in sf_features.column_types()])

            ## Conditions necessary for ball tree to work and be worth it
            if ((_composites[0][1] in ['euclidean',
                                       'manhattan',
                                       _graphlab.distances.euclidean,
                                       _graphlab.distances.manhattan])
                    and numeric_type_flag is True
                    and num_variables <= 100):

                    _method = 'ball_tree'

            else:
                _method = 'brute_force'

        else:
            _method = method


    ## Pick the right model name for the method
    if _method == 'ball_tree':
        model_name = 'nearest_neighbors_ball_tree'
        _mt._get_metric_tracker().track('toolkit.nearest_neighbors_balltree.create')

    elif _method == 'brute_force':
        model_name = 'nearest_neighbors_brute_force'
        _mt._get_metric_tracker().track('toolkit.nearest_neighbors_brute.create')

    else:
        raise ValueError("Method must be 'brute_force', 'ball_tree', or 'auto'")


    ## Package the model options
    opts = {}
    opts.update(_method_options)
    opts.update(
        {'model_name': model_name,
        'sf_label': sf_label,
        'sf_features': sf_features,
        'composite_params': _composites})

    ## Construct the nearest neighbors model
    if verbose:
        print "Starting model construction..."

    result = _graphlab.extensions._nearest_neighbors.train(opts)
    model_proxy = result['model']
    model = NearestNeighborsModel(model_proxy)

    if verbose:
        model.summary()
        print

    return model


class NearestNeighborsModel(_Model):
    """
    The NearestNeighborsModel represents rows of an SFrame in a structure that
    is used to quickly and efficiently find the nearest neighbors of a query
    point.

    This model cannot be constructed directly.  Instead, use
    :func:`graphlab.nearest_neighbors.create` to create an instance
    of this model. A detailed list of parameter options and code samples
    are available in the documentation for the create function.
    """

    def __init__(self, model_proxy):
        """___init__(self)"""
        self.__proxy__ = model_proxy
        self.__name__ = 'nearest_neighbors'

    def _get_wrapper(self):
        def model_wrapper(model_proxy):
            return NearestNeighborsModel(model_proxy)
        return model_wrapper

    def __str__(self):
        """
        Return a string description of the model to the ``print`` method.

        Returns
        -------
        out : string
            A description of the NearestNeighborsModel.
        """
        return self.__repr__()

    def __repr__(self):
        """
        Print a string description of the model when the model name is entered
        in the terminal.
        """

        model_fields = [
            ("Distance", 'distance'),
            ("Method", 'method'),
            ("Number of examples", 'num_examples'),
            ("Number of feature columns", 'num_features'),
            ("Number of unpacked features", 'num_unpacked_features'),
            ("Total training time (seconds)", 'training_time')]

        ball_tree_fields = [
            ("Tree depth", 'tree_depth'),
            ("Leaf size", 'leaf_size')]

        out = [model_fields]
        if self.get('method') == 'ball tree':
            out += [ball_tree_fields]

        return _tkutl._toolkit_repr_print(self, out, width=30)

    def summary(self):
        """
        Display a summary of the NearestNeighborsModel.

        Examples
        --------
        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')
        >>> model.summary()
                                    Model summary
        --------------------------------------------------------
        Class                   : NearestNeighborsModel
        Distance                : euclidean
        Method                  : ball tree
        Examples                : 3
        Features                : 2
        Variables               : 2
        Training time (sec)     : 0.0232
        Tree depth              : 1
        Leaf size               : 1000
        """

        _mt._get_metric_tracker().track('toolkit.nearest_neighbors.summary')

        print ""
        print "                    Model summary                       "
        print "--------------------------------------------------------"
        print self.__repr__()

    def list_fields(self):
        """
        List the fields stored in the model, including data, model, and training
        options. Each field can be queried with the ``get`` method.

        Returns
        -------
        out : list
            List of fields queryable with the ``get`` method.

        See Also
        --------
        get

        Examples
        --------
        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')
        >>> model.list_fields()
        ['distance',
         'features',
         'label',
         'leaf_size',
         'method',
         'num_examples',
         'num_features',
         'num_variables',
         'training_time',
         'tree_depth']
        """

        _mt._get_metric_tracker().track('toolkit.nearest_neighbors.list_fields')

        opts = {'model': self.__proxy__, 'model_name': self.__name__}
        response = _graphlab.toolkits._main.run('_nearest_neighbors.list_keys',
                                               opts)

        return sorted(response.keys())

    def get(self, field):
        """
        Return the value of a given field. The list of all queryable fields is
        detailed below, and can be obtained with the
        :func:`~graphlab.nearest_neighbors.NearestNeighborsModel.list_fields`
        method.

        +-----------------------+----------------------------------------------+
        |      Field            | Description                                  |
        +=======================+==============================================+
        | distance              | Measure of dissimilarity between two points  |
        +-----------------------+----------------------------------------------+
        | features              | Feature column names                         |
        +-----------------------+----------------------------------------------+
        | unpacked_features     | Names of the individual features used        |
        +-----------------------+----------------------------------------------+
        | label                 | Label column names                           |
        +-----------------------+----------------------------------------------+
        | leaf_size             | Max size of leaf nodes (ball tree only)      |
        +-----------------------+----------------------------------------------+
        | method                | Method of organizing reference data          |
        +-----------------------+----------------------------------------------+
        | num_examples          | Number of reference data observations        |
        +-----------------------+----------------------------------------------+
        | num_features          | Number of features for distance computation  |
        +-----------------------+----------------------------------------------+
        | num_unpacked_features | Number of unpacked features                  |
        +-----------------------+----------------------------------------------+
        | num_variables         | Number of variables for distance computation |
        +-----------------------+----------------------------------------------+
        | training_time         | Time to create the reference structure       |
        +-----------------------+----------------------------------------------+
        | tree_depth            | Number of levels in the tree (ball tree only)|
        +-----------------------+----------------------------------------------+

        Parameters
        ----------
        field : string
            Name of the field to be retrieved.

        Returns
        -------
        out
            Value of the requested field.

        See Also
        --------
        list_fields

        Examples
        --------
        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')
        >>> model.get('num_features')
        2
        >>> model['training_time']
        0.023223
        """

        _mt._get_metric_tracker().track('toolkit.nearest_neighbors.get')

        opts = {'model': self.__proxy__,
                'model_name': self.__name__,
                'field': field}
        response = _graphlab.toolkits._main.run('_nearest_neighbors.get_value',
                                                opts)
        return response['value']


    def get_current_options(self):
        """
        Return a dictionary with the options used to define and create the
        current NearestNeighborModel instance.

        Returns
        -------
        out : dict
            Dictionary of options used to train the current instance of the
            NearestNeighborsModel.

        See Also
        --------
        get_default_options, list_fields, get

        Examples
        --------
        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')
        >>> model.get_current_options()
        {'distance': 'euclidean', 'leaf_size': 1000}
        """

        _mt._get_metric_tracker().track(
            'toolkit.nearest_neighbors.get_current_options')

        opts = {'model': self.__proxy__, 'model_name': self.__name__}

        return _graphlab.toolkits._main.run(
            '_nearest_neighbors.get_current_options', opts)

    def _training_stats(self):
        """
        Return a dictionary of statistics collected during creation of the
        model. These statistics are also available with the ``get`` method and
        are described in more detail in that method's documentation.

        Returns
        -------
        out : dict
            Dictionary of statistics compiled during creation of the
            NearestNeighborsModel.

        See Also
        --------
        summary

        Examples
        --------
        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')
        >>> model.training_stats()
        {'features': 'feature1, feature2',
         'label': 'label',
         'leaf_size': 1000,
         'num_examples': 3,
         'num_features': 2,
         'num_variables': 2,
         'training_time': 0.023223,
         'tree_depth': 1}
        """

        _mt._get_metric_tracker().track(
            'toolkit.nearest_neighbors.training_stats')

        opts = {'model': self.__proxy__, 'model_name': self.__name__}
        return _graphlab.toolkits._main.run("_nearest_neighbors.training_stats",
                opts)

    def query(self, dataset, label=None, k=5, radius=None, verbose=True):
        """
        Retrieve the nearest neighbors from the reference set for each element
        of the query set. The query SFrame must include columns with the same
        names as the label and feature columns used to create the
        NearestNeighborsModel.

        Parameters
        ----------
        dataset : SFrame
            Query data. Must contain columns with the same names and types as
            the features used to train the model. Additional columns are
            allowed, but ignored. Please see the nearest neighbors
            :func:`~graphlab.nearest_neighbors.create` documentation for more
            detail on allowable data types.

        label : string, optional
            Name of the query SFrame column with row labels. If 'label' is not
            specified, row numbers are used to identify query dataset rows in
            the output SFrame.

        k : int, optional
            Number of nearest neighbors to return from the reference set for
            each query observation. The default is 5 neighbors, but setting it
            to ``None`` will return all neighbors within ``radius`` of the query
            point.

        radius : float, optional
            Only neighbors whose distance to a query point is smaller than this
            value are returned. The default is ``None``, in which case the ``k``
            nearest neighbors are returned for each query point, regardless of
            distance.

        verbose: bool, optional
            If True, print progress updates and model details.

        Returns
        -------
        out : SFrame
            An SFrame with the k-nearest neighbors of each query observation.
            The result contains four columns: the first is the label of the
            query observation, the second is the label of the nearby reference
            observation, the third is the distance between the query and
            reference observations, and the fourth is the rank of the reference
            observation among the query's k-nearest neighbors.

        Notes
        -----
        - If both ``k`` and ``radius`` are set to ``None``, each query point
          returns all of the reference set. If the reference dataset has
          :math:`n` rows and the query dataset has :math:`m` rows, the output is
          an SFrame with :math:`nm` rows.

        Examples
        --------
        First construct a toy SFrame and create a nearest neighbors model:

        >>> sf = graphlab.SFrame({'label': range(3),
        ...                       'feature1': [0.98, 0.62, 0.11],
        ...                       'feature2': [0.69, 0.58, 0.36]})
        >>> model = graphlab.nearest_neighbors.create(sf, 'label')

        A new SFrame contains query observations with same schema as the
        reference SFrame. This SFrame is passed to the ``query`` method.

        >>> queries = graphlab.SFrame({'label': range(3),
        ...                            'feature1': [0.05, 0.61, 0.99],
        ...                            'feature2': [0.06, 0.97, 0.86]})
        >>> model.query(queries, 'label', k=2)
        +-------------+-----------------+----------------+------+
        | query_label | reference_label |    distance    | rank |
        +-------------+-----------------+----------------+------+
        |      0      |        2        | 0.305941170816 |  1   |
        |      0      |        1        | 0.771556867638 |  2   |
        |      1      |        1        | 0.390128184063 |  1   |
        |      1      |        0        | 0.464004310325 |  2   |
        |      2      |        0        | 0.170293863659 |  1   |
        |      2      |        1        | 0.464004310325 |  2   |
        +-------------+-----------------+----------------+------+
        """

        _mt._get_metric_tracker().track(
            'toolkit.nearest_neighbors.query')

        ## Validate the 'dataset' input
        _tkutl._raise_error_if_not_sframe(dataset, "dataset")
        _tkutl._raise_error_if_sframe_empty(dataset, "dataset")

        ## Get model features
        ref_features = self.get('features')
        sf_features = _tkutl._toolkits_select_columns(dataset, ref_features)

        ## Validate and preprocess the 'label' input
        ref_label = self.get('label')

        if label is None:
            sf_features = sf_features.add_row_number(column_name=ref_label)
            sf_label = sf_features[[ref_label]]
            sf_features.remove_column(ref_label)

        else:
            if not label in dataset.column_names():
                raise ValueError(
                    "Input 'label' must be a string matching the name of a " +\
                    "column in the reference SFrame 'dataset'.")

            if not dataset[label].dtype() == str and not dataset[label].dtype() == int:
                raise TypeError("The label column must contain integers or strings.")

            if label in ref_features:
                raise ValueError("The label column cannot be one of the features.")

            sf_label = _tkutl._toolkits_select_columns(dataset, [label])

            if label != ref_label:
                sf_label.rename({label: ref_label})


        ## Validate neighborhood parameters 'k' and 'radius'
        if k is not None:
            if not isinstance(k, int):
                raise ValueError("Input 'k' must be an integer.")

            if k <= 0:
                raise ValueError("Input 'k' must be larger than 0.")

        if radius is not None:
            if not isinstance(radius, (int, float)):
                raise ValueError("Input 'radius' must be an integer or float.")

            if radius < 0:
                raise ValueError("Input 'radius' must be non-negative.")


        ## Set k and radius to special values to indicate 'None'
        if k is None:
            k = -1

        if radius is None:
            radius = -1.0

        opts = {'model': self.__proxy__,
                'model_name': self.__name__,
                'features': sf_features,
                'label': sf_label,
                'k': k,
                'radius': radius}

        if verbose is True:
            print "Starting model querying..."

        result = _graphlab.toolkits._main.run('_nearest_neighbors.query', opts,
                                             verbose)
        return _SFrame(None, _proxy=result['neighbors'])

    @classmethod
    def _get_queryable_methods(cls):
        '''Returns a list of method names that are queryable through Predictive
        Service'''
        return {'query':{'dataset':'sframe'}}

