from graphlab.canvas.views.bivariate import BivariateView as _bivariate

import graphlab as gl
import math
import threading
import re

class HeatmapView(_bivariate):

    #default bins on side
    BINS_ON_SIDE = 300;
    #maximum # of rows in individually processed SFrame chunks
    MAX_CHUNK_SIZE = 50000;
    #sample size from which to draw initial extrema
    INITIAL_EXTREMA_SAMPLE_SIZE = 1000;
    #(sample size to probe to find initial extrema upper bound)
    PROBING_SAMPLE_SIZE = 1000000;

    def __init__(self, obj, params):
        # obj is an SFrame
        # col1 is the name of the column in the X axis
        # col2 is the name of the column in the Y axis
        super(HeatmapView, self).__init__(obj, params)
        self.bins = None
        self.binsX = None
        self.binsY = None
        self.complete = False
        self.progress = 0
        self.thread = None
        self.xWidth = 0
        self.yWidth = 0

        #this extrema is the extrema of the binning domain (the area that defines the bins we are binning into)
        self.xMin = None
        self.xMax = None
        self.yMin = None
        self.yMax = None

        #this extrema is the actual extrema values we have seen so far and is used to define the viewport we show
        self.realXMin = None
        self.realXMax = None
        self.realYMin = None
        self.realYMax = None

        #whether or not to terminate binning thread
        self.shouldTerminateThread = False

        self.lock = threading.Lock()

        self.col1 = self.get_numeric_column('x')
        self.col2 = self.get_numeric_column('y', self.col1)

        #setting number of bins ons side based on number of values
        if(self.obj.num_rows() < 90000):
            self.BINS_ON_SIDE = int(math.sqrt(self.obj.num_rows())) * 2
        else:
            self.BINS_ON_SIDE = 300

        self.currentSliceIndex = 0

    def get_staticdata(self):
        return self._get_bins_ipython()


    #this function runs on another thread and incrementally updates self.bins
    def create_bins(self):
        #initializing the values
        self.bins = [[0]* self.BINS_ON_SIDE for _ in range(self.BINS_ON_SIDE)]
        self.currentSliceIndex = 0;

        #setting up initial bounds of extrema (these will later be double / expanded upon if points lie outside its bounds...)
        extremaSampleArray = self.obj[:self.INITIAL_EXTREMA_SAMPLE_SIZE]
        self.xMin = float(extremaSampleArray[self.col1].min())
        self.xMax = float(extremaSampleArray[self.col1].max())
        self.yMin = float(extremaSampleArray[self.col2].min())
        self.yMax = float(extremaSampleArray[self.col2].max())

        #the actual extrema of the dataset
        self.realXMin = self.xMin
        self.realXMax = self.xMax
        self.realYMin = self.yMin
        self.realYMax = self.yMax

        #probing the chunk to find initial extrema to minimize the need to rebin (rebinning -> poor performance)
        probingSlice = self.obj[:self.PROBING_SAMPLE_SIZE]
        probingXMin = probingSlice[self.col1].min()
        probingXMax = probingSlice[self.col1].max()
        probingYMin = probingSlice[self.col2].min()
        probingYMax = probingSlice[self.col2].max()

        #go through first couple points to reduce majority of extrema doubling in the main loop
        if probingXMin < self.xMin:
            self.xMin -= self.xMax - self.xMin

        if probingXMax > self.xMax:
            self.xMax += self.xMax - self.xMin

        if probingYMin < self.yMin:
            self.yMin -= self.yMax - self.yMin

        if probingYMax > self.yMax:
            self.yMax += self.yMax - self.yMin

        #convenience variables
        xLabel = self.col1
        yLabel = self.col2
        binsOnSide = self.BINS_ON_SIDE
        self.complete = False

        #the main loop that continues throughout the binning process
        while not self.complete:
            #finding the start and end points of the current slice
            start = self.currentSliceIndex
            end = start + self.MAX_CHUNK_SIZE
            if end > self.obj.num_rows():
                end = self.obj.num_rows()

            #using a lock to update variables
            with self.lock:


                #if we need to terminate the thread (due to another thread), do so
                if self.shouldTerminateThread:
                    self.shouldTerminateThread = False
                    return

                
                #if the columns have changed, end the task
                if(xLabel != self.col1 or yLabel != self.col2):
                    self.complete = True
                    self.progress = 1
                    return


                #getting the slice we are about to iterate through and finding its extrema
                # dropna to avoid capturing null values
                pointsSlice = self.obj[start:end].dropna(columns=list(set([self.col1, self.col2])))
                sliceXMin = pointsSlice[self.col1].min()
                sliceXMax = pointsSlice[self.col1].max()
                sliceYMin = pointsSlice[self.col2].min()
                sliceYMax = pointsSlice[self.col2].max()


                #setting the actual extrema (distinct from binning extrema) to any newly found max/min vals(where there are points!)
                if sliceXMin < self.realXMin:
                    self.realXMin = sliceXMin
                if sliceXMax > self.realXMax:
                    self.realXMax = sliceXMax
                if sliceYMin < self.realYMin:
                    self.realYMin = sliceYMin
                if sliceYMax > self.realYMax:
                    self.realYMax = sliceYMax


                #if extrema are outside the binning bounds, reset extrema, and rebin
                if sliceXMin < self.xMin:
                    self.xMin -= max(self.xMax-self.xMin, self.xMin - sliceXmin)
                    self.bins = [[0]* self.BINS_ON_SIDE for _ in range(self.BINS_ON_SIDE)]
                    self.currentSliceIndex = 0;
                    continue;

                if sliceXMax > self.xMax:
                    self.xMax += max(self.xMax-self.xMin, sliceXMax - self.xMax)
                    self.bins = [[0]* self.BINS_ON_SIDE for _ in range(self.BINS_ON_SIDE)]
                    self.currentSliceIndex = 0;
                    continue;

                if sliceYMin < self.yMin:
                    self.yMin -= max(self.yMax-self.yMin, self.yMin - sliceYMin)
                    self.bins = [[0]* self.BINS_ON_SIDE for _ in range(self.BINS_ON_SIDE)]
                    self.currentSliceIndex = 0;
                    continue;

                if sliceYMax > self.yMax:
                    self.yMax += max(self.yMax-self.yMin, sliceYMax - self.yMax)
                    self.bins = [[0]* self.BINS_ON_SIDE for _ in range(self.BINS_ON_SIDE)]
                    self.currentSliceIndex = 0;
                    continue;


                #width of the bin / height of the bin
                self.xWidth = (self.xMax - self.xMin)/self.BINS_ON_SIDE 
                self.yWidth = (self.yMax - self.yMin)/self.BINS_ON_SIDE

                #setting self variables to temporary local variables (for use inside the lambda)
                xMin = self.xMin
                yMax = self.yMax
                xMax = self.xMax
                yMin = self.yMin
                xWidth = self.xWidth
                yWidth = self.yWidth

                
                #lambda implementation
                #gets the indices of the region every point maps to (uses ternary operators to deal with the edge case of the point being equal to xMax or yMin) (encodes i, j values to a float)
                #mappedToIndices = gl.SFrame({ 'indices': pointsSlice.apply(lambda row: binsOnSide * (((row[xLabel] - xMin)//xWidth) if row[xLabel] != xMax else binsOnSide-1) + (((yMax - row[yLabel])//yWidth) if row[yLabel] != yMin else binsOnSide-1))})

                #vectorized implementation
                #finds the indices that are mapped to by every single value pair in the slice
                mappedToIndices = gl.SFrame({ 'indices': binsOnSide * ((pointsSlice[xLabel] - xMin)/xWidth).astype(int) + ((yMax - pointsSlice[yLabel])/yWidth).astype(int)})

                #aggregating all of the elements that maped to the same index in a count SFrame
                countSF = mappedToIndices.groupby('indices', {'count': gl.aggregate.COUNT})


                #for every index mapped to, increment corresponding weight in bin
                for row in countSF:
                    iIndex = int(row["indices"]//binsOnSide)
                    jIndex = int(row["indices"] % binsOnSide)
                    if(iIndex >= binsOnSide):
                        iIndex = binsOnSide-1
                    if(jIndex >= binsOnSide):
                        jIndex = binsOnSide-1
                    #increment the bins, if an error occurs due to edge cases in vectorized implementation (i.e. weird value) move on
                    try:
                        self.bins[iIndex][jIndex] += row["count"]
                    except(IndexError):
                        pass

                #setting progress and completion variables
                self.currentSliceIndex = end
                self.progress = float(end) / float(self.obj.num_rows())
                if self.progress >= 1:
                    self.complete = True

    def _get_bins_ipython(self):
        # same as _get_bins but synchronous, for IPython Notebook output
        if (self.bins == None):
            self.create_bins()
        extrema = {'minX': self.xMin, 'maxX': self.xMax, 'minY': self.yMin, 'maxY': self.yMax}
        #the ACTUAL  extrema of the data set
        domainToShow = {'minX': self.realXMin, 'maxX': self.realXMax, 'minY': self.realYMin, 'maxY': self.realYMax}
        return {'bins': self.bins, 'binWidth': self.xWidth, 'binHeight': self.yWidth, 'extrema': extrema, 'domainToShow': domainToShow, 'pointsLoaded': self.currentSliceIndex, 'progress': self.progress, 'complete': self.complete}

    #called to return all the values. When dealing with large datasets, called periodically while binning occurs
    def _get_bins(self, url, handler):
        if self.col1 is None or \
           self.col2 is None:
            handler.write({})
            return

        #if the process of binning the points hasn't started, start a thread to do that
        if self.bins == None or \
           self.binsX != self.col1 or \
           self.binsY != self.col2:
            # reset to new bin columns
            self.binsX = self.col1
            self.binsY = self.col2

            # terminate existing thread
            if self.thread is not None:
                self.shouldTerminateThread = True
                self.thread.join()
                self.thread = None
                self.progress = 0
                self.complete = False

            # start new thread
            self.shouldTerminateThread = False
            self.thread = threading.Thread(target = self.create_bins)
            self.thread.start()

        #lock before writing to prevent multithreaded conflicts
        with self.lock:
            #the extrema used for binning
            extrema = {'minX': self.xMin, 'maxX': self.xMax, 'minY': self.yMin, 'maxY': self.yMax}
            #the ACTUAL  extrema of the data set
            domainToShow = {'minX': self.realXMin, 'maxX': self.realXMax, 'minY': self.realYMin, 'maxY': self.realYMax}
            handler.write({'bins': self.bins, 'binWidth': self.xWidth, 'binHeight': self.yWidth, 'extrema': extrema, 'domainToShow': domainToShow, 'pointsLoaded': self.currentSliceIndex, 'progress': self.progress, 'complete': self.complete})
