import graphlab_util.cloudpickle as _cloudpickle
import graphlab as _gl
import _artifact
import graphlab.canvas as _canvas

import boto.ec2 as _boto
from boto import connect_s3 as _connect_s3
from boto.s3.key import Key as _s3_key

import cStringIO as _cStringIO
import os as _os
import pickle as _pickle
import pprint as _pprint
import tempfile as _tempfile
import zipfile as _zipfile
import graphlab_psutil as _psutil
import requests as _requests
import os as _os
import shutil as _shutil
import copy as _copy
import subprocess as _subprocess
import datetime as _datetime
import json

import logging
from inspect import isfunction as _isfunction, ismethod as _ismethod


__LOGGER__ = logging.getLogger(__name__)


class Job(object):
    """
    Jobs represent executions of Tasks. Job objects should not be instantiated
    directly, and are intended to be created using
    :func:`graphlab.deploy.job.create` API.

    Parameters
    ----------
    name : str
        Name of this Job, must be unique.

    tasks : list[string] | list[:py:class:`~graphlab.deploy.Task`]
        Collection of Tasks to be executed.

    environment : Environment, optional
        Environment used for this execution. See
        :py:class:`~graphlab.deploy.environment.LocalAsync` for an example
        environment.
    
    """

    def __init__(self, name, tasks=[], environment=None, _sequence=None):
        """__init__(name, tasks=[], environment=None)
        Construct a Job, with a unique name. Job objects should not be
        instantiated directly, and are intended to be created using
        :func:`graphlab.deploy.job.create` API.

        """
        if not isinstance(name, str):
            raise TypeError("Name is required to be a string.")

        self.name = name
        self.environment = environment
        self.tasks = tasks
        self._artifacts = tasks
        self.metrics = {}
        self._task_status = {}
        self._status = None
        self._modified_since_last_saved = None
        self._typename = Job.__name__

        self._session = _gl.deploy._default_session

        if _sequence is not None:
            self._sequence = _sequence
        else:
            # Break apart the artifacts into steps (sets of tasks which can be executed in parallel)
            self._sequence = []
            for cur_artifact in tasks:
                if isinstance(cur_artifact, _artifact.Task):
                    self._sequence.append([cur_artifact])
                elif isinstance(cur_artifact, _artifact.Pipeline):
                    self._sequence += cur_artifact.steps
                elif isinstance(cur_artifact, list):
                    self._sequence.append(cur_artifact)
                else:
                    raise Exception("Jobs can only run tasks, received: %s" % tasks)

    def get_status(self):
        """
        Returns status information about execution.

        Returns
        -------
        String representation of status. One of the following strings:
        'Pending', 'Running', 'Completed', 'Failed', 'Canceled', 'Unknown'
        """
        return self._status

    def get_metrics(self):
        """
        Returns metrics information about execution

        Returns
        -------
        Dictionary representation of metrics.
        """
        return self.metrics

    def _get_start_time(self):
        """
        Returns the start time of this execution

        Returns
        -------
        String representation of the start time
        """
        if hasattr(self, "_start_time"):
            return self._start_time
        return None

    def _get_end_time(self):
        """
        Returns the end time of this execution

        Returns
        -------
        String representation of the start time
        """
        if hasattr(self, "_end_time"):
            return self._end_time
        return None
   
    def _get_task_status(self):
        """
        Returns the status of the tasks in the job

        Returns
        -------
        A dictionary representation of the tasks status
        """
        if hasattr(self, "_task_status"):
            return self._task_status
        return None
    
    def _get_log_file_path(self):
        """
        Returns the path to the execution log file of the job

        Returns
        -------
        A string representation of execution log file path
        """
        if hasattr(self, 'log_file_path'):
            return self.log_file_path
        return None

    def cancel(self):
        """
        Cancels this execution.

        Returns
        -------
        True if successfully canceled, False otherwise.
        """
        pass

    def get_required_packages(self):
        """
        Retrieve the set of required packages for this job to execute.
        """
        packages = set()
        for artifact in self._artifacts:
            if isinstance(artifact, list):
                for task in artifact:
                    packages.update(task.get_required_packages())
            else:
                packages.update(artifact.get_required_packages())
        return packages

    @staticmethod
    def _deserialize(file_path):
        """
        Takes a path to a serialized job file. Returns the deserialized job object.
        IMPORTANT: any changes made to this function must also be made to Java deserializer.
        """
        zf = _zipfile.ZipFile(file_path)
        tmp_dir = _tempfile.mkdtemp()
        zf.extractall(tmp_dir)

        step_dir = _os.path.join(tmp_dir, 'steps')

        step_ids = [int(i) for i in _os.listdir(step_dir)]   # Save as ints so we can properly sort them.

        sequence = []
        for step_idx in sorted(step_ids):
            cur_step_dir = _os.path.join(step_dir, str(step_idx))
            sequence.append([])

            task_ids = [int(i) for i in _os.listdir(cur_step_dir)]
            for task_idx in sorted(task_ids):
                cur_task_file_path = _os.path.join(cur_step_dir, str(task_idx))

                cur_task_file = open(cur_task_file_path, 'r')
                cur_task = _pickle.load(cur_task_file)
                cur_task_file.close()

                sequence[-1].append(cur_task)

        # TODO: this is broken, deserialized job doesn't have a name, uses filepath instead
        return Job(file_path, _sequence=sequence)

    def _serialize(self, file_path):
        """
        Serializes the Job to the provided file_path.

        Special-case sequential job definitions (jobs where all the tasks are to be sequentially executed) and
        there are no parallel executions.

        In those cases, instead of serializing the tasks into the appropriate folder, just pickle the entire job
        and write it that instead.

        This way when executing if a Job is found run_job will be called, vs run_task.
        """
        aggregate_file = _zipfile.ZipFile(file_path, 'w')
        base_path = _tempfile.mkdtemp()

        # TODO: serialize job attributes (like .name) or else won't be in deserialized object

        # identify if only one task per step, and special-case that
        if all([len(step) == 1 for step in self._sequence]):
            # one task per step, so handle with serial execution
            __LOGGER__.debug("Special casing sequential execution")
            step_idx = 0
            job_idx = 0
            step_dir = _os.path.join(base_path, str(step_idx))
            _os.mkdir(step_dir)

            job_file_path = _os.path.join(step_dir, str(job_idx))
            with open(job_file_path, 'w') as job_file:
                _cloudpickle.dump(self, job_file)

            relative_path = _os.path.join('steps', str(step_idx), str(job_idx))
            aggregate_file.write(job_file_path, relative_path)

        else:

            for step_idx, cur_step in enumerate(self._sequence):
                step_dir = _os.path.join(base_path, str(step_idx))
                _os.mkdir(step_dir)

                for task_idx, cur_task in enumerate(cur_step):
                    task_file_path = _os.path.join(step_dir, str(task_idx))
                    with open(task_file_path, 'w') as task_file:
                        _cloudpickle.dump(cur_task, task_file)

                    relative_path = _os.path.join('steps', str(step_idx), str(task_idx))
                    aggregate_file.write(task_file_path, relative_path)

        aggregate_file.close()

        # delete the tempdir created
        _shutil.rmtree(base_path)

    def save(self):
        """Saves this Job to disk, so it can retrieved later."""
        self._session.save(self, typename=self._typename)
        __LOGGER__.info("Job saved successfully.")

    @_canvas.inspect.find_vars
    def show(self):
        """
        show()
        Visualize the Job with GraphLab Canvas. This function starts Canvas
        if it is not already running.
        
        Returns
        -------
        view: graphlab.canvas.view.View
            An object representing the GraphLab Canvas view

        See Also
        --------
        canvas
        """
        filename = self._session._get_filename_from_name(self.name)
        _canvas.get_target().state.set_selected_variable(('Jobs', filename))
        return _canvas.show()

    @staticmethod
    def _get_task_names(tasks):
        names = []
        for task in tasks:
            if isinstance(task, list):
                names += Job._get_task_names(task)
            elif isinstance(task, tuple):
                names.append(task[0].name)
            elif hasattr(task, 'name'):
                names.append(task.name)
            else:
                __LOGGER__.warning("Unable to get task name for '%s' task. Skipping")
        return names

    def _get_str(self):
        output = _cStringIO.StringIO()
        pp = _pprint.PrettyPrinter(indent=2, stream=output)

        header = "Job: %s\n\n" % self.name
        tasks = "Tasks: %s\n\n" % Job._get_task_names(self.tasks)
        environment = "Environment: %s\n" % self.environment

        output.write(header)
        output.write(tasks)
        if hasattr(self, 'metrics'):
            output.write("Metrics: \n\n")
            pp.pprint(self.metrics)
            output.write("\n\n")
        output.write(environment)
        output.write('\n')
        return output.getvalue()

    def __getstate__(self):
        odict = dict.copy(self.__dict__)
        if '_session' in odict:
            del odict['_session']
        return odict

    def __str__(self):
        return self._get_str()

    def __repr__(self):
        return self.__str__()


class LocalAsynchronousJob(Job):
    def __init__(self, proc, name, log_file_path, tasks=[], environment=None):
        super(LocalAsynchronousJob, self).__init__(name, tasks=tasks, environment=environment)
        self._pid = proc.pid
        self.log_file_path = log_file_path
        self.exec_dir = _os.path.dirname(log_file_path)
        self._terminated = False
        self._finalized = False

        # remove the metrics attribute so we add it only once the job is completed
        del self.metrics

    def __str__(self):
        status = self.get_status()

        if status == 'Completed' and not hasattr(self, 'metrics'):
            self._get_metrics()

        result = self._get_str()
        result += "Status: %s\n\n" % status
        result += "Execution Information:\n"
        if status not in ['Completed', 'Failed', 'Canceled']:
            result += "  Process pid: %d\n" % self._pid
        result += "  Execution Directory: %s\n" % self.exec_dir
        result += "  Log file: %s\n" % self.log_file_path
        return result

    def cancel(self):
        if _psutil.pid_exists(self._pid):
            _psutil.Process(pid=self._pid).kill()
            self._terminated = True
            self._status = 'Canceled'
            self.__finalize_save()

    def get_status(self):
        if hasattr(self, '_status') and self._status is not None:
            return self._status

        if(self._terminated):
            return self._status

        # if pid still exists then return 'Running'
        # else if check process exit code in exec dir
        if _psutil.pid_exists(self._pid):
            proc = _psutil.Process(pid=self._pid)
            if _isfunction(proc.status) or _ismethod(proc.status):
                status = str(proc.status()).capitalize()
            else:
                status = str(proc.status).capitalize()
            if status in ['Zombie']:
                return self._get_status()
            elif status in ['Sleeping', 'Disk-sleep']:
                return 'Running'
            else:
                return status
        else:
            return self._get_status()
    
    def _get_start_time(self):
        if '__starttime__' not in self._task_status:
            self._get_task_status_from_file()
        if '__starttime__' in self._task_status:
            return self._task_status["__starttime__"]
        return None

    def _get_end_time(self):
        if '__endtime__' in self._task_status:
            return self._task_status["__endtime__"]
        return None
    
    def get_metrics(self):
        # return metrics only when the job is finished and metrics attr is present
        if hasattr(self, 'metrics'):
            return getattr(self, 'metrics')
        return {}

    def _get_task_status_from_file(self):
        task_list = {}
        try:
            with open(_os.path.join(self.exec_dir, 'status'), 'r') as f:
                for line in f:
                    line_split = line.split(":");
                    if (len(line_split) == 2 or len(line_split) == 4):
                        task_list[line_split[0]] = ":".join(line_split[1:]).rstrip('\n')
                    else:
                        if '__starttime__' in task_list:
                            task_list["__endtime__"] = line.rstrip('\n')
                        else:
                            task_list["__starttime__"] = line.rstrip('\n')
        except IOError:
            pass

        self._task_status = task_list
        return task_list

    def _get_task_status(self):
        # pull from task status file until the job finishes
        if self._finalized is False:
            self._get_task_status_from_file()
        return self._task_status

    def _get_metrics(self):
        """ Go to executed job and pull the metrics out
        """
        try:
            with open(_os.path.join(self.exec_dir, 'metrics'), 'r') as f:
                metrics = f.read()
        except IOError:
            self.metrics = {}
            return

        import ast
        self.metrics = ast.literal_eval(metrics)

    def _get_status(self):
        # get exit code from dir
        try:
            with(open(_os.path.join(self.exec_dir, 'exitcode'), 'r')) as f:
                code = f.read()
        except IOError:
            self._status = 'Unknown' # if we cannot query exitcode, something is wrong
            self.__finalize_save()
            return self._status

        if code == '0':
            self._status = 'Completed'
            self.__finalize_save()
            return self._status
        else:
            self._status = 'Failed'
            self.__finalize_save()
            return self._status

    def __finalize_save(self):
        # when the job finishes, query the metrics and task status one last time, and save
        self._finalized = True
        self._get_metrics()
        self._get_task_status_from_file()
        self.save()

class Ec2Job(Job):
    def __init__(self, name, tasks, environment, ec2_dns_name, ec2_instance_id, s3_bucket,
                 s3_state_key, s3_log_dir, s3_status_key_prefix):
        super(Ec2Job, self).__init__(name, tasks=tasks, environment=environment)

        self.ec2_dns_name = ec2_dns_name
        self.ec2_instance_id = ec2_instance_id
        self.s3_bucket = s3_bucket
        self.s3_state_key = s3_state_key
        self.s3_log_dir = s3_log_dir
        self.s3_status_key_prefix = s3_status_key_prefix
        self.__s3_log_url_expire = None
        self.__s3_log_url = None
        self._start_time = None
        self._end_time = None
        self._final_status = None


    def flush_logs(self):
        '''
        Flushes all logs associated with this job.
        '''
        status = self.get_status()
        if status in ('Completed', 'Failed', 'Canceled'):
            logging.info("The job is %s. Logs have already been flushed." % status)
            return

        url = "http://%s:9004/flush" % self.ec2_dns_name
        try:
            _requests.post(url)
        except:
            logging.error("Unable to flush logs. Please try again in a few minutes")


    def get_status(self):
        '''
        Returns status information about the job.

        Returns
        -------
        String representation of status. One of the following strings:
        'Pending', 'Running', 'Completed', 'Failed', 'Canceled', 'Unknown'
        '''
        # First check if we've already know we've reached a final state/status.
        if self._final_status:
            return self._final_status

        # Check s3 for the status key.
        credentials = {}   # If this is empty, boto will look in environment variables.
        if(hasattr(self.environment, 'aws_access_key') and hasattr(self.environment, 'aws_secret_key')):
            credentials['aws_access_key_id'] = self.environment.aws_access_key
            credentials['aws_secret_access_key'] = self.environment.aws_secret_key
        s3_conn = _connect_s3(**credentials)
        bucket = s3_conn.get_bucket(self.s3_bucket, validate=False)
        k = bucket.get_key(self.s3_state_key)

        if not k:
            self._get_task_status_from_s3()
            self._final_status = 'Unknown'
            self.save()
            return self._final_status

        status_from_s3 = k.read()
        if status_from_s3 in ('Failed', 'Completed'):
            self._get_task_status_from_s3()
            self._final_status = status_from_s3
            self.save()
            return self._final_status

        return status_from_s3

    def get_task_status_logs(self):
        '''
        Returns detailed information about task status. This information can also
        be found by calling .show() on your job.
        
        Returns
        -------
        A dict with keys for each task that is either completed, running or failed.

        See Also
        --------
        show

        Examples
        --------
        >>> my_job.get_task_status_logs()
        { 'my_task-0': 'Completed',
          'my_task-0_runtime': '1.95039892197',
          'my_task-0_starttime': '2014-12-18 18:52:40.811694',
          'my_task-1': 'Failed',
          'my_task-1_starttime': '2014-12-18 19:05:29.037241',
          'my_task-2': 'Running',
          'my_task-2_starttime': '2014-12-18 18:52:42.763024'
        }
        '''
        return self._get_task_status_from_s3()
        

    def _get_start_time(self):
        if self._start_time:
            return str(self._start_time.isoformat())

        # get starttime from job status
        self._get_task_status_from_s3()
        cur_starttime = _datetime.datetime.now()
        if self._task_status and len(self._task_status) > 0:
            for key in self._task_status:
                if key.endswith("starttime"):
                    starttime_str = self._task_status[key]
                    starttime = _datetime.datetime.strptime(starttime_str, "%Y-%m-%d %H:%M:%S.%f")
                    if cur_starttime > starttime:
                        cur_starttime = starttime
            self._start_time = cur_starttime
            return str(self._start_time.isoformat())
        else:
            return None

    def _get_task_status_from_s3(self):
        task_list = {}
        
        # Check s3 for the status key.
        credentials = {}   # If this is empty, boto will look in environment variables.
        if(hasattr(self.environment, 'aws_access_key') and hasattr(self.environment, 'aws_secret_key')):
            credentials['aws_access_key_id'] = self.environment.aws_access_key
            credentials['aws_secret_access_key'] = self.environment.aws_secret_key
        s3_conn = _connect_s3(**credentials)
        bucket = s3_conn.get_bucket(self.s3_bucket, validate=False)
        if hasattr(self, 's3_status_key'): # Is this a job from prior to 1.2?
            self.s3_status_key_prefix = self.s3_status_key
        status_keys = bucket.get_all_keys(prefix = self.s3_status_key_prefix)

        for k in status_keys:
            try:
                jobstatus = k.read()
                jobstatus_blob = jobstatus.split('\n')
                for line in jobstatus_blob:
                    line_split = line.split(":");
                    # spliting the line read, to key value pairs
                    if (len(line_split) > 1):
                        task_list[line_split[0]] = ":".join(line_split[1:]).rstrip('\n')
            except IOError:
                __LOGGER__.warning("Unable to get job status files from %s with prefix %s" % (self.s3_bucket, self.s3_status_key_prefix))

        self._task_status = task_list
        return task_list

    def _get_task_status(self):
        # pull from task status file until the job finishes
        if not self._final_status or not self._task_status:
            self._get_task_status_from_s3()
        return self._task_status

    def _get_end_time(self):
        if self._end_time:
            return str(self._end_time.isoformat())

        # return None if job has not completed
        if not self._final_status:
            return None
            
        # get endtime from job status
        self._get_task_status_from_s3()
        if self._task_status and len(self._task_status) > 0:
            self._get_start_time()
            starttime = self._start_time
            total_time_sec = 0
            for key in self._task_status:
                if key.endswith("runtime"):
                    duration = int(float(self._task_status[key]))
                    total_time_sec += duration
            self._end_time = starttime + _datetime.timedelta(0, total_time_sec)
            return str(self._end_time.isoformat())
        else:
            return None

    def _get_log_file_path(self):
        # return generated url if the url has not expire
        if self.__s3_log_url_expire and self.__s3_log_url:
            delta = _datetime.datetime.now() - self.__s3_log_url_expire
            if delta.total_seconds() < 300:
                return self.__s3_log_url

        self.__s3_log_url_expire = _datetime.datetime.now()

        # Check s3 for the status key.
        credentials = {}   # If this is empty, boto will look in environment variables.
        if(hasattr(self.environment, 'aws_access_key') and hasattr(self.environment, 'aws_secret_key')):
            credentials['aws_access_key_id'] = self.environment.aws_access_key
            credentials['aws_secret_access_key'] = self.environment.aws_secret_key
        s3_conn = _connect_s3(**credentials)
        bucket = s3_conn.get_bucket(self.s3_bucket, validate=False)
        if hasattr(self, 's3_log_key'): # Is this a job from prior to 1.2?
            keys = [bucket.get_key(self.s3_log_key)]
        else:
            keys = bucket.get_all_keys(prefix = self.s3_log_dir + 'i-')

        if not keys:
            return "Log file not available on S3."

        results = []
        for k in keys:
            url = k.generate_url(expires_in=300)
            results.append((url, k.size))

        self.__s3_log_url = results
        return results


    def cancel(self):
        """
        Cancels this job. This will terminate all associated EC2 instances. Job status
        will be marked as 'Canceled'.
        """
        cancel_url = "http://%s:9004/cancel" % self.ec2_dns_name
        try:
            _requests.post(cancel_url)
        except:
            logging.error("Unable to cancel EC2 instance(s). Please check AWS Console to make sure "
                          + "any EC2 instances launched have been terminated.")
        else:
            self._final_status = 'Canceled'

    def __str__(self):
        status = self.get_status()
        result = self._get_str()
        result += "Status: %s\n\n" % status
        result += "Execution Information:\n"
        result += "  Commander Public DNS: %s\n" % self.ec2_dns_name
        result += "  Commander Instance ID: %s\n" % self.ec2_instance_id
        if hasattr(self, 's3_log_key'): # Is this a job from prior to 1.2?
            result += "  Log file: bucket = %s, key = %s\n" % (self.s3_bucket, self.s3_log_key)
        else:
            result += "  Log files: bucket = %s, directory = %s\n" % (self.s3_bucket, self.s3_log_dir)
        return result


class HadoopJob(Job):
    # need hadoop and yarn on the command line
    # need if source = none need virtualenv-2.7 on the commandline and python

    _json_flag = '$$JSONTAG##'
    def __init__(self, name, tasks, environment, application_id ):
        super(HadoopJob, self).__init__(name, tasks=tasks, environment=environment)
        self.app_id = application_id
        self._app_state = None
        self._yarn_AM_state = None
        self._start_time = None
        self._end_time = None
        self._report_time = None

    def _hadoop_applicaton_check(self):
        """
        returns None if not found
        :rtype: dict
        """
        # if state != finished query hadoop
        if self.environment.get_config_dir() is not None:
            hadoop_base = "hadoop --config %s " % self.environment.get_config_dir()
        else:
            hadoop_base = "hadoop "
        hadoop_jar = _os.path.join(_os.path.dirname(_os.path.realpath(__file__)), "graphlabHadoopYarn-1.0.jar")
        hadoop_cmd = "%s jar %s Client -debug -jar %s -checkAppId %s -json" % \
                     (hadoop_base, hadoop_jar, hadoop_jar, self.app_id)
        __LOGGER__.info("Retrieving current job status from Hadoop cluster using command= \n %s" % hadoop_cmd)
        proc = _subprocess.Popen(hadoop_cmd, shell=True,
                                 stderr=_subprocess.STDOUT,
                                 stdout=_subprocess.PIPE)
        app_report = None
        for line in proc.stdout:
            __LOGGER__.debug(line.strip('\n'))
            if HadoopJob._json_flag in line:
                clean = line.split(self._json_flag)
                app_report=json.loads(clean[1])
                self._yarn_AM_state = app_report['AppState']
                self._app_state = app_report['DistributedFinalState']

        return app_report

    def get_status(self):
        if hasattr(self, '_status') and self._status is not None:
            return self._status

        if not self.app_id:
            return 'Unknown'

        report = self._hadoop_applicaton_check()

        #TODO should we output something here to say Unknown means it's not findable on the cluster?
        if not report:
            return 'Unknown'

        state = report['AppState']
        self._end_time = report['FinishTime']
        #Gross
        if state == 'FAILED' or state == 'KILLED':
            self._status = 'Failed'
            return self._status

        elif state == 'FINISHED':

            if report['DistributedFinalState'] == 'SUCCEEDED':
                self._status = 'Completed'

            elif report['DistributedFinalState'] == 'KILLED':
                self._status = 'Canceled'

            else :
                self._status = 'Failed'

            return self._status


        elif state == 'ACCEPTED' or 'NEW' in state or state == 'SUBMITTED':
            return 'Pending'


        elif state == 'RUNNING':
            return 'Running'

        else:
            return 'Unknown'


    def get_yarn_application_state(self):
        """
        Enumeration of ApplicationMaster states
        ACCEPTED, FAILED, FINISHED, KILLED, NEW, NEW_SAVING, RUNNING, SUBMITTED
        :return: str
        """
        report = self._hadoop_applicaton_check()
        return report['AppState']

    def get_application_state(self):
        """
        final states of an application
        FAILED, KILLED, SUCCEEDED, UNDEFINED (app not yet finished)
        :return: str
        """
        report = self._hadoop_applicaton_check()
        return report['DistributedFinalState']

    def cancel(self):
        if self._status == 'Canceled':
            return 'Application already canceled'

        yarn_cmd = "yarn "
        if self.environment.get_config_dir():
            yarn_cmd += " --config %s " % self.environment.get_config_dir()

        yarn_cmd += " application -kill %s " % self.app_id
        print yarn_cmd
        proc = _subprocess.Popen(yarn_cmd, shell=True,
                                  stderr=_subprocess.STDOUT,
                                  stdout=_subprocess.PIPE )
        __LOGGER__.info("Attempting to kill job")

        for line in proc.stdout:
            print line.strip('\n')
            if 'Killing application' in line:
                __LOGGER__.info("Success killing application")
                self._status = 'Cancelled'
            elif 'ApplicationNotFoundException' in line:
                __LOGGER__.info('Application not found to kill')
                self._status= 'Unknown'
            elif 'has already finished' in line:
                __LOGGER__.info("Application has already finished")


    def __str__(self):
        status = self.get_status()
        result = self._get_str()
        result += "Application Id: %s\n" % self.app_id
        result += "Status: %s\n\n" % status
        result += "Execution Information:\n"
        result += "  Yarn Application State: %s\n" % self._yarn_AM_state
        result += "  Application Status: %s\n" % self._app_state
        result += "  Start Time: %s\n" % self._start_time
        result += "  Finish Time: %s\n" % self._end_time
        if self.environment.get_config_dir():
            result += "Command to view yarn logs: \n "\
                      "yarn --config %s logs -applicationId %s" % (self.environment.get_config_dir(), self.app_id)
        else:
            result += "Command to view yarn logs: \n  " \
                      "yarn logs -applicationId %s" % self.app_id
        return result

