#! /usr/bin/env python
# coding: utf-8
"""
 :copyright: (c) 2012 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import
import os.path

try:
    import json
except ImportError:
    import simplejson as json


class Repository(object):
    """ The Repository is responsible for finding the correct templates
    """

    def __init__(self):
        self._paths = []

    def add_path(self, path):
        """ Adds a path, in which templates are searched.

        Later added paths have a lower priority than earlier added paths.

        path
            the path
        """
        self._paths.append(path)

    def add_buildins(self):
        """ Adds the path for the builtin templates
        """
        self.add_path(os.path.join(os.path.dirname(__file__), 'templates'))

    def get_template(self, templatename):
        """ Finds a template in the repository

        raises `UsageError` with user-presentable Message, when no template
        could be found.

        templatename
            name of the Template. A template's name is its directory name

        returns
            a `Template` object
        """
        for path in self._paths:
            candidate = os.path.join(path, templatename)
            if os.path.exists(candidate):
                return Template(candidate)
        raise UsageError(u"no template '%s' found." % templatename)

    def get_template_names(self):
        """ get the names of all templates in the repository

        returns
            an iterable over the names
        """
        for path in self._paths:
            for filename in os.listdir(path):
                if os.path.isdir(os.path.join(path, filename)):
                    yield os.path.basename(filename)


class Template(object):

    def __init__(self, path):
        """ Constructor.

        path
            Path to the template's directory. In the directory, a configuration
            file named 'greygoo.json' must be present.
        """
        self._path = path
        with open(os.path.join(path, 'graygoo.json')) as fp:
            self._config = json.load(fp)

    def render(self, path, args):
        """ Renders the template to the given path

        path
            where to the template to
        args
            dictionary with variables used for the rendering
        """
        for tpl in self._config['render']:
            target = os.path.join(path, tpl['target'] % args)
            self._render_source_file(target, tpl['source'], args)

    def _render_source_file(self, target, sourcefile, args):
        """ Renders the targetfile from sourcefile and args

        target
            the filename of the target file
        sourcefile
            the filename of the source file
        args
            the arguments for rendering
        """
        # already existing files are not touched.
        if os.path.exists(target):
            self.log(u'%s already exists, skipping it' % target)
            return

        self._enforce_directory(os.path.dirname(target))
        self._write_file(target, self._get_source_file(sourcefile, args))

    def _enforce_directory(self, directory):
        """ Creates the directory, if it does not exist

        directory
            filename of the directory
        """
        if directory != "" and not os.path.exists(directory):
            self.log(u"creating directory '%s'" % directory)
            os.makedirs(directory)

    def _write_file(self, filename, content):
        """ Writes a file

        filename
            the file to write to
        content
            the new contents of the file
        """
        self.log(u"generating '%s'" % filename)
        with open(filename, 'wb') as fp:
            fp.write(content)

    def _get_source_file(self, filename, args):
        with open(os.path.join(self._path, filename), 'rb') as fp:
            return fp.read() % args

    def log(self, text):
        print(text)


class UsageError(Exception):
    pass
