#! /usr/bin/env python
# coding: utf-8
"""
 :copyright: (c) 2012 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import, unicode_literals

import os.path
from optparse import OptionParser

from .templating import Repository, UsageError


option_parser = OptionParser(usage='\t%prog [TEMPLATENAME] [key=value] ...\n'
                                   '\t%prog list\n'
                                   '\t%prog info [TEMPLATENAME]')


class Cli(object):
    """ The Command Line Interface.
    """

    def __init__(self):
        """ Constructor.
        """
        self.repo = Repository()
        self.repo.add_path(os.path.expanduser('~/.graygoo'))
        self.repo.add_buildins()

    def run(self, args=None):
        """ Runs the Command Line Interface

        args:
          The command line arguments (sys.argv[1:])
        """
        options, args = option_parser.parse_args(args)
        if not args:
            option_parser.error('please provide a command or a template name')
        try:
            # if handle_xxx is found, dispatch that way...
            func = getattr(self, 'handle_%s' % args[0], None)
            if func:
                func(options, args[1:])
            # ...else assume it was a template name
            else:
                self.render_template(options, args)
        except UsageError as exc:
            self.error(unicode(exc))

    def error(self, message):
        """ Indicates an error.

        message
            the error message
        """
        option_parser.error(message)

    def output(self, text):
        """ Writes a user output to the screen.

        text
            the text to show to the user
        """
        print(text)

    def render_template(self, options, args):
        """ Handles rendering of a template.

        CLI-syntax:
        ``graygoo [templatename] [name1]=[value1] [name2]=[value2] ...``
        """
        template_args = [pair.split('=', 1) for pair in args[1:]]

        if any(len(pair) != 2 for pair in template_args):
            self.error('please provide all template arguments in '
                                                          'the form key=value')
        # render the template
        template = self.repo.get_template(args[0])
        template.render('', dict(template_args))

    def handle_list(self, options, args):
        """ Handles the 'list'-command.

        CLI-syntax: ``graygoo list``
        """
        for name in sorted(self.repo.get_template_names()):
            self.output(name)

    def handle_info(self, options, args):
        """ Handles the 'info'-command.

        CLI-syntax: ``graygoo info [templatename]``
        """
        if not len(args) >= 1:
            self.error('please provide a template name')

        template = self.repo.get_template(args[0])
        self.output('%s (%s)\n' % (template.name, args[0]))
        self.output(template.description + '\n')

        if template.argument_descriptions:
            self.output('Arguments:\n')
            for name, desc in template.argument_descriptions.items():
                self.output(name)
                self.output("  %s\n" % desc)


def main():
    """ Entry Point for setup.py/setuptools """
    cli = Cli()
    cli.run()
