# coding: utf-8
"""
 :copyright: (c) 2012 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import, unicode_literals
import os.path

try:
    import json
except ImportError:
    import simplejson as json


class Repository(object):
    """ The Repository is responsible for finding the correct templates
    """

    def __init__(self):
        self._paths = []

    def add_path(self, path):
        """ Adds a path, in which templates are searched.

        Later added paths have a lower priority than earlier added paths.

        path
            the path
        """
        self._paths.append(path)

    def add_buildins(self):
        """ Adds the path for the builtin templates
        """
        self.add_path(os.path.join(os.path.dirname(__file__), 'templates'))

    def get_template(self, templatename):
        """ Finds a template in the repository

        raises `UsageError` with user-presentable Message, when no template
        could be found.

        templatename
            name of the Template. A template's name is its directory name

        returns
            a `Template` object
        """
        for path in self._paths:
            candidate = os.path.join(path, templatename)
            if os.path.exists(candidate):
                return Template(candidate)
        raise UsageError("no template '%s' found." % templatename)

    def get_template_names(self):
        """ get the names of all templates in the repository

        returns
            an iterable over the names
        """
        for path in self._paths:
            if os.path.exists(path):
                for filename in os.listdir(path):
                    if os.path.isdir(os.path.join(path, filename)):
                        yield os.path.basename(filename)


class Template(object):
    """ Representation of a Template

    Templates are directories with a configuration file (graygoo.json) and
    template files in it.

    The configuration file must provide theses keys:

    name
      a human readable name for the template, e.g. 'Gray Goo Welcome Page'.
      (String)
    description
      a short description of what this template provides. (String)
    arguments
      a dictionary of argument names and their descriptions, e.g.
      {'name': 'Who should be adressed in the greeting?'}
    render
      a list of rendering informations. [TODO: be more elaborate here]

    """

    def __init__(self, path):
        """ Constructor.

        path
            Path to the template's directory. In the directory, a configuration
            file named 'greygoo.json' must be present.
        """
        self.path = path
        with open(os.path.join(path, 'graygoo.json')) as configfile:
            self._config = json.load(configfile)

    @property
    def name(self):
        """ The human readable name of this Template """
        return self._config['name']

    @property
    def description(self):
        """ The description of what this template provides """
        return self._config['description']

    @property
    def argument_descriptions(self):
        """ a dictionary {argument_name: argument_description, ...} """
        return self._config['arguments']

    def render(self, path, args):
        """ Renders the template to the given path

        path
            where to the template to
        args
            dictionary with variables used for the rendering
        """
        missing = [(name, desc)
                   for name, desc in self._config['arguments'].items()
                   if name not in args]

        if missing:
            raise UsageError('please provide these arguments:\n' +
                  '\n'.join('%s: %s' % (name, desc) for name, desc in missing))

        for tpl in self._config['render']:
            target = os.path.join(path, tpl['target'] % args)
            self._render_source_file(target, tpl['source'], args)

    def _render_source_file(self, target, sourcefile, args):
        """ Renders the targetfile from sourcefile and args

        target
            the filename of the target file
        sourcefile
            the filename of the source file
        args
            the arguments for rendering
        """
        # already existing files are not touched.
        if os.path.exists(target):
            self.log('%s already exists, skipping it' % target)
            return

        self._enforce_directory(os.path.dirname(target))
        self._write_file(target, self._get_source_file(sourcefile) % args)

    def _enforce_directory(self, directory):
        """ Creates the directory, if it does not exist

        directory
            filename of the directory
        """
        if directory != "" and not os.path.exists(directory):
            self.log("creating directory '%s'" % directory)
            os.makedirs(directory)

    def _write_file(self, filename, content):
        """ Writes a file

        filename
            the file to write to
        content
            the new contents of the file
        """
        self.log("generating '%s'" % filename)
        with open(filename, 'wb') as outfile:
            outfile.write(content)

    def _get_source_file(self, filename):
        """ returns the contents of `filename` """
        with open(os.path.join(self.path, filename), 'rb') as sourcefile:
            return sourcefile.read()

    def log(self, text):
        """ Logs an information.

        By default, it is printed to stdout
        """
        print(text)


class UsageError(Exception):
    """ TODO: split up in specific errors
    """
    pass
