"""
Simple V1 model

Andrew P. Davison, UNIC, CNRS, November 2007
"""
from NeuroTools.facets import fkbtools
from NeuroTools import parameters
from NeuroTools.stgen import StGen
from NeuroTools.signals import SpikeTrain, SpikeList
import shutil
import os
import atexit
import numpy
import matplotlib
from numpy import sin, cos, exp, pi, sqrt
import sys
sys.path.append('/home/thierry/Benchmarks_Project/svn-trunk/examples')
from myutilities import *
from Benchmarks import plotting
import Benchmarks.analysis
from Component.component import *
from Component.custom_component import TuningCurves, DebugTuningCurves, FitTriangles, Histogram, DebugHistogram, PeakAccuracy, StoreResult
MAX_COL = 2
# Global variables (to be set later)
frame_duration = None
sim_duration = None
images = None 
recordedvariable = None
tmpdir = None
cells = []
orientations = None
stimulus_id = ''
cells_cache = None
figures = {}

# === Define interface for visionbenchmark.py ==================================

def build(n_cells, gabor_parameters, non_linearity_parameters):
    """
    Create a list of `GaborCell`s with identical parameters except for
    orientation.
    """
    logging.info("Creating %d GaborCells" % n_cells)
    logging.debug("  gabor_parameters = %s" % gabor_parameters)
    logging.debug("  non_linearity_parameters = %s" % non_linearity_parameters)
    cells = []
    for i in range(n_cells):
        cells.append(GaborCell(gabor_parameters, non_linearity_parameters) )
    return cells

def record(measureable, brain_region_dict):
    global recordedvariable, cells
    recordedvariable = measureable
    layer = brain_region_dict.keys()[0]
    assert layer == 'not-specified', "Recording from specific layers not yet supported."
    cell_type = brain_region_dict[layer].keys()[0]
    assert cell_type == 'not-specified', "Recording from specific cell types not yet supported."
    n_cells = brain_region_dict[layer][cell_type]
    assert isinstance(n_cells, int), "n_cells must be an integer, not %s" % type(n_cells)
    logging.info("Recording %s" % recordedvariable)
    # Build the model - doing it here allows the script to be run standalone or via visionbenchmark.py
    cells = build(n_cells, P['gabor'], P['non_linearity'])
    if P['debug']:
        ncols = int(sqrt(n_cells/0.6)); nrows = n_cells/ncols
        if n_cells%ncols != 0: nrows += 1
        figures['orientations'] = plotting.SimpleMultiplot(nrows, ncols, xlabel="Orientation (deg)", ylabel="Number of spikes")
        figures['responses'] = plotting.SimpleMultiplot(nrows, ncols, xlabel="Time (ms)", ylabel="Response")
        figures['gabors'] = plotting.SimpleMultiplot(nrows, ncols)
        Benchmarks.analysis.debug_fig = figures['orientations']

def set_stimulus(stimulus_url,scale_factor,max_luminance,background_luminance):
    global frame_duration, images, orientations, tmpdir, stimulus_id
    logging.info("Setting stimulus from %s" % stimulus_url)
    logging.info("Scale factor: %g pixels/degree" % scale_factor)
    tmpdir, image_sequence, frame_duration = fkbtools.getStimulus(stimulus_url)
    logging.info("Frame duration = %g ms" % frame_duration)
    images, frame_variables = load_frames(image_sequence)
    stimulus_id = stimulus_url
    return frame_variables

def present_stimulus(cells, images, orientations, frame_duration):
    """
    Present the stimulus and generate spiketrains.
    
    If this is the first time that function has been called, the cells are
    created and the image frames presented to them to calculate the `x` and `y`
    variables. These cells are then cached.
    
    On subsequent calls to the function, `cells` and `images` are ignored and
    the cached cells used. However, a new spike train is generated by each cell.
    
    (`orientations` does not seem to be used).
    """
    global stimulus_id, cells_cache
    if cells_cache is not None:
        cells = cells_cache
        logging.info("Retrieving cached cells")
    else:
        logging.debug("  images[0:2] = %s" % images[0:2])
        logging.debug("  images[-3:-1] = %s" % images[-3:-1])
        assert os.path.isfile(images[0]), "%s is not a file or does not exist" % images[0]
        img0 = matplotlib.image.imread(images[0])
        width_pixels = img0.shape[0]
        height_pixels = img0.shape[1]
        logmsg = "Frame size: %d x %d   " % (height_pixels, width_pixels)
        logmsg += "Number of frames: %d   " % len(images)
        logmsg += "Frame duration: %g ms" % frame_duration
        logging.info(logmsg)

        for cell in cells:
            cell.setup(len(images), frame_duration)
            
        logging.info("Presenting image frames to %d cells and calculating response." % len(cells))
        for i,filename in enumerate(images):
            img = matplotlib.image.imread(filename)
            img = img[:,:,0:3].mean(axis=2) # rgb --> greyscale
            for cell in cells:
                cell.present_image(img)
        cells_cache = cells
        
    logging.info("Generating spikes for %d frames each of duration %g ms" % (len(images),frame_duration))
    time_bins = numpy.arange(0.0, len(images)*frame_duration, frame_duration)
    for cell in cells:
        cell.generate_spikes(time_bins)

def run(quiet=True):
    global cells, frame_duration, images, tmpdir, sim_duration, orientations, time_bins
    # Run the model
    if recordedvariable is None:
        raise Exception("You must call record(<variable_to_record>) before run()")
    elif images is None or frame_duration is None:
        raise Exception("You must call set_stimulus(<stimulus_file>,<scale_factor>) before run()")
    else:
        if not quiet:
            logging.info("Running V1 model...")
        
        sim_duration = frame_duration * len(images)
        present_stimulus(cells, images, orientations, frame_duration)
        time_bins = numpy.arange(0.0, len(images)*frame_duration, frame_duration)
        
    # Return the results
    if recordedvariable == "spikes":
        result = SpikeList([], [])
        for cell in cells:
            result[cell.id] = cell.spike_train
    else:
        raise "Can only record spikes"
    # Return the result
    return {'not-specified': result}

class Recorder(BaseComponent) :
    """Component that records each sample"""
    def __init__(self) :
        #Inputs
        self.P = Input()
        self.measureable = Input()
        self.brain_region_dict = Input()
        self.figures = Input()
        #Outputs
        self.recordedvariable = Output()
        self.cells = Output()
        self.figures_changed = Output()
        super(Recorder, self).__init__()

    def execute(self) :
        self.figures_changed.potential = self.figures.potential.copy()
        self.recordedvariable.potential = self.measureable.potential
        layer = self.brain_region_dict.potential.keys()[0]
        assert layer == 'not-specified', "Recording from specific layers not yet supported."
        cell_type = self.brain_region_dict.potential[layer].keys()[0]
        assert cell_type == 'not-specified', "Recording from specific cell types not yet supported."
        n_cells = self.brain_region_dict.potential[layer][cell_type]
        assert isinstance(n_cells, int), "n_cells must be an integer, not %s" % type(n_cells)
        logging.info("Recording %s" % self.recordedvariable.potential)
        # Build the model - doing it here allows the script to be run standalone or via visionbenchmark.py
        self.cells.potential = build(n_cells, self.P.potential['gabor'], self.P.potential['non_linearity'])
        if self.P.potential['debug']:
            ncols = int(sqrt(n_cells/0.6)); nrows = n_cells/ncols
            if n_cells%ncols != 0: nrows += 1
            self.figures_changed.potential['orientations'] = plotting.SimpleMultiplot(nrows, ncols, xlabel="Orientation (deg)", ylabel="Number of spikes")
            self.figures_changed.potential['responses'] = plotting.SimpleMultiplot(nrows, ncols, xlabel="Time (ms)", ylabel="Response")
            self.figures_changed.potential['gabors'] = plotting.SimpleMultiplot(nrows, ncols)
            Benchmarks.analysis.debug_fig = self.figures_changed.potential['orientations']

class StimSetter(BaseComponent) :
    """Component that creates stimuli"""
    
    def __init__(self) :
        #Inputs
        self.stimulus_url = Parameter()
        self.scale_factor = Parameter()
        self.max_luminance = Parameter()
        self.background_luminance = Parameter()
        #Outputs
        self.frame_duration = Output()
        self.images = Output()
        self.tmpdir = Output()
        self.frame_variables = Output()
        self.stimulus_id = Output()
        super(StimSetter, self).__init__()
    
    def execute(self) :
        logging.info("Setting stimulus from %s" % self.stimulus_url.potential)
        logging.info("Scale factor: %g pixels/degree" % self.scale_factor.potential)
        self.tmpdir.potential, image_sequence, self.frame_duration.potential = fkbtools.getStimulus(self.stimulus_url.potential)
        logging.info("Frame duration = %g ms" % self.frame_duration.potential)
        self.images.potential, self.frame_variables.potential = load_frames(image_sequence)
        self.stimulus_id.potential = self.stimulus_url.potential

class Simulator(BaseComponent) :
    """Component that simulates"""
    
    def __init__(self) :
        #Inputs
        self.quiet = Input()
        self.repetitions = Input()
        self.recordedvariable = Input()
        self.cells = Input()
        self.frame_duration = Input()
        self.images = Input()
        self.tmpdir = Input()
        self.figures = Input()
        #Outputs
        self.spikelists = Output()
        self.cells_cache = Output()
        self.time_bins = Output()
        self.sim_duration = Output()
        self.figures_changed = Output()
        #
        super(Simulator, self).__init__()
    
    def present_stimulus(self):
        """
        Present the stimulus and generate spiketrains.
        
        If this is the first time that function has been called, the cells are
        created and the image frames presented to them to calculate the `x` and `y`
        variables. These cells are then cached.
        
        On subsequent calls to the function, `cells` and `images` are ignored and
        the cached cells used. However, a new spike train is generated by each cell.
        
        (`orientations` does not seem to be used).
        """
     
        if self.cells_cache.potential is not None:
            cells = self.cells_cache.potential
            logging.info("Retrieving cached cells")
        else:
            cells = [k for k in self.cells.potential]
            logging.debug("  images[0:2] = %s" % self.images.potential[0:2])
            logging.debug("  images[-3:-1] = %s" % self.images.potential[-3:-1])
            assert os.path.isfile(self.images.potential[0]), "%s is not a file or does not exist" % self.images.potential[0]
            img0 = matplotlib.image.imread(self.images.potential[0])
            width_pixels = img0.shape[0]
            height_pixels = img0.shape[1]
            logmsg = "Frame size: %d x %d   " % (height_pixels, width_pixels)
            logmsg += "Number of frames: %d   " % len(self.images.potential)
            logmsg += "Frame duration: %g ms" % self.frame_duration.potential
            logging.info(logmsg)
         
            for cell in cells :
                cell.setup(len(self.images.potential), self.frame_duration.potential)
            
            logging.info("Presenting image frames to %d cells and calculating response." % len(cells))
            for i,filename in enumerate(self.images.potential):
                img = matplotlib.image.imread(filename)
                img = img[:,:,0:3].mean(axis=2) # rgb --> greyscale
                for cell in cells:
                    cell.present_image(img)
            self.cells_cache.potential = cells
        
        logging.info("Generating spikes for %d frames each of duration %g ms" % (len(self.images.potential),self.frame_duration.potential))
        self.time_bins.potential = numpy.arange(0.0, len(self.images.potential)*self.frame_duration.potential, self.frame_duration.potential)
        for cell in cells:
            cell.generate_spikes(self.time_bins.potential)
    
    def run(self):
        # Run the model
        if self.recordedvariable.potential is None:
            raise Exception("You must call record(<variable_to_record>) before run()")
        elif self.images.potential is None or self.frame_duration.potential is None:
            raise Exception("You must call set_stimulus(<stimulus_file>,<scale_factor>) before run()")
        else:
            if not self.quiet.potential :
                logging.info("Running V1 model...")
            
            self.sim_duration.potential = self.frame_duration.potential * len(self.images.potential)
            self.present_stimulus()
            self.time_bins.potential = numpy.arange(0.0, len(self.images.potential)*self.frame_duration.potential, self.frame_duration.potential)
        
        # Return the results
        if self.recordedvariable.potential == "spikes":
            result = SpikeList([], [])
            for cell in self.cells_cache.potential:
                result[cell.id] = cell.spike_train
        else:
            raise "Can only record spikes"
        # Return the result
        return {'not-specified': result}
    
    def execute(self) :
        global figures
        figures = self.figures.potential
        figures['spikes'] = plotting.SimpleMultiplot(len(self.cells.potential)+2, min(MAX_COL, self.repetitions.potential))
        self.cells_cache.potential = None
        results = []
        for i in range(self.repetitions.potential):
            logging.info("Trial #%d" % i)
            spike_list = self.run().values()[0]
            logging.debug("  Spike train for cell #0: %s" % str(spike_list[0]))
            results.append(spike_list)
        self.figures_changed.potential = figures
        self.spikelists.potential = results

class SpikeGenerator(Component) :
    "Component that generates spikes"
    
    #Inputs
    P = Parameter()
    measureable = Parameter()
    brain_region_dict = Parameter()
    figures = Input()
    stimulus_url = Parameter()
    scale_factor = Parameter()
    max_luminance = Parameter()
    background_luminance = Parameter()
    quiet = Parameter()
    repetitions = Parameter()
    #Outputs
    recordedvariable = Output()
    cells = Output()
    figures_changed = Output()
    frame_duration = Output()
    images = Output()
    orientations = Output()
    tmpdir = Output()
    stimulus_id = Output()
    spikelists = Output()
    cells_cache = Output()
    time_bins = Output()
    sim_duration = Output()
    #Components
    recorder = Recorder()
    setter = StimSetter()
    simulator = Simulator()
    #Connections 
    schema = [#Inputs
                     [P, recorder.P],
                     [measureable, recorder.measureable],
                     [brain_region_dict, recorder.brain_region_dict],
                     [figures, recorder.figures],
                     [stimulus_url, setter.stimulus_url],
                     [scale_factor, setter.scale_factor],
                     [max_luminance, setter.max_luminance],
                     [background_luminance, setter.background_luminance],
                     [quiet, simulator.quiet],
                     [repetitions, simulator.repetitions],
                     #Outputs
                     [figures_changed, simulator.figures_changed],
                     [stimulus_id, setter.stimulus_id],
                     [orientations, setter.frame_variables],
                     [spikelists, simulator.spikelists],
                     [cells_cache, simulator.cells_cache],
                     [time_bins, simulator.time_bins],
                     [sim_duration, simulator.sim_duration],
                     #Interconnections
                     [recorder.recordedvariable, simulator.recordedvariable],
                     [recorder.cells, simulator.cells],
                     [recorder.figures_changed, simulator.figures],
                     [setter.frame_duration, simulator.frame_duration],
                     [setter.images, simulator.images],
                     [setter.tmpdir, simulator.tmpdir]]

def clean_up() :
    """Delete temporary files."""
    global tmpdir
    if tmpdir is not None:
        shutil.rmtree(tmpdir)
    if P['debug']:
        for name,fig in figures.items():
            fig.save("simpleV1_%s.png" % name)

atexit.register(clean_up)


# === Define model =============================================================

"""
# Get list of image files and orientations
images, orientations = load_frames(image_sequence)
# Create cells
cells = build(n_cells, gabor_parameters, non_linearity_parameters)
# Present stimuli to the cells
present_stimulus(cells, images, orientations, frame_duration)
"""

def gabor(x, y, theta, gamma, sigma, k, psi):
    xp = x*cos(theta) + y*sin(theta)
    yp = -x*sin(theta) + y*cos(theta)
    return exp(-(xp*xp + gamma*gamma*yp*yp)/(2*sigma*sigma))*cos(2*pi*k*xp + psi)

def sigmoid(x,x0,k):
    return 1/(1+exp(-k*(x-x0)))

def load_frames(image_sequence):
    """
    Read image filenames and the frame variable value(s) for each frame.
    
    `image_sequence` should be a tuple containing the filename of the sequence
    file and the directory containing the image files.
    """
    if isinstance(image_sequence,tuple): # file listing frames
        sequence_file, image_dir = image_sequence
        f = open(sequence_file, 'r')
        lines = f.readlines()
        f.close()
    else:
        raise Exception("image_sequence must be a tuple (sequence_file, image_dir)")
    ncol = len(lines[0].split())
    if ncol == 1:
        return ["%s/%s" % (image_dir, line.strip()) for line in lines], []
    elif ncol > 1:
        images = ["%s/%s" % (image_dir, line.split()[0]) for line in lines]
        frame_variables = [line.split()[1:] for line in lines]
        for i in range(len(frame_variables)):
            frame_variables[i] = ", ".join(frame_variables[i])
        return images, frame_variables        
    else:
        raise Exception("Invalid frames file")

def build_kernel(kernel_size, theta, gamma, sigma, k, psi, normalized=True):
    """
    Create a Gabor-filter kernel
      `kernel_size` is the length of the sides of the (square) kernel in pixels
      `k` represents the spatial frequency of the stripes,
      `theta` represents the orientation of the stripes,
      `psi` is the phase offset,
      `gamma` specifies the ellipticity.
    If `normalized`, the sum of the elements of the kernel is set to zero
    (by subtraction of the mean value.
    """
    offset = kernel_size/2
    f = lambda x,y: gabor(x-offset, y-offset,
                          theta=theta,
                          gamma=gamma,
                          sigma=sigma*kernel_size,
                          k=k/float(kernel_size),
                          psi=psi)
    kernel = numpy.fromfunction(f, (kernel_size, kernel_size))
    
    if normalized: # kernel should sum to zero
        sum = kernel.sum()
        kernel -= sum/kernel_size**2
    return kernel

def convolve(img, kernel):
    """
    Convolve the part of the image within the receptive field, as given by the
    kernel size, with the kernel. The receptive field is always centred within
    the image.
    """
    # all units are in pixels
    img_height = img.shape[0]
    img_width = img.shape[1]
    kernel_height = kernel.shape[0]
    kernel_width = kernel.shape[1]
    assert kernel_height <= img_height
    assert kernel_width <= img_width
    #for x in img_height, img_width, kernel_height, kernel_width:
    #    assert x%2 == 1
    margin_tb = int((img_height-kernel_height)/2.0)
    margin_lr = int((img_width-kernel_width)/2.0)
    
    cropped_img = img[margin_tb:margin_tb+kernel_height,
                      margin_lr:margin_lr+kernel_width]
    return (cropped_img * kernel).sum()

def plot_hist(panel, hist, bins, width):
    for t,n in zip(bins,hist):
        panel.bar(t,n,width=width)


class GaborCell(object):
    """
    A spike generator with a Gabor-filter receptive field.
    
    It has two variables: `x` is something like the membrane potential and is
    obtained by convolving the part of the current image frame within the
    receptive field with the Gabor kernel. `y` is the instantanous firing rate
    and is obtained by applying a sigmoid function to `x`.
    
    The cell produces an inhomogeneous Poisson spike train, accessible as the
    attribute `spike_train`, with instantaneous firing rate given by y.
    
    Usage::
    
        gc = GaborCell(gabor_parameters, non_linearity_parameters)
        gc.setup(nframes, frame_duration)
        for img in list_of_image_frames:
            gc.present_image(img)
        gc.generate_spikes(time_bins)
        
    """
    
    highest_id = -1
    
    def __init__(self, gabor_parameters, non_linearity_parameters):
        """
        `gabor_parameters` should contain parameters named `'gamma'`,
        `'sigma'`, `'k'`, `'psi'`, `'kernel_size'` and `'normalized'`.
        See `build_kernel()` for a definition of the parameters.
        
        `non_linearity_parameters` should contain parameters named `'x0'`,
        `'k'` and `'gain'`.
        
        The Gabor orientation is chosen randomly.
        """
        GaborCell.highest_id += 1
        self.id = GaborCell.highest_id
        for name,val in gabor_parameters.items():
            setattr(self, name, val)
        self.theta = numpy.random.uniform(0, pi)
        logging.info("Cell #%d has orientation %5.1f degrees" % (self.id, self.theta*180/pi))
        self.kernel = build_kernel(self.kernel_size, self.theta, self.gamma,
                                   self.sigma, self.k, self.psi, self.normalized)
        logging.debug("  kernel max = %g, min = %g (cell %d)" % (self.kernel.max(), self.kernel.min(), self.id))
        for name,val in non_linearity_parameters.items():
            setattr(self, name, val)
        self.spike_generator = StGen()
        
    def setup(self, nframes, frame_duration):
        """Prepare the cell to receive input."""
        self.x = numpy.zeros((nframes,), 'float')
        self.y = numpy.zeros((nframes,), 'float')
        self.orientation_tuning_curve = {}
        self.sim_duration = nframes*frame_duration
        self.frame_id = 0

    def present_image(self, img):
        """Present an image frame to the cell and calculate `x` and `y`."""
        i = self.frame_id
        self.x[i] = convolve(img, self.kernel)
        self.y[i] = self.gain * sigmoid(self.x[i], self.x0, self.k)
        self.frame_id += 1

    def generate_spikes(self, time_bins):
        """Once the `y` array has been calculated, generate the spike array."""
        global figures
        if P['debug']:
                panel = figures['gabors'].panel(self.id)
                panel.imshow(self.kernel)
                panel = figures['responses'].panel(self.id)
                panel.plot(time_bins, self.x)
                panel.plot(time_bins, self.y)
        assert len(time_bins) == len(self.y)
        spike_times = self.spike_generator.poissondyn_generator(time_bins,
                                                                self.y/1000.0,
                                                                self.sim_duration) # times in ms, rate in spikes/ms
        self.spike_train = SpikeTrain(spike_times, t_start=0, t_stop=self.sim_duration)
        logging.debug("  Generating spikes for GaborCell %d: %d spikes" % (self.id, len(self.spike_train)))
        logging.debug("  Maximum firing rate: %g spikes/s" % self.y.max())

    def spontaneous_firing_rate(self):
        """Return the value of `y` that would be obtained for zero input."""
        return self.gain * sigmoid(0.5*self.kernel.sum(), self.x0, self.k)

class AnalysisPipeline(Component):
    
    orientations = Input(None)
    sim_results = Input(None)
    tuning_method = Input(None)
    halfwidth_binwidth = Input(None)
    halfwidth_max = Input(None)
    histogram_min = Input(None)
    peaks = Output(None)
    hist = Output(None)
    bins = Output(None)
    curves = Output(None)
    
    tuning_curves = TuningCurves()
    fit_triangles = FitTriangles()
    histogram = Histogram()
    
    schema= [
                    #Inputs
                    [orientations, tuning_curves.category_time_series],
                    [sim_results, tuning_curves.spikelists],
                    [tuning_method, tuning_curves.method],
                    [halfwidth_binwidth, histogram.binwidth],
                    [halfwidth_max, histogram.maximum],
                    [histogram_min, histogram.minimum],
                    #Connection between subcomponents
                    [tuning_curves.stats, fit_triangles.tuning_curves],
                    [fit_triangles.widths, histogram.values],
                    #Outputs
                    [tuning_curves.stats, curves],
                    [fit_triangles.peaks, peaks],
                    [histogram.hist, hist],
                    [histogram.bins, bins]
                    ]

class AnalysisPostProcessing(Component) :
    
    output_dir = Input(None)
    figures = Input(None)
    cells = Input(None)
    peaks = Input(None)  
    hist = Input(None)
    bins = Input(None)
    curves = Input(None) 
    saved = Output(None)
    
    debug_tuning_curves = DebugTuningCurves()
    debug_histogram = DebugHistogram()
    peak_accuracy = PeakAccuracy()
    save_analysis = StoreResult()
    
    schema = [[figures, debug_tuning_curves.figures],
                     [curves, debug_tuning_curves.tuning_curves],
                     [hist, debug_histogram.hist],
                     [bins, debug_histogram.bins],
                     [cells, peak_accuracy.cells],
                     [peaks, peak_accuracy.peaks],
                     [output_dir, save_analysis.output_dir],
                     [debug_tuning_curves.figures_changed, debug_histogram.figures],
                     [debug_histogram.figures_changed, peak_accuracy.figures],
                     [peak_accuracy.figures_changed, save_analysis.figures],
                     [save_analysis.saved, saved]
                    ]

class SimpleV1Analysis(Component) :
    """Simple V1 analysis to test the Component Framework."""
    
    orientations = Input(None)
    sim_results = Input(None)
    tuning_method = Input(None)
    halfwidth_binwidth = Input(None)
    halfwidth_max = Input(None)
    histogram_min = Input(None)
    output_dir = Input(None)
    figures = Input(None)
    cells = Input(None)
    saved = Output(None)
    
    pipeline = AnalysisPipeline()
    tracer = AnalysisPostProcessing()
    
    schema = [#Inputs
                     [orientations, pipeline.orientations],
                     [sim_results, pipeline.sim_results],
                     [tuning_method, pipeline.tuning_method],
                     [halfwidth_binwidth, pipeline.halfwidth_binwidth],
                     [halfwidth_max, pipeline.halfwidth_max],
                     [histogram_min, pipeline.histogram_min],
                     [figures, tracer.figures],
                     [cells, tracer.cells],
                     [output_dir, tracer.output_dir],
                     #Subconnections
                     [pipeline.peaks, tracer.peaks],  
                     [pipeline.hist, tracer.hist],
                     [pipeline.bins, tracer.bins],
                     [pipeline.curves, tracer.curves],
                     #Outputs
                     [saved, tracer.saved]
                     ]

# === Read parameter file on import ============================================

parameter_file = "simpleV1.param"
curdir,pyc = os.path.split(os.path.abspath(__file__))
P = parameters.read_parameters(os.path.join(curdir,parameter_file))
console.setLevel(logging.WARNING)

# === If run as a standalone script, run the model and analysis ================

if __name__ == "__main__":
    
#    logging.warning('Warning : Launch the original stand alone script.')
    
    if P['debug']:
        matplotlib.rcParams['interactive'] = True
    stimulus_url = "/home/thierry/Benchmarks_Project/benchmarks/moving_gbar10_w0.10_c0.5_v10.0.zip"
    scale_factor = 10.0 # pixels/degree
    repetitions = 10
    halfwidth_binwidth = 2.5
    halfwidth_max = 70

#    record('spikes', {'not-specified': {'not-specified': P['n_cells']}})
#    orientations = set_stimulus(stimulus_url, scale_factor, None, None) # this sets orientations as a global, but could also return orientations?
#    results = []
#    for i in range(repetitions):
#        logging.info("Trial #%d" % i)
#        spike_list = run(quiet=False).values()[0]
#        logging.debug("  Spike train for cell #0: %s" % str(spike_list[0]))
#        results.append(spike_list)
#    
#    # Analyse results
#    tuning_curves = Benchmarks.analysis.tuning_curves(results, orientations)
#    
#    if P['debug']:
#        xdata = tuning_curves[0]['mean'].keys()
#        xdata.remove("null")
#        xdata.sort()
#        cdata = [float(k) for k in xdata]
#        for cell_id, tuning_curve in tuning_curves.items():
#            panel = figures['orientations'].next_panel()
#            ydata = [tuning_curve['mean'][x] for x in xdata]
#            yerr  = [tuning_curve['stderr'][x] for x in xdata]
#            panel.errorbar(cdata, ydata, yerr)
#    
#    # Fit triangles to tuning curves
#    widths = []
#    peaks = []
#    if P['debug']:
#        figures['orientations']._curr_panel = 0
#    for cell_id, tuning_curve in tuning_curves.items():
#        logging.info("Calculating orientation tuning curve width for cell #%d" % cell_id)
#        triangle = Benchmarks.analysis.tuning_curve_fit_triangle(tuning_curve['mean'], tuning_curve['stderr'])
#        hwhm = triangle.halfwidth()
#        peak = triangle.peak_location()
#        logging.info("Half-width at half-maximum for cell #%d: %g degrees" % (cell_id, hwhm))
#        widths.append(hwhm)
#        peaks.append(peak)
#        
#    # Calculate histogram of tuning curve widths
#    hist, bins = Benchmarks.analysis.histogram(widths, halfwidth_binwidth, 0, halfwidth_max)
#    logging.info("Tuning curve width distribution:")
#    logging.info("%s" % bins)
#    logging.info("%s" % hist)
#    if P['debug']:
#        figures['width_histogram'] = plotting.SimpleMultiplot(1, 1, xlabel="Average half-width (deg)", ylabel="Number of cells")
#        panel = figures['width_histogram'].next_panel()
#        plot_hist(panel, hist, bins, width=2.0)
#    
#    # Plot accuracy of peak estimation
#    if P['debug']:
#        real_peaks = [cell.theta*180/pi for cell in cells]
#        figures['peaks'] = plotting.SimpleMultiplot(1, 1, xlabel="Gabor orientation (deg)", ylabel="Estimated orientation")
#        panel = figures['peaks'].next_panel()
#        panel.plot(real_peaks, peaks, 'ro')
#    
#    # Save figures to file
#    if P['debug']:
#        if os.path.exists(P['output_dir']): 
#            for label,fig in figures.items():
#                fig.save(os.path.join(P['output_dir'], "simpleV1_%s.png" % label))
#        else:
#            print "Output directory %s does not exist." % P['output_dir']
#    
#    logging.warning('Warning : Launch AnalysisPipeline and AnalysisPostProcessing with original V1 Simulator.')
#    figures_1 = figures.copy()
#    figures_2 = figures.copy()
    #Analysis (not multiprocessing)
#    pipeline = AnalysisPipeline()
#    pipeline.register(auto=True)
#    inputs = {pipeline.orientations : orientations,
#                    pipeline.sim_results : results,
#                    pipeline.tuning_method : 'sum',
#                    pipeline.halfwidth_binwidth : 2.5,
#                    pipeline.halfwidth_max : 70,
#                    pipeline.histogram_min : 0}
#    pipeline.power_on(inputs)
#    pipeline.process(parallel=False)
#    pipeline.snapshot('snapshot test for pipeline 1')
#    pipeline.snapshot('snapshot test for pipeline 2')
#    pipeline.db_info()
    #pipeline.info()
    
    #PostProcessing (not multiprocessing)
#    tracer = AnalysisPostProcessing()
#    tracer.register(auto=True)
#    inputs = {tracer.figures : figures,
#                    tracer.cells : cells,
#                    tracer.peaks : pipeline.peaks.potential,
#                    tracer.hist : pipeline.hist.potential,
#                    tracer.bins : pipeline.bins.potential,
#                    tracer.curves : pipeline.curves.potential,
#                    tracer.output_dir : P['output_dir']
#                    }
#    tracer.power_on(inputs)
#    tracer.process(parallel=False)
#    tracer.snapshot('snapshot test for tracer 1')
#    tracer.snapshot('snapshot test for tracer 2')
#    tracer.db_info()
    #tracer.info()

    logging.warning('Warning : Launch SpikeGenerator')
    # Global variables (to be set later)
    inputs = {'P' : P, 
                    'measureable' : 'spikes',
                    'brain_region_dict' : {'not-specified': {'not-specified': P['n_cells']}},
                    'figures' : {},
                    'stimulus_url' : "/home/thierry/Benchmarks_Project/benchmarks/moving_gbar10_w0.10_c0.5_v10.0.zip",
                    'scale_factor' : 10.0,
                    'max_luminance' : None,
                    'background_luminance' : None,
                    'quiet' : False,
                    'repetitions' : 10}
    spikegenerator = SpikeGenerator(inputs)
    spikegenerator.register(auto=True)
    spikegenerator.process(parallel=False)
    spikegenerator.snapshot("SpikeGenerator Test 1")
    spikegenerator.db_info()

    #Analysis (not multiprocessing)
#    pipeline = AnalysisPipeline()
#    inputs = {pipeline.orientations : spikegenerator.orientations,
#                    pipeline.sim_results : spikegenerator.spikelists,
#                    pipeline.tuning_method : 'sum',
#                    pipeline.halfwidth_binwidth : 2.5,
#                    pipeline.halfwidth_max : 70,
#                    pipeline.histogram_min : 0}
#    pipeline.power_on(inputs)
#    pipeline.process(parallel=False)
    #pipeline.info()
    
    #PostProcessing (not multiprocessing)
#    tracer = AnalysisPostProcessing()
#    inputs = {tracer.figures : spikegenerator.figures_changed,
#                    tracer.cells : spikegenerator.cells_cache,
#                    tracer.peaks : pipeline.peaks.potential,
#                    tracer.hist : pipeline.hist.potential,
#                    tracer.bins : pipeline.bins.potential,
#                    tracer.curves : pipeline.curves.potential,
#                    tracer.output_dir : P['output_dir']
#                    }
#    tracer.power_on(inputs)
#    tracer.process(parallel=False)
    #tracer.info()
    
    #Analysis (not multiprocessing)
#    inputs = {'orientations' : orientations,
#                    'sim_results' : results,
#                    'tuning_method' : 'sum',
#                    'halfwidth_binwidth' : halfwidth_binwidth,
#                    'halfwidth_max' : halfwidth_max,
#                    'histogram_min' : 0,
#                    'figures' : figures,
#                    'cells' : cells,
#                    'output_dir' : P['output_dir']
#                    }

#    analysis = SimpleV1Analysis()
#    analysis.register(auto=True) # be careful auto=True force registration of subcomponents if they do not exist and delete all data
#    inputs = {analysis.orientations : spikegenerator.orientations.potential,
#                    analysis.sim_results : spikegenerator.spikelists.potential,
#                    analysis.tuning_method : 'sum',
#                    analysis.halfwidth_binwidth : halfwidth_binwidth,
#                    analysis.halfwidth_max : halfwidth_max,
#                    analysis.histogram_min : 0,
#                    analysis.figures : spikegenerator.figures_changed.potential,
#                    analysis.cells : spikegenerator.cells_cache.potential,
#                    analysis.output_dir : P['output_dir']
#                    }
#    analysis.power_on(inputs)
#    analysis.process(parallel = False)
#    intermediates = [analysis.pipeline.tuning_curves.method, analysis.pipeline.tuning_curves.spikelists,  analysis.tracer.curves]
#    analysis.snapshot('snapshot test for analysis', comments = "test snapshot function.")
#    analysis.snapshot('snapshot test for analysis 2', comments = "test snapshot function.")
#    analysis.snapshot('snapshot test for analysis 3', comments = "test snapshot function.")
#    #analysis.info()
#    analysis.db_info()
#    potentials = models.Potential.objects.filter(python_object__exact = 2.5)
#    print potentials[0].python_object
#    
#    tuning = TuningCurves()
#    tuning.info()
#    #tuning.db_info()
    
