from urllib import urlencode

__author__ = 'shillaker'


# Builds up the request string for the given query
def build_select_request(query, start_idx, n_rows):

    spec = query.spec

    request_list = list()

    # Build Q string
    add_q_field_to_request(request_list, spec.q_sub_queries)

    # Add FQ string
    add_field_value_list_to_request(request_list, 'fq', spec.fq_sub_queries)

    # Add sort
    add_sort_string_to_request(request_list, spec)

    # Add result fields
    add_comma_separated_field_to_request(request_list, 'fl', spec.result_fields)

    # Add facets
    add_facets_to_request(request_list, spec)

    # Add facet ranges
    add_facets_ranges_to_request(request_list, spec.facet_ranges)

    # Add number of rows
    add_to_request(request_list, 'rows', n_rows)

    # Add start index
    add_to_request(request_list, 'start', start_idx)

    # Add JSON response type
    add_to_request(request_list, 'wt', 'json')

    # De-dupe
    request_list = list(set(request_list))

    # Encode the full list
    request = urlencode(request_list)

    request = '/select?%s' % request

    return request


# Add generic field/ value to list
def add_to_request(request_list, prefix, value):
    if value is None or value == '':
        return
    request_list.append((prefix, value))


# Encodes a value
def encode_value(value):
    if value == '*':
        return value

    # Encode lists
    if not is_single_value(value):
        encoded_values = list()
        for v in value:
            encoded_values.append(encode_value(v))
        return encoded_values

    # Quote single strings
    if isinstance(value, basestring):
        return '"%s"' % value

    # Stringify booleans
    if isinstance(value, bool):
        return 'true' if value else 'false'

    # Encode all other values as unicode
    return unicode(value)


# Builds the facet string
def add_facets_to_request(request_list, query):
    if query.facet_fields is None or len(query.facet_fields) == 0:
        return

    # Append list of repeated facet.field elements
    facet_field_list = list()
    for ff in query.facet_fields:
        facet_field_list.append(('facet.field', ff))
    request_list.extend(facet_field_list)

    # Switch on faceting
    request_list.append(('facet', 'true'))


# Builds a facet range string
def add_facets_ranges_to_request(request_list, facet_ranges):
    if len(facet_ranges) == 0:
        return

    for facet_range in facet_ranges:
        field = facet_range.field
        request_list.append(('facet.range', field))
        request_list.append(('f.' + field + '.facet.range.start', facet_range.start))
        request_list.append(('f.' + field + '.facet.range.end', facet_range.end))
        request_list.append(('f.' + field + '.facet.range.gap', facet_range.gap))

        if facet_range.hardend is not None:
            request_list.append(('f.' + field + '.facet.range.hardend', facet_range.hardend))

        # There can be 0-n include parameters
        for include in facet_range.includes:
            request_list.append(('f.' + field + '.facet.range.include', include))

        # There can be 0-n other parameters
        for other in facet_range.others:
            request_list.append(('f.' + field + '.facet.range.other', other))

    # Switch on faceting
    request_list.append(('facet', 'true'))


# Builds a comma-separated field
def add_comma_separated_field_to_request(request_list, prefix, value_list):
    if value_list is None or len(value_list) == 0:
        return

    comma_sep_list = ','.join(value_list)
    request_list.append((prefix, comma_sep_list))


# Build a sort string
def add_sort_string_to_request(request_list, query):
    if query.sort_field is None:
        return

    if query.sort_order is None:
        sort_order = 'asc'
    else:
        sort_order = query.sort_order

    request_list.append(('sort', query.sort_field + ' ' + sort_order))


# Add Q field
def add_q_field_to_request(result_list, subquery_list):
    if len(subquery_list) == 0:
        result_list.append(('q', '*'))
    else:
        add_field_value_list_to_request(result_list, 'q', subquery_list)


# Builds up a list of tuples representing field/ value pairs from a subquery list
def add_field_value_list_to_request(result_list, prefix, subquery_list):
    if len(subquery_list) == 0:
        return

    # Normal query params
    query_string = ''
    for sq in subquery_list:
        # Stringify subquery and append
        sq_string = field_value_subquery_to_string(sq)
        if query_string == '':
            query_string = sq_string
        else:
            query_string += ' ' + sq.type + ' ' + sq_string

    result_list.append((prefix, query_string))


# Converts a field/ value subquery to a request string
def field_value_subquery_to_string(subquery):
    # Build up list of parameters for each field/ value pair involved
    res_string = subquery.field + ':'

    encoded_value = encode_value(subquery.value)
    if is_single_value(subquery.value):
        res_string += str(encoded_value)
    else:
        join_string = ' ' + subquery.internal_type + ' '
        value_list = join_string.join(encoded_value)
        res_string += '(' + value_list + ')'

    return res_string


# Returns true/ false depending on whether we see this as a single value
def is_single_value(value):
    if isinstance(value, (frozenset, set, list, tuple)):
        return False
    else:
        return True