from icarus.http.solr_http import http_get_json_response
from icarus.query.select_request_builder import build_select_request
from icarus.result.select_response_handler import build_select_result

__author__ = 'shillaker'

OR = 'OR'
AND = 'AND'
ASCENDING = 'asc'
DESCENDING = 'desc'


# Generic subquery
class FieldValuesSubQuery():
    def __init__(self, field, value, subquery_type, internal_type):
        self.field = field
        self.value = value
        self.type = subquery_type
        self.internal_type = internal_type


# Select sub query
class SelectSubQuery(FieldValuesSubQuery):
    def __init__(self, field, value, subquery_type, internal_type):
        FieldValuesSubQuery.__init__(self, field, value, subquery_type, internal_type)


# Filter sub query
class FilterSubQuery(FieldValuesSubQuery):
    def __init__(self, field, value, subquery_type, internal_type):
        FieldValuesSubQuery.__init__(self, field, value, subquery_type, internal_type)


# Facet range field
class FacetRange():
    def __init__(self, field, start, end, gap):
        self.field = field
        self.start = start
        self.end = end
        self.gap = gap
        self.hardend = None
        self.includes = list()
        self.others = list()


# Internal class used to specify queries
class SearchQuerySpec():
    def __init__(self):
        self.q_sub_queries = list()
        self.fq_sub_queries = list()
        self.sort_field = None
        self.sort_order = None
        self.result_fields = None
        self.facet_fields = None
        self.facet_ranges = list()


# Wrapper class for a Solr query
class SearchQuery():
    def __init__(self, interface):
        self.interface = interface
        self.spec = SearchQuerySpec()

    # Adds multiple Q subqueries
    def q_multi(self, external, field_values, internal=OR):
        if field_values is None:
            return
        for field in sorted(field_values.keys()):
            self.q(external, field, field_values[field], internal)
        return self

    # ANDs multiple Q subqueries
    def q_multi_and(self, field_values, internal=OR):
        self.q_multi(AND, field_values, internal)
        return self

    # ANDs multiple Q subqueries
    def q_multi_or(self, field_values, internal=OR):
        self.q_multi(OR, field_values, internal)
        return self

    # Adds multiple FQ subqueries
    def fq_multi(self, external, field_values, internal=OR):
        if field_values is None:
            return
        for field, value in field_values.iteritems():
            self.fq(external, field, value, internal)
        return self

    # ANDs multiple FQ subqueries
    def fq_multi_and(self, field_values, internal=OR):
        self.fq_multi(AND, field_values, internal)
        return self

    # ANDs multiple FQ subqueries
    def fq_multi_or(self, field_values, internal=OR):
        self.fq_multi(OR, field_values, internal)
        return self

    # Adds a Q subquery
    def q(self, external, field, value, internal=OR):
        self.spec.q_sub_queries.append(SelectSubQuery(field, value, external, internal))
        return self

    # ANDs a Q subquery
    def q_and(self, field, value, internal=OR):
        self.q(AND, field, value, internal)
        return self

    # ORs a Q subquery
    def q_or(self, field, value, internal=OR):
        self.q(OR, field, value, internal)
        return self

    # Adds a FQ subquery
    def fq(self, external, field, value, internal=OR):
        self.spec.fq_sub_queries.append(FilterSubQuery(field, value, external, internal))
        return self

    # ANDs a FQ subquery
    def fq_and(self, field, value, internal=OR):
        self.fq(AND, field, value, internal)
        return self

    # ORs a FQ subquery
    def fq_or(self, field, value, internal=OR):
        self.fq(OR, field, value, internal)
        return self

    # Add sort to normal query
    def sort(self, field, order=ASCENDING):
        self.spec.sort_field = field
        self.spec.sort_order = order
        return self

    # Add result fields
    def result_fields(self, fields):
        self.spec.result_fields = fields
        return self

    # Add result fields
    def facet_fields(self, facet_fields):
        self.spec.facet_fields = facet_fields
        return self

    # Add a facet range field
    def add_facet_range(self, facet_range):
        self.spec.facet_ranges.append(facet_range)
        return self

    # Execute the query
    def execute(self, start_idx, n_items=10):
        request_string = build_select_request(self, start_idx, n_items)
        response = http_get_json_response(self.interface, request_string)
        result = build_select_result(response)

        return result
