# -*- coding: utf-8 -*-

"""
:copyright: Copyright (C) 2013 Gianluca Costa.
:license: LGPLv3, see LICENSE for details.
"""

from __future__ import unicode_literals

import unittest
import os

from ..web import WebPath, SiteInfoService, RequestInfoService, Page
from ..strings import String


class WebPathTests(unittest.TestCase):
	def testIsAbsoluteShouldReturnTrueForAbsolutePaths(self):
		webPath = WebPath("/test/tau/")
		assert(webPath.isAbsolute())

	def testIsAbsoluteShouldReturnFalseForRelativePaths(self):
		webPath = WebPath("test/tau2/")
		assert(not webPath.isAbsolute())

	def testStrShouldReturnTheSourcePathForAbsolutePaths(self):
		webPathString = "/test/tau/"
		webPath = WebPath(webPathString)
		self.assertEquals(webPathString, String(webPath))


	def testStrShouldReturnTheSourcePathForRelativePaths(self):
		webPathString = "test/tau/"
		webPath = WebPath(webPathString)
		self.assertEquals(webPathString, String(webPath))


	def testGetAbsoluteShouldIgnoreTheReferencePathIfThisIsAbsolute(self):
		webPath = WebPath("/test/tau")
		assert(webPath.getAbsolute(None).isAbsolute())


	def testAnAbsoluteWebPathShouldEqualAStringEqualToItsString(self):
		webPathString = "/test/tau/"
		webPath = WebPath(webPathString)
		self.assertEquals(webPathString, webPath)

	def testARelativeWebPathShouldEqualAStringEqualToItsString(self):
		webPathString = "test/tau/"
		webPath = WebPath(webPathString)
		self.assertEquals(webPathString, webPath)


	def testAnAbsoluteWebPathShouldEqualAWebPathEqualToItsString(self):
		webPathString = "/test/tau/"
		referenceWebPath = WebPath(webPathString)
		webPath = WebPath(webPathString)
		self.assertEquals(referenceWebPath, webPath)


	def testARelativeWebPathShouldEqualAWebPathEqualToItsString(self):
		webPathString = "test/tau/"
		referenceWebPath = WebPath(webPathString)
		webPath = WebPath(webPathString)
		self.assertEquals(referenceWebPath, webPath)


	def testGetAbsoluteShouldWorkWithReferencePathStringsWithTrailingSlash(self):
		webPath = WebPath("alpha")
		absWebPath = webPath.getAbsolute("/test/tau/")
		self.assertEquals("/test/tau/alpha", absWebPath)


	def testGetAbsoluteShouldWorkWithReferencePathStringsWithoutTrailingSlash(self):
		webPath = WebPath("alpha")
		absWebPath = webPath.getAbsolute("/test/tau")
		self.assertEquals("/test/tau/alpha", absWebPath)

	def testGetAbsoluteShouldWorkWithReferenceWebPathWithTrailingSlash(self):
		webPath = WebPath("alpha")
		absWebPath = webPath.getAbsolute(WebPath("/test/tau/"))
		self.assertEquals("/test/tau/alpha", absWebPath)


	def testGetAbsoluteShouldWorkWithReferenceWebPathWithoutTrailingSlash(self):
		webPath = WebPath("alpha")
		absWebPath = webPath.getAbsolute(WebPath("/test/tau"))
		self.assertEquals("/test/tau/alpha", absWebPath)

	def testAddShouldWorkForAbsoluteLeftOperand(self):
		leftWebPath = WebPath("/alpha")
		rightWebPath = "beta"
		self.assertEquals("/alpha/beta", leftWebPath + rightWebPath)

	def testAddShouldWorkForRelativeLeftOperand(self):
		leftWebPath = WebPath("alpha")
		rightWebPath = "beta"
		self.assertEquals("alpha/beta", leftWebPath + rightWebPath)


	def testGetParentShouldWorkWithThreeLevelsOfAbsolutePath(self):
		webPath = WebPath("/alpha/beta/gamma")
		self.assertEquals("/alpha/beta", webPath.getParent())


	def testGetParentShouldWorkWithThreeLevelsOfRelativePath(self):
		webPath = WebPath("alpha/beta/gamma")
		self.assertEquals("alpha/beta", webPath.getParent())


	def testGetParentShouldReturnRootWithOneLevelOfAbsolutePath(self):
		webPath = WebPath("/alpha")
		self.assertEquals("/", webPath.getParent())

	def testGetParentShouldReturnNoneWithOneLevelOfRelativePath(self):
		webPath = WebPath("alpha")
		self.assertIs(None, webPath.getParent())



class SiteInfoServiceTests(unittest.TestCase):

	def setUp(self):
		self._fakeDocumentRoot = os.path.join("fake", "usr", "dir")
		os.environ["DOCUMENT_ROOT"] = self._fakeDocumentRoot
		self._siteInfoService = SiteInfoService()

	def testGetRootPathShouldReturnTheDocumentRoot(self):
		self.assertEquals(self._fakeDocumentRoot, self._siteInfoService.getRootPath())

	def testGetPathShouldWorkWithAbsolutePathString(self):
		path = self._siteInfoService.getPath("/website/test.htm")
		self.assertEquals(os.path.join("fake", "usr", "dir", "website", "test.htm"), path)

	def testGetPathShouldFailWithAbsolutePathString(self):
		self.assertRaises(AssertionError, self._siteInfoService.getPath, "website/test.htm")



class RequestInfoServiceTests(unittest.TestCase):
	def setUp(self):
		self._requestWebPath = WebPath("/web/service/test")
		self._requestScriptWebPath = WebPath("/web/service/test.py")

		os.environ["REQUEST_URI"] = String(self._requestWebPath)
		os.environ["SCRIPT_NAME"] = String(self._requestScriptWebPath)
		self._requestInfoService = RequestInfoService()


	def testGetRequestScriptWebPathShouldWork(self):
		self.assertEquals(self._requestScriptWebPath, self._requestInfoService.getRequestScriptWebPath())


	def testGetRequestWebPathShouldWork(self):
		self.assertEquals(self._requestWebPath, self._requestInfoService.getRequestWebPath())


	def testGetAbsolutePathShouldReturnThePassedPathIfItIsAbsolute(self):
		path = "/software/test/x.htm"

		self.assertEquals(path, self._requestInfoService.getAbsoluteWebPath(path))

	def testGetAbsolutePathShouldReturnAnAbsolutePath(self):
		path = "software/test/x.htm"
		absolutePath = self._requestInfoService.getAbsoluteWebPath(path)
		self.assertEquals("/web/service/software/test/x.htm", String(absolutePath))


	def testIsInDebuggingShouldReturnTrueIfServerEndsWithLocalHost(self):
		os.environ["SERVER_NAME"] = "mytest.alpha.localhost"
		self.assertTrue(self._requestInfoService.isInDebugging())	


	def testIsInDebuggingShouldReturnFalseIfServerDoesNotEndWithLocalHost(self):
		os.environ["SERVER_NAME"] = "mytest.beta"
		self.assertFalse(self._requestInfoService.isInDebugging())
		
		
	def testGetServerNameShouldWork(self):
		serverName = "test.localhost"
		os.environ["SERVER_NAME"] = serverName
		
		self.assertEquals(serverName, self._requestInfoService.getServerName())
		
		
	def testGetServerPortShouldWork(self):
		serverPort = "80"
		os.environ["SERVER_PORT"] = serverPort
		
		self.assertEquals(serverPort, self._requestInfoService.getServerPort())
		
		
	def testGetCookieStringShouldWork(self):
		os.environ["HTTP_COOKIE"] = "My cookie string"
		self.assertEquals("My cookie string", self._requestInfoService.getCookieString())
		
		


class PageTests(unittest.TestCase):
	def testPageRenderingShouldWorkWithAnIntegerView(self):				
		pageContent = 90
		
		pageMimeType = "text/plain"
		pageEncoding = "utf-8"
		
		page = Page(pageContent, pageMimeType, pageEncoding)
		
		expectedRendering = "Content-Type: {0}; charset={1}\r\n\r\n{2}".format(pageMimeType, pageEncoding, pageContent)
		
		self.assertEquals(expectedRendering, String(page))
		
		
	def testAddHeaderShouldWork(self):				
		pageContent = 90
		
		pageMimeType = "text/plain"
		pageEncoding = "utf-8"
		
		page = Page(pageContent, pageMimeType, pageEncoding)
		
		page.addHeader("Max-Forwards: 15")
		
		expectedRendering = "Content-Type: {0}; charset={1}\r\nMax-Forwards: 15\r\n\r\n{2}".format(pageMimeType, pageEncoding, pageContent)
		
		self.assertEquals(expectedRendering, String(page))

		
		
	def testPageRenderingShouldWorkWithAStringHavingAccents(self):				
		pageContent = """
		Hello, this is a test page.
		It contains multiple lines of text.
		C'è anche, però, qualche carattere accentato.
		Et voilà, bien évidemment, d'autres accents.		
		"""
		
		pageMimeType = "text/plain"
		pageEncoding = "utf-8"
		
		page = Page(pageContent, pageMimeType, pageEncoding)
		
		expectedRendering = "Content-Type: {0}; charset={1}\r\n\r\n{2}".format(pageMimeType, pageEncoding, pageContent)
		
		self.assertEquals(expectedRendering, String(page))