# -*- coding: utf-8 -*-

"""
:copyright: Copyright (C) 2013 Gianluca Costa.
:license: LGPLv3, see LICENSE for details.
"""

from __future__ import unicode_literals

import unittest
import os
import shutil

from ..io import PathOperations


class PathOperationsTests(unittest.TestCase):
	def setUp(self):
		self._ioTestsPath = os.path.join(os.path.dirname(__file__), "ioTests")
	
	
	def tearDown(self):
		if os.path.exists(self._ioTestsPath):
			shutil.rmtree(self._ioTestsPath)
		
	
	def testSafeMakeDirsShouldWork(self):
		pathToCreate = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		assert(PathOperations.safeMakeDirs(pathToCreate))
				
		assert(os.path.isdir(pathToCreate))
		
	
	def testSafeMakeDirsShouldReturnFalseForAnExistingPath(self):
		pathToCreate = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		PathOperations.safeMakeDirs(pathToCreate)
		
		self.assertFalse(PathOperations.safeMakeDirs(pathToCreate))
		
	
	def testTouchShouldCreateIntermediateDirs(self):
		pathToCreate = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		PathOperations.touch(pathToCreate)
		
		assert(os.path.isfile(pathToCreate))
		
	
	def testTouchShouldDoNothingIfTheFileAlreadyExists(self):
		pathToCreate = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		PathOperations.touch(pathToCreate)
		PathOperations.touch(pathToCreate)
		
	
	def testSafeRemoveShouldWork(self):
		pathToCreate = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		PathOperations.touch(pathToCreate)
		
		assert(PathOperations.safeRemove(pathToCreate))
		
	
	def testSafeRemoveShouldReturnFalseIfThePathDoesNotExist(self):
		pathToRemove = os.path.join(self._ioTestsPath, "alpha", "beta", "gamma")
		
		self.assertFalse(PathOperations.touch(pathToRemove))