# -*- coding: utf-8 -*-

"""
:copyright: Copyright (C) 2013 Gianluca Costa.
:license: LGPLv3, see LICENSE for details.
"""


from __future__ import unicode_literals

import unittest
import os

from ..versioning import Version, VersionDirectory, InvalidVersionException
from ..strings import String


class VersionTests(unittest.TestCase):
	def testConstructorShouldInitFieldsCorrectly(self):
		version = Version("1.2.3.4")
		self.assertEquals(1, version.getMajor())
		self.assertEquals(2, version.getMinor())
		self.assertEquals(3, version.getBuild())
		self.assertEquals(4, version.getRevision())
		
	
	def testConstructorShouldFailIfTheVersionStringIsTooLong(self):
		self.assertRaises(InvalidVersionException, Version, "1.2.3.4.5")
		
	
	def testConstructorShouldFailIfTheVersionStringIsInvalid(self):
		self.assertRaises(InvalidVersionException, Version, "AnInvalidVersionString")
		
	
	def testConstructorTakingAnotherVersionShouldWork(self):
		version = Version("6.7.8.9")
		self.assertEquals(version, Version(version))
	
	
	def testReprShouldReturnTheRawString(self):
		rawString = "7.6.0"
		version = Version(rawString)
		self.assertEquals(rawString, repr(version))
		

	def testGetFriendlyString_ShouldWorkWithOnlyRevisionNonZero(self):
		versionString = "0.0.0.4"
		version = Version(versionString)
		self.assertEquals(versionString, String(version))
		
	
	def testGetFriendlyString_ShouldWorkWithAlternateZeros(self):
		versionString = "0.2.0.4"
		version = Version(versionString)
		self.assertEquals(versionString, String(version))



	def testStringComparisonShouldWork(self):
		version = Version("1.2.3.4")
		self.assertEquals(version, "1.2.3.4")


	def testStringComparisonShouldAllowOmittingTrailingZeros(self):
		version = Version("1.2")
		self.assertEquals(version, "1.2")
		self.assertEquals(version, "1.2.0")
		self.assertEquals(version, "1.2.0.0")


	def testConstructorShouldWorkWithOnlyMajor(self):
		version = Version("5")
		self.assertEquals(version, "5")


	def testConstructorShouldWorkWithMajorAndMinor(self):
		version = Version("5.6")
		self.assertEquals(version, "5.6")


	def testConstructorShouldWorkWithMajorMinorAndBuild(self):
		version = Version("5.6.7")
		self.assertEquals(version, "5.6.7")


	def testConstructorShouldWorkWithMajorMinorBuildAndRelease(self):
		version = Version("5.6.7.8")
		self.assertEquals(version, "5.6.7.8")


	def testComparisonShouldWorkWithMajor(self):
		laterVersion = Version("5")
		olderVersion = Version("4")

		assert(olderVersion < laterVersion)


	def testComparisonShouldWorkWithMinor(self):
		laterVersion = Version("5.4")
		olderVersion = Version("5.3")

		assert(olderVersion < laterVersion)


	def testComparisonShouldWorkWithBuild(self):
		laterVersion = Version("5.4.8")
		olderVersion = Version("5.4.7")

		assert(olderVersion < laterVersion)


	def testComparisonShouldWorkWithRevision(self):
		laterVersion = Version("5.4.8.3")
		olderVersion = Version("5.4.8.2")

		assert(olderVersion < laterVersion)


	def testEqualityShouldWorkWithMajor(self):
		firstVersion = Version("4")
		secondVersion = Version("4")

		self.assertEquals(firstVersion, secondVersion)


	def testEqualityShouldWorkWithMinor(self):
		firstVersion = Version("4.5")
		secondVersion = Version("4.5")

		self.assertEquals(firstVersion, secondVersion)


	def testEqualityShouldWorkWithBuild(self):
		firstVersion = Version("4.5.6")
		secondVersion = Version("4.5.6")

		self.assertEquals(firstVersion, secondVersion)


	def testEqualityShouldWorkWithRevision(self):
		firstVersion = Version("4.5.6.7")
		secondVersion = Version("4.5.6.7")

		self.assertEquals(firstVersion, secondVersion)



class VersionDirectoryTests(unittest.TestCase):
	def setUp(self):
		self._versionDirectory = VersionDirectory(os.path.join(os.path.dirname(__file__), "versions"))
		self._inexistingVersionDirectory = VersionDirectory(os.path.join(os.path.dirname(__file__), "INEXISTING_VERSION_DIR"))


	def testGetVersionsShouldWork(self):
		versions = self._versionDirectory.getVersions()
		versions.sort()

		self.assertEquals([Version("2"), Version("3.5.6.7"), Version("4.8"), Version("6.1.2")], versions)
		
	
	def testGetVersionsShouldReturnAnEmptyListForAnInexistingDirectory(self):
		versions = self._inexistingVersionDirectory.getVersions()
		
		self.assertEquals([], versions)


	def testGetLatestVersionShouldWork(self):
		latestVersion = self._versionDirectory.getLatestVersion()

		self.assertEquals(Version("6.1.2"), latestVersion)


	def testGetLatestVersionShouldReturnNoneForAnInexistingDirectory(self):
		latestVersion = self._inexistingVersionDirectory.getLatestVersion()
		
		self.assertIsNone(latestVersion)		