# -*- coding: utf-8 -*-

"""
:copyright: Copyright (C) 2013 Gianluca Costa.
:license: LGPLv3, see LICENSE for details.
"""

import unittest

from ..ioc import Container, TransientRegistration, SingletonRegistration


class MyIocClass(object):
	_instances = 0
	
	def __init__(self):
		MyIocClass._instances += 1
		self.val = 0
		
	def dispose(self):
		MyIocClass._instances -= 1
	

class ContainerTests(unittest.TestCase):
	def setUp(self):
		MyIocClass._instances = 0
		self._container = Container()
		
		
	def testRegisterTransientShouldCreateTransientRegistrations(self):
		self._container.registerTransient(MyIocClass, lambda container, key: MyIocClass())
		
		alpha = self._container.resolve(MyIocClass)
		beta = self._container.resolve(MyIocClass)
		
		alpha.val = 9
		beta.val = 10
		
		self.assertEquals(2, MyIocClass._instances)
		self.assertNotEquals(alpha.val, beta.val)
		
		
	def testRegisterSingletonShouldCreateSingletonRegistration(self):
		self._container.registerSingleton(MyIocClass, lambda container, key: MyIocClass())
		
		alpha = self._container.resolve(MyIocClass)
		beta = self._container.resolve(MyIocClass)
		
		alpha.val = 9
		beta.val = 10
		
		self.assertEquals(1, MyIocClass._instances)
		self.assertEquals(alpha.val, beta.val)
		
		
	def testAddRegistrationShouldFailWhenUsingADuplicateKey(self):
		transientRegistration = TransientRegistration(lambda container, key: MyIocClass())
		singletonRegistration = SingletonRegistration(lambda container, key: MyIocClass())
		
		self._container.addRegistration(MyIocClass, transientRegistration)
		self.assertRaises(KeyError, self._container.addRegistration, MyIocClass, singletonRegistration)
	
	
	def testFactoryMethodShouldNotBeNullForTransientRegistration(self):
		self.assertRaises(ValueError, self._container.registerTransient, MyIocClass, None)	
		
	
	def testFactoryMethodShouldNotBeNullForSingletonRegistration(self):
		self.assertRaises(ValueError, self._container.registerSingleton, MyIocClass, None)

	
	def testResolveShouldFailWhenUsingAnUnknownKey(self):
		self.assertRaises(KeyError, self._container.resolve, "Unknown key")
		
				
				
	def testDisposeShouldCallDisposalFunctionForSingletonInstances(self):
		self._container.registerSingleton(
			MyIocClass, 
			lambda container, key: MyIocClass(),
			lambda instance: instance.dispose())
		
		self._container.resolve(MyIocClass)
		
		self._container.dispose()
		
		self.assertEquals(0, MyIocClass._instances)
		
		
	
	def testDisposeShouldDoNothingForSingletonInstancesIfNoDisposalFunctionIsPassed(self):
		self._container.registerSingleton(
			MyIocClass, 
			lambda container, key: MyIocClass())
		
		self._container.resolve(MyIocClass)
		
		self._container.dispose()
		
		self.assertEquals(1, MyIocClass._instances)
		
	
	def testDisposeShouldDoNothingWithTransientInstances(self):
		self._container.registerTransient(MyIocClass, lambda container, key: MyIocClass())
		
		self._container.resolve(MyIocClass)
		self._container.resolve(MyIocClass)
				
		self._container.dispose()
		self.assertEquals(2, MyIocClass._instances)	
		
	
	def testChainingMethodsShouldWork(self):
		self._container\
				.registerTransient(MyIocClass, lambda container, key: MyIocClass())\
				.resolve(MyIocClass)
				
		self.assertEquals(1, MyIocClass._instances)