# -*- coding: utf-8 -*-

"""
:copyright: Copyright (C) 2013 Gianluca Costa.
:license: LGPLv3, see LICENSE for details.
"""

from __future__ import unicode_literals

import unittest
import os

from ..maven import MavenArtifact, MavenRepository
from ..strings import String


class MavenArtifactTests(unittest.TestCase):
	def setUp(self):
		self._fullArtifact = MavenArtifact("psi.tau", "alpha.beta", "4.5")
		
		
	def testConstructorShouldInitFieldsCorrectly(self):
		groupId = "myGroup"
		artifactId = "myArtifact"
		versionString = "8.10"
		description = "My description"
		scope = "My scope"
		
		artifact = MavenArtifact(groupId, artifactId, versionString, description, scope)
		self.assertEquals(groupId, artifact.getGroupId())
		self.assertEquals(artifactId, artifact.getArtifactId())
		self.assertEquals(versionString, artifact.getVersion())
		self.assertEquals(description, artifact.getDescription())
		self.assertEquals(scope, artifact.getScope())


	def testGetFileNameShouldWork(self):
		self.assertEquals("alpha.beta-4.5.jar", self._fullArtifact.getFileName())


	def testGetFileNameShouldWorkWithSuffix(self):
		self.assertEquals("alpha.beta-4.5-javadoc.jar", self._fullArtifact.getFileName("javadoc"))

	def testGetFileNameShouldWorkWithSuffixHavingLeadingDash(self):
		self.assertEquals("alpha.beta-4.5-javadoc.jar", self._fullArtifact.getFileName("-javadoc"))


	def testGetFileNameShouldWorkWithExtension(self):
		self.assertEquals("alpha.beta-4.5.nbm", self._fullArtifact.getFileName(extension="nbm"))

	def testGetFileNameShouldWorkWithExtensionHavingLeadingDot(self):
		self.assertEquals("alpha.beta-4.5.nbm", self._fullArtifact.getFileName(extension=".nbm"))


	def testGetFileNameShouldWorkWithSuffixAndExtension(self):
		self.assertEquals("alpha.beta-4.5-javadoc.nbm", self._fullArtifact.getFileName("javadoc", "nbm"))


	def testGetPathShouldWorkForGroupIdOnly(self):
		artifact = MavenArtifact("psi.tau")
		self.assertEquals("psi{0}tau".format(os.sep), artifact.getPath())


	def testGetPathShouldWorkForGroupIdAndArtifactId(self):
		artifact = MavenArtifact("psi.tau", "alpha.beta")
		self.assertEquals("psi{0}tau{0}alpha.beta".format(os.sep), artifact.getPath())


	def testGetPathShouldWorkForFullArtifact(self):
		artifact = self._fullArtifact
		self.assertEquals("psi{0}tau{0}alpha.beta{0}4.5{0}alpha.beta-4.5.jar".format(os.sep), artifact.getPath())


	def testGetPathShouldWorkForFullArtifactAndSuffix(self):
		artifact = self._fullArtifact
		self.assertEquals("psi{0}tau{0}alpha.beta{0}4.5{0}alpha.beta-4.5-javadoc.jar".format(os.sep), artifact.getPath("javadoc"))


	def testGetPathShouldWorkForFullArtifactAndExtension(self):
		artifact = self._fullArtifact
		self.assertEquals("psi{0}tau{0}alpha.beta{0}4.5{0}alpha.beta-4.5.nbm".format(os.sep), artifact.getPath(extension="nbm"))


	def testGetPathShouldWorkForFullArtifactAndSuffixAndExtension(self):
		artifact = self._fullArtifact
		self.assertEquals("psi{0}tau{0}alpha.beta{0}4.5{0}alpha.beta-4.5-javadoc.nbm".format(os.sep), artifact.getPath("javadoc", "nbm"))


	def testStrShouldWork(self):
		self.assertEquals("psi.tau:alpha.beta:4.5", String(self._fullArtifact))



class MavenRepositoryTests(unittest.TestCase):
	def setUp(self):
		self._rootPath = os.path.join(os.path.dirname(__file__), "mvn")
		self._mavenRepository = MavenRepository(self._rootPath)
		self._fullArtifact = MavenArtifact("psi.tau", "alpha.beta", "4.5")


	def testGetRootPathShouldWork(self):
		self.assertEquals(self._rootPath, self._mavenRepository.getRootPath())


	def testGetArtifactPathShouldWorkForGroupIdOnly(self):
		artifact = MavenArtifact("psi.tau")
		artifactPath = self._mavenRepository.getArtifactPath(artifact)

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau"), artifactPath)


	def testGetArtifactPathShouldWorkForGroupIdAndArtifactId(self):
		artifact = MavenArtifact("psi.tau", "alpha.beta")
		artifactPath = self._mavenRepository.getArtifactPath(artifact)

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau", "alpha.beta"), artifactPath)


	def testGetArtifactPathShouldWorkForFullArtifact(self):
		artifact = self._fullArtifact
		artifactPath = self._mavenRepository.getArtifactPath(artifact)

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau", "alpha.beta", "4.5", "alpha.beta-4.5.jar"), artifactPath)


	def testGetArtifactPathShouldWorkForFullArtifactAndSuffix(self):
		artifact = self._fullArtifact
		artifactPath = self._mavenRepository.getArtifactPath(artifact, "javadoc")

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau", "alpha.beta", "4.5", "alpha.beta-4.5-javadoc.jar"), artifactPath)


	def testGetArtifactPathShouldWorkForFullArtifactAndExtension(self):
		artifact = self._fullArtifact
		artifactPath = self._mavenRepository.getArtifactPath(artifact, extension="nbm")

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau", "alpha.beta", "4.5", "alpha.beta-4.5.nbm"), artifactPath)


	def testGetArtifactPathShouldWorkForFullArtifactAndSuffixAndExtension(self):
		artifact = self._fullArtifact
		artifactPath = self._mavenRepository.getArtifactPath(artifact, "javadoc", "nbm")

		self.assertEquals(os.path.join(self._rootPath, "psi", "tau", "alpha.beta", "4.5", "alpha.beta-4.5-javadoc.nbm"), artifactPath)
		
	
	def testLatestArtifactVersionShouldWork(self):		
		latestVersion = self._mavenRepository.getLatestArtifactVersion("test.group", "sample.artifact")
		
		self.assertEquals("28.3", latestVersion)
		
