from sys import stdout
from khronos.utils import indentation, MethodGroup

class Component(object):
    """This class supports the creation of trees of components, allowing a hierarchical 
    organization of models. Each component may be inside a parent component, and may contain 
    other components as members. A component with no parent is called a root component. 
    Components also have a name (which must be locally unique, i.e. unique among its siblings), 
    and a globally unique path that identifies how to reach the component starting from the 
    root component."""
    # -----------------------------------------------------
    # Path format control attributes ----------------------
    path_separator = "."
    path_parent    = "<"
    path_root      = "#"
    path_illegal   = set(path_separator + path_parent + path_root)
    
    # -----------------------------------------------------
    # Magic methods ---------------------------------------
    def __init__(self, name=None, parent=None, members=()):
        if name is None:
            name = self.__class__.autoname_generate()
        else:
            name = str(name)
        Component.validate_name(name)
        self.__name = name
        self.__path = name
        self.__parent = None
        self.__root = self
        self.__members = {}
        if parent is not None:
            parent.__members_add(self)
        for member in members:
            self.__members_add(member)
            
    def __repr__(self):
        return "<%s - %s object at 0x%08x>" % (self.__path, self.__class__.__name__, id(self))
        
    def __str__(self):
        return self.__path
        
    # -----------------------------------------------------
    # Basic component properties --------------------------
    @property
    def name(self):
        return self.__name
        
    @name.setter
    def name(self, name):
        name = str(name)
        Component.validate_name(name)
        if self.__parent is not None:
            if name in self.__parent.__members:
                raise KeyError("name '%s' already in use at %s" % (name, self.__parent.__path))
            # Rebind name in parent component
            del self.__parent.__members[self.__name]
            self.__parent.__members[name] = self
        self.__name = name
        self.__hierarchy_update()
        
    @property
    def path(self):
        return self.__path
        
    @property
    def parent(self):
        return self.__parent
    
    @parent.setter
    def parent(self, parent):
        if parent is not self.__parent:
            if self.__parent is not None:
                self.__parent.__members_remove(self)
            if parent is not None:
                parent.__members_add(self)
                
    @property
    def root(self):
        return self.__root
        
    # -----------------------------------------------------
    # 'members' method group ------------------------------
    members = MethodGroup("members")
    
    @members("__len__")
    def __len__(self):
        """Returns the number of members of the component."""
        return len(self.__members)
        
    @members("__iter__")
    def __iter__(self):
        """Iterates over the members of the component."""
        return self.__members.itervalues()
        
    @members("__contains__")
    def __contains__(self, member):
        """When a component is provided, this method checks that there is a member with 
        the same name in the local space, AND it must also be the exact same object that 
        was passed as argument.
        If an object of another type is provided, it checks whether the component has a 
        member whose name is equal to the string representation of the argument."""
        if isinstance(member, Component):
            try:
                return member is self.__members[member.__name]
            except KeyError:
                return False
        else:
            return str(member) in self.__members
            
    @members("__getitem__")
    def __getitem__(self, name):
        return self.__members[str(name)]
    
    @members("__setitem__")
    def __setitem__(self, name, member):
        name = str(name)
        Component.validate_name(name)
        if name in self.__members:
            raise KeyError("name '%s' already in use at %s" % (name, self.__path)) 
        if member.__parent is not None:
            member.__parent.__members_remove(member)
        member.name = name
        self.__members_add(member)
        
    @members("__delitem__")
    def __delitem__(self, name):
        target = self.__members[str(name)]
        self.__members_remove(target)
        
    @members("add")
    def add(self, member):
        """Add a member to a component."""
        name = member.__name
        if name in self.__members:
            raise KeyError("name '%s' already in use at %s" % (name, self.__path))
        if member.__parent is not None:
            member.__parent.__members_remove(member)
        member.__parent = self
        member.__hierarchy_update()
        self.__members[name] = member
        
    @members("extend")
    def extend(self, members):
        for member in members:
            self.__members_add(member)
            
    @members("remove")
    def remove(self, member):
        """Remove a member from a component."""
        if not self.__members_contains(member):
            raise KeyError("member %s not found in %s" % (member.__path, self.__path))
        member.__parent = None
        member.__hierarchy_update()
        del self.__members[member.__name]
        
    @members("clear")
    def clear(self):
        """Clear a component of all its members."""
        for member in list(self.__members.itervalues()):
            self.__members_remove(member)
            
    # Keep private references to the original methods for internal use.
    __members_len      = __len__
    __members_iter     = __iter__
    __members_contains = __contains__ 
    __members_getitem  = __getitem__
    __members_setitem  = __setitem__
    __members_delitem  = __delitem__
    __members_add      = add
    __members_extend   = extend
    __members_remove   = remove
    __members_clear    = clear 
    # -----------------------------------------------------
    # 'tree' method group ---------------------------------
    tree = MethodGroup("tree")
    
    @tree("__len__")
    def tree_len(self):
        if len(self.__members) == 0:
            return 1
        return 1 + sum(member.__tree_len() for member in self.__members.itervalues())
        
    @tree("__getitem__")
    def tree_get(self, path):
        """Find a component in the hierarchy tree by its path. 
        TODO: write documentation for path format."""
        target = self
        path = str(path)
        if len(path) > 0:
            for symbol in path.split(Component.path_separator):
                if symbol == Component.path_root:
                    target = target.__root
                elif symbol == Component.path_parent:
                    target = target.__parent
                else:
                    target = target.__members[symbol]
        return target
        
    @tree("__setitem__")
    def tree_set(self, path, member):
        """Add a component to the model tree at the point specified by 'path'."""
        parts = str(path).split(Component.path_separator)
        parent = self.__tree_getitem(Component.path_separator.join(parts[:-1]))
        parent.__members_setitem(parts[-1], member)
        
    @tree("__delitem__")
    def tree_del(self, path):
        """Remove the component specified by 'path' from the model tree."""
        parts = str(path).split(Component.path_separator)
        parent = self.__tree_getitem(Component.path_separator.join(parts[:-1]))
        parent.__members_delitem(parts[-1])
        
    @tree("iter")
    def tree_iter(self, order=None, depth=0):
        """Make a pre-order visit to all the components in the model tree, from this point down. 
        The order by which the component's members are visited is controlled by the 'order' 
        argument, which should be a function as taken by the built-in sorted() function's 'key' 
        argument. By default members are ordered by their name.
        This method returns a generator, traversing the model using the specified ordering. The 
        generator yields (component, depth) tuples, where depth represents the level in the 
        hierarchy tree where the component is located (0 for the component where the first call
        was made)."""
        if order is None:
            order = Component.name.__get__
        yield self, depth
        for member in sorted(self.__members.itervalues(), key=order):
            for m, d in member.__tree_iter(order, depth + 1):
                yield m, d
                
    @tree("status")
    def tree_status(self, order=None, out=stdout):
        """This method prints the status of a whole model tree, using __tree_iter() and status().
        Text is indented using __tree_iter()'s depth information, providing an intuitive view
        of the model's tree structure. An output stream may be provided, such as a file or string 
        buffer (default=sys.stdout), where the status tree will be printed. Additionally, the 
        'order' argument of the __tree_iter() method can be specified."""
        nl = "\n"
        for comp, depth in self.__tree_iter(order=order):
            status = str(comp.status()).replace(nl, nl + indentation(depth + 1))
            out.write("%s%s (%s:%d) %s\n" % (indentation(depth), comp.__name, 
                                             comp.__class__.__name__, 
                                             len(comp.__members), status))
        out.flush()
        
    # Keep private references to the original methods for internal use.
    __tree_len     = tree_len
    __tree_getitem = tree_get
    __tree_setitem = tree_set
    __tree_delitem = tree_del
    __tree_iter    = tree_iter
    __tree_status  = tree_status
    # -----------------------------------------------------
    # Utilities -------------------------------------------
    __autoname_counter = {}
    
    @classmethod
    def autoname_generate(cls):
        """Generate a name for an instance of the argument class, by appending an integer to the
        class' name, e.g. Foo.autoname_generate() -> 'Foo0'."""
        n = Component.__autoname_counter.get(cls, 0)
        name = "%s%d" % (cls.__name__, n)
        Component.__autoname_counter[cls] = n + 1
        return name
        
    @classmethod
    def autoname_reset(cls):
        """Reset the counter of class instances used for automatic name generation. The next 
        instance of class 'Foo' that requests an automatic name will be named 'Foo0'."""
        if cls in Component.__autoname_counter:
            del Component.__autoname_counter[cls]
            
    @staticmethod
    def validate_name(name):
        """Component name validation procedure. This method checks the type and value of a name. 
        Names must be non-empty strings and may not contain any illegal character (found in 
        'Component.path_illegal')."""
        if not isinstance(name, str):
            raise TypeError("expected str, got %s instead" % (name.__class__.__name__,))
        if len(name) == 0:
            raise NameError("empty component name")
        illegal = Component.path_illegal.intersection(name)
        if len(illegal) > 0:
            raise NameError("illegal chars found %s" % (list(illegal),))
            
    def __hierarchy_update(self):
        """This private method is called after changes in the hierarchy. Hierarchy changes 
        include setting a component's parent or name."""
        if self.__parent is None:
            self.__root = self
            self.__path = self.__name
        else:
            self.__root = self.__parent.__root
            self.__path = self.__parent.__path + Component.path_separator + self.__name
        for member in self.__members.itervalues():
            member.__hierarchy_update()
            
    def status(self):
        """This method should be redefined in subclasses. It is used for printing a model's state, 
        so it should return any information that is relevant for understanding the component's 
        current (at the time of the call) state."""
        return ""
        
