from contextlib import contextmanager

UNDEPLOYED = "Undeployed"
DEPLOYING  = "Deploying"
ONGOING    = "Ongoing"
SUCCEEDED  = "Succeeded"
FAILED     = "Failed"
CANCELLED  = "Cancelled"

class Action(object):
    """Base class for simulation primitives. Simulation primitives implement dynamic behavior 
    which is performed by the action's owner, like waiting for a given amount of time, sending 
    signals to other processes, requesting a resource, etc. Complex behaviors can be created by 
    building trees of actions ('parent' attribute) using operators such as And, Or, Chain, or 
    Sequence. Common attributes to all action objects are:
        state - for tracking the action's deployment and completion state
        owner - process/component who is executing the action
        start_time - simulation time at which the action was started 
        elapsed_time - time taken for the action to complete (succeed, fail, or cancel)
        parent - parent action (for compound actions)
        parent_remove - function set by parent action during connection
    """
    def __init__(self):
        self.state = UNDEPLOYED
        self.owner = None
        self.start_time = None
        self.elapsed_time = None
        self.parent = None
        self.parent_remove = None
        
    def __str__(self):
        return "%s(%s)" % (self.__class__.__name__, self.info())
        
    def info(self):
        """The information displayed inside the action's string representation."""
        return ""
        
    def bind(self, owner):
        """Associate this action to a process. This is a mandatory step before executing the 
        action, since access to a simulator object is obtained through the owner."""
        self.owner = owner
        
    @contextmanager
    def in_stack(self, event_type):
        sim = self.owner.sim
        self.push_to(sim, event_type)
        yield
        self.pop_from(sim)
        
    def push_to(self, sim, event_type):
        raise NotImplementedError()
        
    def pop_from(self, sim):
        raise NotImplementedError()
        
    def initialize(self):
        """Reset the action's internal state and calls reset() on self. Subclasses that add new 
        attributes are responsible for resetting them inside the reset() method."""
        self.state = UNDEPLOYED
        self.start_time = None
        self.elapsed_time = None
        self.reset()
        
    def start(self):
        """The start() method assumes that the action's state is not ongoing (under deployment or 
        ongoing). Multiple calls to start() will raise an ValueError."""
        if self.state in (DEPLOYING, ONGOING):
            raise ValueError("cannot start() - ongoing action")
        with self.in_stack(" "):
            self.initialize()
            self.start_time = self.owner.sim.time
            self.state = DEPLOYING
            self.deploy()
            if self.state is DEPLOYING:
                self.state = ONGOING
                
    def succeed(self):
        """Complete the action with a successful state. With the action pushed to the simulation 
        stack, the parent action is warned of the child's success."""
        if self.state not in (DEPLOYING, ONGOING):
            raise ValueError("cannot succeed() - unexpected action state '%s'" % (self.state,))
        with self.in_stack("S"):
            if self.state is ONGOING:
                self.retract()
            self.state = SUCCEEDED
            self.elapsed_time = self.owner.sim.time - self.start_time
            if self.parent is not None:
                self.parent.child_succeeded(self)
                
    def fail(self):
        """Complete the action with a failure state. With the action pushed to the simulation 
        stack, the parent action is warned of the child's failure."""
        if self.state not in (DEPLOYING, ONGOING):
            raise ValueError("cannot fail() - unexpected action state '%s'" % (self.state,))
        with self.in_stack("F"):
            if self.state is ONGOING:
                self.retract()
            self.state = FAILED
            self.elapsed_time = self.owner.sim.time - self.start_time
            if self.parent is not None:
                self.parent.child_failed(self)
                
    def cancel(self):
        """Cancel an ongoing action. This retracts the action before it completes.
        NOTE: this method only works on ongoing actions. Nothing is done if the action is in any 
        other state."""
        if self.state is ONGOING:
            self.retract()
            self.state = CANCELLED
            self.elapsed_time = self.owner.sim.time - self.start_time
            
    def reset(self):
        pass
        
    def deploy(self):
        pass
        
    def retract(self):
        pass
        
    # -------------------------------------------
    def parent_connect(self, parent, remove_fnc):
        """Connect an action to an action operator (parent). This is called automatically by 
        the parent action, providing a remove function to the operand to allow it to remove 
        itself from the operator when requested, e.g. by moving the action to another operator, 
        it should first leave its previous operator. Both actions must be undeployed during the 
        connection, otherwise ValueError will be raised.
        Additionally, if the child action is unbound (owner is None) and the parent is bound, 
        the child is automatically bound to the parent's owner."""
        if self.state in (DEPLOYING, ONGOING) or parent.state in (DEPLOYING, ONGOING):
            raise ValueError("cannot connect to parent - both actions must be undeployed")
        if self.parent is not None:
            self.parent_disconnect()
        self.parent = parent
        self.parent_remove = remove_fnc
        if self.owner is None and parent.owner is not None:
            self.bind(parent.owner)
            
    def parent_disconnect(self):
        """Disconnect an action from its current operator. The action's 'parent_remove' function 
        is called and the parent and remove function are set to None. As for connection, both 
        actions must be undeployed for disconnection."""
        if self.state in (DEPLOYING, ONGOING) or self.parent.state in (DEPLOYING, ONGOING):
            raise ValueError("cannot disconnect from parent - both actions must be undeployed")
        self.parent_remove(self)
        self.parent_remove = None
        self.parent = None
        
    def child_succeeded(self, child):
        """Method called by child actions to warn an operator of their success and allow it to 
        take action accordingly. This method should only be defined for action operators."""
        raise ValueError("child method attempted, but action is not an operator")
        
    def child_failed(self, child):
        """Method called by child actions to warn an operator of their failure and allow it to 
        take action accordingly. This method should only be defined for action operators."""
        raise ValueError("child method attempted, but action is not an operator")
        
    # --------------------------------------------
    def undeployed(self): return self.state is UNDEPLOYED
    def deploying(self):  return self.state is DEPLOYING
    def ongoing(self):    return self.state is ONGOING
    def succeeded(self):  return self.state is SUCCEEDED
    def failed(self):     return self.state is FAILED
    def cancelled(self):  return self.state is CANCELLED
    