from types import GeneratorType

from khronos.des.primitives.operators.unary_op import UnaryOp
from khronos.des.primitives.action import Action, UNDEPLOYED, DEPLOYING, ONGOING
from khronos.des.primitives.dummy import DUMMY
from khronos.des.primitives.delay import Delay
from khronos.utils import Call

class ChainGetter(Action):
    """Action used to get the currently running chain object inside the generator function (this 
    is not a very good idea though). The ChainGetter class is made indirectly available through 
    the 'get' attribute of the Chain class. Example:
        @Chain
        def test(self):
            ...
            this_chain = yield Chain.get()
            ... # do something with the chain object"""
    def start(self):
        self.parent.step(self.parent)
        
class ChainResult(Action):
    """Action used to set an arbitrary object to a chain's result attribute. The ChainResult 
    class is made indirectly available through the 'result' attribute of the Chain class. 
    Example:
        x = [123, "abc", (self,)]
        yield Chain.result(x)"""
    def __init__(self, value):
        Action.__init__(self)
        self.value = value
        
    def start(self):
        self.parent.result = self.value
        self.parent.step()
        
class ChainSuccess(StopIteration):
    """This exception class is used to force a success of the chain. If it is raised inside a 
    chain's generator function, the chain object will succeed(). The ChainSuccess class is 
    made available through the 'success' attribute of the Chain class. Example:
        @Chain
        def foo(self):
            yield 10 * MINUTE
            if condition_to_succeed():
                raise Chain.success()
            else:
                ...
    Note that this is nothing more than a StopIteration exception. It exists for consistency 
    and increased readability, but you could obtain the same result by explicitly raising 
    StopIteration, simply using an empty return, or in most cases just let the generator 
    function run to the end, where a StopIteration exception is implicitly raised. However, 
    it is arguably better to raise a Chain.success() instead of putting an empty return or 
    raising StopIteration, because that way whoever reads the program will think/know that the 
    chain will succeed, and return/StopIteration do not give this idea to readers."""
    
class ChainFailure(Exception):
    """This exception class is used to force a failure of the chain. If it is raised inside a 
    chain's generator function, the chain object will fail(). The ChainFailure class is made 
    available through the 'failure' attribute of the Chain class. Example:
        @Chain
        def foo(self):
            yield 10 * MINUTE
            if not condition_to_continue():
                raise Chain.failure()
            else:
                ..."""
    
class Chain(UnaryOp):
    get     = ChainGetter
    result  = ChainResult
    success = ChainSuccess
    failure = ChainFailure
    
    @classmethod
    def create_from(cls, constructor):
        """This is the actual constructor of the Chain class, because the __new__() special 
        function works as a convenient function decorator. In fact, action chains should primarily 
        be created by calling Chain-decorated functions, and manual creation of chain objects 
        should be avoided. To manually create a Chain object from a callable, simply write
            c = Chain.create_from(my_callable)
        instead of 
            c = Chain(my_callable)
        since the latter version will actually return a new function (the decorated version) which 
        works as a chain factory."""
        self = UnaryOp.__new__(cls)
        UnaryOp.__init__(self, DUMMY)
        self.constructor = constructor
        self.generator = None
        self.result = None
        return self
        
    def info(self):
        info = self.constructor.__name__
        if self.result is not None:
            info += " -> %s" % (self.result,)
        return info
        
    def reset(self):
        self.generator = None
        self.result = None
        
    def deploy(self):
        try:
            generator = self.constructor()
        except StopIteration:
            UnaryOp.succeed(self)
        else:
            if not isinstance(generator, GeneratorType):
                raise TypeError("invalid chain (constructor did not return a generator)")
            self.generator = generator
            self.step()
            
    def step(self, send_value=None):
        try:
            self.state = DEPLOYING
            result = self.generator.send(send_value)
        except StopIteration:
            UnaryOp.succeed(self)
        except ChainFailure:
            UnaryOp.fail(self)
        else:
            if not isinstance(result, Action):
                result = Delay(result)
            if result is not self.operand:
                self.state = UNDEPLOYED
                self.set_operand(result)
            self.state = ONGOING
            result.start()
            
    def succeed(self):
        self.operand.succeed()
        
    def fail(self):
        self.operand.fail()
        
    def child_activated(self, child): 
        if child is not self.operand:
            raise ValueError("invalid child action provided")
        self.step(child)
        
    child_succeeded = child_activated
    child_failed    = child_activated
    # -----------------------------------------------------
    # Function/method decorators --------------------------
    # NOTE: These should be used with generator functions only!!!
    def __new__(cls, fnc):
        """The default class constructor works as a function decorator for functions or methods 
        defining action chains. When the decorated function is called, an unbound Chain object is 
        created and returned. This makes the creation of chains very convenient.
            @Chain
            def my_chain(self):
                do_stuff_here"""
        def new_fnc(*args, **kwargs):
            chain = cls.create_from(Call(fnc, *args, **kwargs))
            return chain
        new_fnc.__name__ = fnc.__name__
        new_fnc.__doc__ = fnc.__doc__
        return new_fnc
        
    @classmethod
    def bound(cls, fnc):
        """Like the __new__() decorator, this one also allows automatic creation of action chains 
        when the decorated method (or function) is called, with the difference that the created 
        chain is bound to the first positional argument. This usage is appropriate on component 
        *methods*, where the component is always the first positional argument (self)."""
        def new_fnc(*args, **kwargs):
            chain = cls.create_from(Call(fnc, *args, **kwargs))
            chain.bind(args[0])  # args[0] is 'self' in object methods
            return chain
        new_fnc.__name__ = fnc.__name__
        new_fnc.__doc__ = fnc.__doc__
        return new_fnc
        